/*
 * Copyright (C) 2007 Andre Beckedorf <evilJazz _AT_ katastrophos _DOT_ net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>

#ifdef QTOPIA
#include <qpe/qpemenubar.h>
#include <qpe/qpetoolbar.h>
#include <qpe/qcopenvelope_qws.h>
#include <qpe/resource.h>
#include <qpe/config.h>
#include <qpe/global.h>
#else
#include <qmenubar.h>
#include <qtoolbar.h>
#include <resource.h>
#include <config.h>
// for Qtopia we are using the QPE variants for toolbar and menubar
// to enable special features like the menu key etc.
// Define aliases to avoid ifdefs...
#define QPEMenuBar QMenuBar
#define QPEToolBar QToolBar
#endif

#include <qstringlist.h>
#include <qstring.h>
#include <qpushbutton.h>
#include <qlayout.h>
#include <qwidget.h>
#include <qlabel.h>
#include <qlistbox.h>
#include <qlistview.h>
#include <qspinbox.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qvgroupbox.h>
#include <qvbox.h>
#include <qhbox.h>
#include <qprogressbar.h>
#include <qtabbar.h>
#include <qdatastream.h>
#include <qwidgetstack.h>
#include <qmessagebox.h>

#include <qurlinfo.h>
#include <qfile.h>
#include <qregexp.h>

#include "qscrobbler.h"
#include "debug.h"
#include "audioscrobbler.h"
#include "kascachelist.h"
#include "configuration.h"
#include "qscrobblerconfig.h"
#include "qscrobbleraddon.h"

#include "kmdcodec.h"

QScrobbler::QScrobbler(AudioScrobbler *asc, QWidget *parent, const char *name, WFlags f)
	:	QDialog(parent, name, false, f)
{
#ifdef QTOPIA
	setWState(WState_Reserved1);
#endif
	setWFlags(getWFlags() | WStyle_SysMenu | WStyle_Maximize | WDestructiveClose);
	setSizeGripEnabled(true);

	setCaption("QScrobbler");

	scrobbler = asc;

    QPopupMenu *help = new QPopupMenu(this);
    help->insertItem(tr("&About..."), this, SLOT(about()));

    QPEMenuBar *menu = new QPEMenuBar(this);
	menu->setMargin(0);
    menu->insertItem("&Help", help);

	QVBoxLayout *vl = new QVBoxLayout(this, 8);
	vl->setAutoAdd(true);

    gbox1 = new QVGroupBox(tr("Song waiting to be scrobbled"), this);
    currentlyPlayingLabel = new QLabel(tr("No song."), gbox1);

    progressStack = new QWidgetStack(gbox1);
    progressStack->setSizePolicy(QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding));
    progressBar = new QProgressBar(progressStack);
    progressStatus = new QLabel(progressStack);

    gbox0 = new QVGroupBox(tr("Scrobbled songs waiting to be submitted"), this);
    scrobbledTracksListView = new QListView(gbox0);
    scrobbledTracksListView->setItemMargin(4);
    scrobbledTracksListView->setAllColumnsShowFocus(true);
    scrobbledTracksListView->setSorting(-1);
    scrobbledTracksListView->setShowSortIndicator(false);

    scrobbledTracksListView->addColumn("Played at (UTC)");
    scrobbledTracksListView->addColumn("Title");
    scrobbledTracksListView->addColumn("Artist");
    scrobbledTracksListView->addColumn("Album");

#ifndef WINDOWS
    QFont font = scrobbledTracksListView->font();
    font.setPointSize(font.pointSize() * 3 / 4);
    scrobbledTracksListView->setFont(font);
#endif

    QHBox *hbox = new QHBox(gbox0);
    QPushButton *pauseSubmissionsButton = new QPushButton("Pause submissions", hbox);
    pauseSubmissionsButton->setToggleButton(true);
    pauseSubmissionsButton->setOn(qScrobblerAddOn.config().submissionPaused());
    connect(pauseSubmissionsButton, SIGNAL(toggled(bool)), scrobbler, SLOT(setSubmissionPaused(bool)));

    QPushButton *clearSubmissionsButton = new QPushButton("Clear list", hbox);
    connect(clearSubmissionsButton, SIGNAL(clicked()), this, SLOT(clearScrobbledTracksList()));

    QPushButton *retrySubmissionButton = new QPushButton("Submit now!", hbox);
    connect(retrySubmissionButton, SIGNAL(clicked()), this, SLOT(retrySubmit()));

    QVGroupBox *gbox2 = new QVGroupBox(tr("Status:"), this);
    hbox = new QHBox(gbox2);
    statusLabel = new QLabel("ready.", hbox, "statuslabel2");

    QPixmap badgePixmap = Resource::loadPixmap("lastfm");
    if (!badgePixmap.isNull())
    {
		QWidget *badge = new QWidget(hbox);
		badge->setBackgroundPixmap(badgePixmap);
		if (badge->backgroundPixmap())
			badge->setFixedSize(badge->backgroundPixmap()->width(), badge->backgroundPixmap()->height());
    }

    connect(scrobbler, SIGNAL(statusMessage(const QString&)), statusLabel, SLOT(setText(const QString&)));
    connect(scrobbler, SIGNAL(cacheSaved()), this, SLOT(cacheSaved()));
    connect(scrobbler, SIGNAL(songAddedForSubmission(const QString&)), this, SLOT(songAddedForSubmission(const QString&)));
    connect(scrobbler, SIGNAL(songRemoved()), this, SLOT(removeCurrentSong()));

    connect(scrobbler, SIGNAL(songAdded(const QString &, const QString &, const QString &, const QString &, unsigned int)),
			this, SLOT(songAdded(const QString &, const QString &, const QString &, const QString &, unsigned int)));

    updateTimer = new QTimer();
    connect(updateTimer, SIGNAL(timeout()), this, SLOT(updateTimerTriggered()));

    if (scrobbler->isActive())
    {
    	songAdded(scrobbler->currentSong().artist, scrobbler->currentSong().songtitle, scrobbler->currentSong().album, scrobbler->currentSong().mbid, scrobbler->currentSong().length);
    }
    else
    {
    	removeCurrentSong();
    }

    cacheSaved();
}

QScrobbler::~QScrobbler()
{
	disconnect(scrobbler, SIGNAL(statusMessage(const QString&)), statusLabel, SLOT(setText(const QString&)));
    disconnect(scrobbler, SIGNAL(cacheSaved()), this, SLOT(cacheSaved()));
    disconnect(scrobbler, SIGNAL(songAddedForSubmission(const QString&)), this, SLOT(songAddedForSubmission(const QString&)));
    disconnect(scrobbler, SIGNAL(songRemoved()), this, SLOT(removeCurrentSong()));
    disconnect(scrobbler, SIGNAL(songAdded(const QString &, const QString &, const QString &, const QString &, unsigned int)),
		this, SLOT(songAdded(const QString &, const QString &, const QString &, const QString &, unsigned int)));
}

void QScrobbler::closeEvent(QCloseEvent *e)
{
	emit closing();
	e->accept();
}

void QScrobbler::done(int r)
{
	emit closing();
	QDialog::done(r);
}

void QScrobbler::showEvent(QShowEvent *)
{
#ifdef QTOPIA
	showMaximized();
#endif;
}

void QScrobbler::cacheSaved()
{
	DENTERMETHOD("QScrobbler::cacheSaved()");

	int subCount = scrobbler->cache().submissionCount();

    scrobbledTracksListView->setUpdatesEnabled(false);
    scrobbledTracksListView->blockSignals(true);
    scrobbledTracksListView->clear();

    for (int i = 0; i < subCount; i++)
    {
    	const ScrobSongData songData = scrobbler->cache().getSubmission(i);
    	QListViewItem *li = new QListViewItem(scrobbledTracksListView);
    	li->setText(0, songData.time);
    	li->setText(1, songData.songtitle);
    	li->setText(2, songData.artist);
    	li->setText(3, songData.album);
    }

    if (scrobbledTracksListView->firstChild())
    {
    	scrobbledTracksListView->setCurrentItem(scrobbledTracksListView->firstChild());
    	scrobbledTracksListView->setSelected(scrobbledTracksListView->firstChild(), true);
    }

    scrobbledTracksListView->blockSignals(false);
    scrobbledTracksListView->setUpdatesEnabled(true);
    scrobbledTracksListView->triggerUpdate();

    gbox0->setTitle(tr("Scrobbled songs waiting to be submitted") + ": " + QString::number(subCount));

    DEXITMETHOD("QScrobbler::cacheSaved()");
}

void QScrobbler::clearScrobbledTracksList()
{
	int ret = QMessageBox::warning(this, tr("Confirmation"),
								tr("Do you really want to clear all scrobbled tracks?"),
								tr("Yes"), tr("No"), QString::null,
								1, 1);

	if (ret == 0)
		scrobbler->clearCache();
}

void QScrobbler::retrySubmit()
{
	scrobbler->reinitialize();
}

void QScrobbler::songAddedForSubmission(const QString& songTitle)
{
	DENTERMETHOD("QScrobbler::songAddedForSubmission(...)");

	gbox1->setTitle(tr("Song currently playing"));
	progressStatus->setText(tr("Song successfully scrobbled."));

	progressStack->raiseWidget(progressStatus);
	progressBar->setProgress(100);
	updateTimer->stop();

	DEXITMETHOD("QScrobbler::songAddedForSubmission(...)");
}

void QScrobbler::about()
{
	QMessageBox mb("About QScrobbler",
			"QScrobbler " + QString("%1.%2").arg(APPVERSION_MAJOR).arg(APPVERSION_MINOR) + "\n"
			"Copyright (C) 2007-2008 by Andre Beckedorf\n"
			"evilJazz@katastrophos.net\n"
			"http://katastrophos.net/qscrobbler\n\n"
			"QScrobbler is based on portions of KAudioScrobbler\n"
			"Copyright (C) 2004 by Steven Scott\n"
			"http://code.google.com/p/kaudioscrobbler/\n\n"
			"This software is licensed under the terms of\n"
			"the GNU Public License version 2.", QMessageBox::NoIcon, 1, 0, 0);
	mb.exec();
}

void QScrobbler::updateTimerTriggered()
{
	DENTERMETHOD("QScrobbler::updateTimerTriggered()");

	qDebug("scrobbler->requiredTime(): %d, scrobbler->currentRemainingTime(): %d", scrobbler->requiredTime(), scrobbler->currentRemainingTime());

	if (scrobbler->currentRemainingTime() <= 0)
	{
		progressBar->setProgress(scrobbler->requiredTime());
		updateTimer->stop();
	}
	else
		progressBar->setProgress(scrobbler->requiredTime() - scrobbler->currentRemainingTime());

	statusLabel->setText(scrobbler->lastStatus());

	DEXITMETHOD("QScrobbler::updateTimerTriggered()");
}

void QScrobbler::songAdded(const QString &artist, const QString &title, const QString &album, const QString &musicbrainzid, unsigned int length)
{
	DENTERMETHOD("QScrobbler::setCurrentSong(...)");

	gbox1->setTitle(tr("Song waiting to be scrobbled"));
	currentlyPlayingLabel->setText(artist + " - " + title);
	progressBar->setTotalSteps(scrobbler->requiredTime());
	progressStack->raiseWidget(progressBar);
	progressBar->setProgress(0);
	updateTimer->start(3000);
	updateTimerTriggered();

	DEXITMETHOD("QScrobbler::setCurrentSong(...)");
}

void QScrobbler::removeCurrentSong()
{
	DPRINTF("stopped...");
	gbox1->setTitle(tr("Song waiting to be scrobbled"));
	currentlyPlayingLabel->setText(tr("No song."));
	progressStatus->setText("");
	progressStack->raiseWidget(progressStatus);

	updateTimer->stop();
}
