/*
 * Copyright (C) 2007 Andre Beckedorf <evilJazz _AT_ katastrophos _DOT_ net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <qapplication.h>
#include <qimage.h>
#include <qdir.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qregexp.h>

#include "coverartprovider.h"
#include "mediaidentifier.h"
#include "qprogressdialog.h"
#include "debug.h"

CoverArtProvider & CoverArtProvider::singleton()
{
	static CoverArtProvider instance;
	return instance;
}


CoverArtProvider::CoverArtProvider()
	:	QObject(),
		sourceDirectory_("")
{
	validExtensionsList_ = QStringList() << "jpeg" << "jpg" << "png" << "bmp" << "tif" << "tiff";
}

CoverArtProvider::~CoverArtProvider()
{
}

void CoverArtProvider::setCoverArtDirectory(const QString &directory)
{
	if (directory != sourceDirectory_)
	{
		sourceDirectory_ = directory;
		rescan();
	}
}

void CoverArtProvider::rescan()
{
	coverArtImageFilenames_.clear();
	simplifiedCoverArtImageFilenames_.clear();

	if (!sourceDirectory_.isEmpty() && QDir(sourceDirectory_).exists())
	{
		QProgressDialog progress(tr("Please wait..."), tr("Cancel"), 0, NULL, "progress", true);

		progress.setCaption(tr("Scanning for cover art images..."));

		progress.setMinimumDuration(0);
		progress.setFixedWidth(QMIN(qApp->desktop()->width() * 0.9, 640));
		progress.setAutoClose(false);
		progress.setAutoReset(false);
		progress.show();
		qApp->processEvents();

		QString root = QFileInfo(sourceDirectory_).absFilePath();

		fillFileList(coverArtImageFilenames_, root, true);

		int rootLength = root.length();

		for (QStringList::Iterator it = coverArtImageFilenames_.begin(); it != coverArtImageFilenames_.end(); ++it )
			simplifiedCoverArtImageFilenames_.append(simplifyString((*it).mid(rootLength).lower()));

		progress.hide();
		qApp->processEvents();
	}
}

void CoverArtProvider::fillFileList(QStringList &filelist, const QString &path, const bool subdir)
{
	QDir dir(path);
	if (!dir.exists()) return;

	dir.setFilter(QDir::Files | (subdir ? QDir::Dirs : 0));
	dir.setSorting(QDir::Unsorted);

	const QFileInfoList *list = dir.entryInfoList();
	QString fileName;
	QFileInfoListIterator it(*list);
	QFileInfo *info;

	for (; (info = it.current()); ++it)
	{
		if (info->isFile())
		{
			// only add the file, if its extension is known...
			if (validExtension(info->extension(false)))
				filelist.append(info->absFilePath());
		}
		else
		{
			fileName = info->fileName();
			if (fileName != "." && fileName != "..")
				fillFileList(filelist, info->absFilePath(), subdir);
		}
	}
}

QString CoverArtProvider::simplifyString(const QString &input)
{
	QString output = input;
	output.replace(QRegExp("\\/"), " "); // AC/DC -> AC DC
	output.replace(QRegExp("\\-"), " "); // AC-DC -> AC DC
	output.replace(QRegExp("\\_"), " "); // AC_DC -> AC DC
	output.replace(QRegExp("\\(.*\\)"), ""); // remove (Reissue)
	output.replace(QRegExp("\\[.*\\]"), ""); // remove [Reissue]
	output.replace(QRegExp("CD.*[0-9]+", false), " ");
	output.replace(QRegExp("[,.+&/=:#!?]"), " ");
	output = output.simplifyWhiteSpace();
	DPRINTF("simplifyString input: '%s', output: '%s'", (const char*)input.utf8(), (const char*)output.utf8());
	return output;
}

bool CoverArtProvider::loadCoverDirectly(QImage &dstImage, const QString &artist, const QString &album, const QString &filename)
{
	bool result = false;

	if (!filename.isEmpty())
	{
		result = gMediaIdentifier.loadCoverArtFromFile(filename, dstImage);

		// Check for cover files in the same directory the media file is in...
		if (!result)
		{
			QStringList imageFilenames;
			fillFileList(imageFilenames, QFileInfo(filename).absFilePath(), false);

			if (imageFilenames.count() > 0)
			{
				int i = 0;
				while (i < imageFilenames.count() && !result)
				{
					 result = dstImage.load(imageFilenames[i]);
					 ++i;
				}
			}
		}
	}

	if (!result && !artist.isEmpty() && !album.isEmpty())
	{
		// Check for cover files in a separate tree...
		QString artistMatch = simplifyString(artist.lower());
		QString albumMatch = simplifyString(album.lower());

		DPRINTF("artist: %s", (const char*)artist.utf8());
		DPRINTF("album: %s", (const char*)album.utf8());

		DPRINTF("artistMatch: %s", (const char*)artistMatch.utf8());
		DPRINTF("albumMatch: %s", (const char*)albumMatch.utf8());

		QStringList matchedFilenames;
		int index = 0;

		for (QStringList::Iterator it = simplifiedCoverArtImageFilenames_.begin(); it != simplifiedCoverArtImageFilenames_.end(); ++it			, ++index)
		{
			if ((*it).findRev(artistMatch, -1) > -1 && (*it).findRev(albumMatch, -1) > -1)
			{
				DPRINTF("found match: %s", (const char *)coverArtImageFilenames_[index].utf8());
				matchedFilenames.append(coverArtImageFilenames_[index]);
			}
		}

		int i = 0;
		while (i < matchedFilenames.count() && !result)
		{
			 result = dstImage.load(matchedFilenames[i]);
			 ++i;
		}
	}

	return result;
}

QStringList CoverArtProvider::findCoverArtImageFilenames(const QString &artist, const QString &album, const QString &filename)
{
	DENTERMETHOD("CoverArtProvider::findCoverArtImageFilenames(%s, %s, %s)", (const char *)artist.utf8(), (const char *)album.utf8(), (const char *)filename.utf8());

	QStringList result;

	// Check for cover files in the same directory the media file is in...
	if (!filename.isEmpty())
		fillFileList(result, QFileInfo(filename).absFilePath(), false);

	if (!artist.isEmpty() && !album.isEmpty())
	{
		// Check for cover files in a separate tree...
		QString artistMatch = simplifyString(artist.lower());
		QString albumMatch = simplifyString(album.lower());
		int index = 0;

		for (QStringList::Iterator it = simplifiedCoverArtImageFilenames_.begin(); it != simplifiedCoverArtImageFilenames_.end(); ++it			, ++index)
		{
			if ((*it).findRev(artistMatch, -1) > -1 && (*it).findRev(albumMatch, -1) > -1)
				result.append(coverArtImageFilenames_[index]);
		}
	}

	DEXITMETHOD("CoverArtProvider::findCoverArtImageFilenames(%s, %s, %s)", (const char *)artist.utf8(), (const char *)album.utf8(), (const char *)filename.utf8());

	return result;
}
