/*
 * Copyright (C) 2006-2007	Andre Beckedorf
 * 					 		<evilJazz _AT_ katastrophos _DOT_ net>
 *
 * This file is based on eMotion by Michael Swieton <mike@swieton.net>.
 * The few remaining parts in this almost completely rewritten file are
 * Copyright (C) 2005 Atmark <atmarkat _AT_ msn _DOT_ com>
 *                    AGAWA Koji <i _AT_ atty _DOT_ jp>
 *                    Michael Swieton <mike@swieton.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <qapplication.h>
#include <qfileinfo.h>
#include <qstring.h>
#include <qfile.h>

#include <cstdio>
#include <cstdlib>

#include <signal.h>
#include <unistd.h>
#include <errno.h>

#ifdef WINDOWS
#include "wincompat.h"
#endif

#include "mplayer.h"
#include "media.h"
#include "configuration.h"

#define DEBUG

#include "debug.h"

static QProcess *checkFeaturesProcess;

// available before version 1.0pre8:
static bool mplayerHasGetPercentPos;
static bool mplayerHasGetTimeLength;

// added in version 1.0pre8:
static bool mplayerHasGetTimePos;
static bool mplayerCanQueueFiles;
static bool mplayerSupportsIdle;
static bool mplayerSupportsPausingKeepToogle;

MPlayer::MPlayer()
	:	quitState_(QuitNull),
		process_(NULL),
		lastMplayerBinLocation_(""),
		buffer_(""),
		messages_(""),
		params_(""),
		isPaused_(false),
		playbackWasStarted_(false),
		errorDetected_(false),
		isSendingPTS_(false),
		parsing_(false),
		watchdogCheck_(0),
		newPlaytime_(-1)
{
	register_handler(SIGTERM, this);
	register_handler(SIGSEGV, this);

	connect(this, SIGNAL(quit()), this, SLOT(cleanUp()));
	connect(this, SIGNAL(killed()), this, SLOT(cleanUp()));

	connect(&timer_, SIGNAL(timeout()), this, SLOT(pollMPlayer()));

	connect(&SystemVolume::singleton(), SIGNAL(levelChanged(int)), this, SLOT(systemVolumeChanged(int)));

	connect(&killTimer_, SIGNAL(timeout()), this, SLOT(killWaitingProcesses()));
	processKillWaitList_.setAutoDelete(true);
}

MPlayer::~MPlayer()
{
	quitMplayer(false);
	unregister_handler(this);
}

void MPlayer::checkFeaturesRunProcess()
{
	// timeout timer...
	QTime time;
	time.start();
	bool timedOut = false;

	checkFeaturesProcess->start();

	while ((checkFeaturesProcess->isRunning() || checkFeaturesProcess->canReadLineStdout()) && !timedOut)
	{
		timedOut = time.elapsed() > 5000;
		checkFeaturesProcessParseLines();
		usleep(1000); // sleep 1 ms
	}

	if (checkFeaturesProcess->isRunning())
	{
		checkFeaturesProcess->tryTerminate();
		usleep(50000); // wait 50 ms
	}

	if (checkFeaturesProcess->isRunning())
	{
		checkFeaturesProcess->kill();
		usleep(100000); // wait 100 ms
	}
}

bool MPlayer::checkFeatures()
{
	QString mplayerBinLocation = qConfig.mplayerBinLocation();

	if (lastMplayerBinLocation_ != mplayerBinLocation)
	{
		mplayerHasGetTimePos = false;
		mplayerHasGetTimeLength = false;
		mplayerHasGetPercentPos = false;
		mplayerCanQueueFiles = false;
		mplayerSupportsIdle = false;
		mplayerSupportsPausingKeepToogle = false;

		QStringList args;
		checkFeaturesProcess = new QProcess();

		args << qConfig.mplayerBinLocation() << "-input" << "cmdlist";
		checkFeaturesProcess->setArguments(args);
		checkFeaturesRunProcess();

		args.clear();
		args << qConfig.mplayerBinLocation() << "-list-options";
		checkFeaturesProcess->setArguments(args);
		checkFeaturesRunProcess();

		delete checkFeaturesProcess;
		checkFeaturesProcess = NULL;

		if (mplayerSupportsIdle)
		{
			mplayerSupportsPausingKeepToogle = true;
		}

		lastMplayerBinLocation_ = mplayerBinLocation;

		return true;
	}

	return false;
}

void MPlayer::checkFeaturesProcessParseLines()
{
	while (checkFeaturesProcess->canReadLineStdout())
	{
		QString line = checkFeaturesProcess->readLineStdout();

		DPRINTF("LINE stdout: %s", (const char*)(line.utf8()));

		if (line.startsWith("get_time_pos"))
		{
			mplayerHasGetTimePos = true;
			qDebug("mplayer supports get_time_pos.");
		}
		else if (line.startsWith("get_time_length"))
		{
			mplayerHasGetTimeLength = true;
			qDebug("mplayer supports get_time_length.");
		}
		else if (line.startsWith("get_percent_pos"))
		{
			mplayerHasGetPercentPos = true;
			qDebug("mplayer supports get_percent_pos.");
		}
		else if (line.startsWith("loadfile") && line.find("[Integer]") != -1)
		{
			mplayerCanQueueFiles = true;
			qDebug("mplayer supports advanced syntax for loadfile, i.e. it can queue files.");
		}
		else if (line.simplifyWhiteSpace().startsWith("idle Flag"))
		{
			mplayerSupportsIdle = true;
			qDebug("mplayer supports idling in slave mode.");
		}
	}
}

QStringList MPlayer::getArgListForMedia(Media *media)
{
	QStringList args;

	args << qConfig.mplayerBinLocation();
	args << "-slave";

	if (mplayerSupportsIdle)
	{
		args << "-idle";
		args << "-v";
	}

	args << "-quiet";
#ifdef QTOPIA
	args << "-nortc";
#endif

	QString videoFilter = "";
	if (media->isVideo())
	{
		// Video Output
		if (qConfig.isOverlayEnabled)
		{
			switch (qConfig.machineCategory())
			{
			case Configuration::MACHINE_SLC700:
				args << "-vo";
				args << "w100";
				break;
			case Configuration::MACHINE_SLC3000:
				args << "-vo";
				args << "bvdd";
				break;
			case Configuration::MACHINE_GENERIC:
				break;
			}
		}

		// Full Screen
		if (qConfig.isFullScreenEnabled) {
			switch (qConfig.machineCategory()) {
			case Configuration::MACHINE_SLC700:
				args << "-fs";
				break;
			case Configuration::MACHINE_SLC3000:
				args << "-vm";
				break;
			case Configuration::MACHINE_GENERIC:
				args << "-fs";
				break;
			}
		}

		// Double Buffering
		if (!qConfig.isDoubleBufferingEnabled)
			args << "-nodouble";
		else
			args << "-double";

		// Frame Drop
		switch (qConfig.framedropMethod)
		{
		case Configuration::FRAMEDROP_NORMAL:
			args << "-framedrop";
			break;
		case Configuration::FRAMEDROP_HARD:
			args << "-hardframedrop";
			break;
		default:
			break;
		}

		if (useSoftVolume())
		{
			args << "-softvol";
			//args << "-volume";
			//args << QString::number(qSystemVolume.getCurrentLevel());
		}

#ifdef QTOPIA
		// 映像と音声の同期を鈍感にする。
		// 一時的に負荷が高まる映像の再生が安定する。
		args << "-autosync";
		args << "100";

		// FIXME: more generic handling
		if (media->width() == 352 && media->height() == 240)
		{
			args << "-vf-add";
			args << "crop=320:240";
		}

		if (media->width() > media->height())
		{
			if (media->width() > qApp->desktop()->width() ||
				media->height() > qApp->desktop()->height())
			{
				args << "-vf-add";
				args << "crop=640:480";	// FIXME: use qApp->desktop()
			}
		}

		if (!qConfig.isOverlayEnabled)
		{
			if (media->width() > media->height())
			{
				args << "-vf-add";
				args << "rotate=1";
			}
		}
#endif
	}

#ifdef QTOPIA
	// これらのサンプリングレート以外をそのまま再生しようとすると非常に
	// 負荷が高くなるので、44100KHzにリサンプリングする
	if (media->audioRate() != 44100 && media->audioRate() != 48000 && media->audioRate() != 22050)
	{
		args << "-af";
		args << "lavcresample=44100";
	}
#endif

	// Cache handling...
	QString cacheSize;
	if (qConfig.isCacheEnabled)
	{
		args << "-cache";
		cacheSize = QString::number(qConfig.cacheSize);
		args << cacheSize;

		if (media->location().isLocalFile())
		{
			// for local files, start playback immediately, ie.
			// don't wait till the cache is filled up.
			// We assume local files are read faster than they are played back.
			args << "-cache-min";
			args << "0";
		}
	}
	else
		args << "-nocache";

	// 初期状態ではOSDを表示しない
	args << "-osdlevel";
	args << "0";

	QString fileExtension = media->location().fileName();
	fileExtension = fileExtension.right(fileExtension.length() -1 - fileExtension.findRev('.'));
	DPRINTF("fileExtension: %s", (const char*)fileExtension.latin1());
	QString customArguments = qConfig.mplayerArguments(fileExtension);
	if (!customArguments.isEmpty())
	{
		DPRINTF("customArguments: %s", (const char*)customArguments.latin1());
		QStringList opts = QStringList::split(' ', customArguments);
		for (QStringList::Iterator it = opts.begin(); it != opts.end(); ++it)
			args << (*it);
	}

	// その他の追加オプション
	if (qConfig.isAdditionalOptionsEnabled)
	{
		QStringList opts = QStringList::split(' ', qConfig.additionalOptions);
		for (QStringList::Iterator it = opts.begin(); it != opts.end(); ++it)
			args << (*it);
	}

	return args;
}

bool MPlayer::canRecycleMplayer(Media *media)
{
	QStringList args;
	return canRecycleMplayer(media, args);
}

bool MPlayer::canRecycleMplayer(Media *media, QStringList &args)
{
	args = getArgListForMedia(media);

	bool mplayerBinLocationUpdated = checkFeatures();

	bool argsDiffer = args.count() != currentProcessArgs_.count();
	if (!argsDiffer)
		for (int i = 0; i < args.count(); ++i)
			if (args[i] != currentProcessArgs_[i])
			{
				argsDiffer = true;
				DPRINTF("%s != %s", (const char *)(currentProcessArgs_[i].latin1()), (const char *)(args[i].latin1()));
				break;
			}

	//return false; // for manual override during debugging...
	return isRunning() && !argsDiffer && !media->isVideo() && !mplayerBinLocationUpdated;
}

void MPlayer::killWaitingProcesses()
{
	DENTERMETHOD("MPlayer::killWaitingProcesses()");
	DPRINTF("It's killing season: %d", processKillWaitList_.count());
	processKillWaitList_.clear();
	DPRINTF("All dead: %d", processKillWaitList_.count());
	DEXITMETHOD("MPlayer::killWaitingProcesses()");
}

void MPlayer::scheduleDestructionOfProcess(QProcess *process)
{
	disconnect(process, SIGNAL(processExited()));
	disconnect(process, SIGNAL(readyReadStdout()));

	if (!processKillWaitList_.containsRef(process))
	{
		processKillWaitList_.append(process);
		killTimer_.start(100, true);
	}
}

MPlayer::PlayResult MPlayer::play(Media *media)
{
	DENTERMETHOD("MPlayer::play(%s)", (const char *)media->location().toString().utf8());

	ASSERT(media != 0);

	/*
	if (media->location().isLocalFile() && !QFile::exists(media->location().toString()))
	{
		DPRINTF("%s was not found", QString(media->location()).latin1());
		DEXITMETHOD("MPlayer::play(%s)", (const char *)media->location().toString().utf8());
		return MediaNotFound;
	}
	*/

	if (!media->isSupported())
	{
		DPRINTF("%s is Not supported", QString(media->location()).latin1());
		DEXITMETHOD("MPlayer::play(%s)", (const char *)media->location().toString().utf8());
		return MediaNotSupported;
	}

	DPRINTF("Determining mplayer binary features...");
	checkFeatures();

	QStringList args;

	if (!canRecycleMplayer(media, args))
	{
		if (isRunning())
			quitMplayer(false);

		// currentProcessArgs may only contain command line
		// arguments without the actual filename of the media
		// to play. This is mandatory because we use it to compare
		// against it next time we try to recycle the MPlayer process.
		currentProcessArgs_ = args;

		args << media->location().toString();

		params_ = args.join(" ") + "\n";
		messages_ = QString::null;
		playbackWasStarted_ = false;
		errorDetected_ = false;
		quitState_ = QuitNull;

		if (qConfig.isBackendLoggingEnabled)
		{
			logStream_.setDevice(new QFile(qConfig.backendLogFileName));
			logStream_.device()->open(IO_WriteOnly|IO_Truncate);
			logStream_ << params_;
			logStream_.device()->flush();
		}

		process_ = new QProcess(args);
		process_->setCommunication(QProcess::Stdin|QProcess::Stdout|QProcess::Stderr|QProcess::DupStderr);
		connect(process_, SIGNAL(readyReadStdout()), this, SLOT(processReadyReadStdout()));
		connect(process_, SIGNAL(processExited()), this, SLOT(processExited()));

		isSendingPTS_ = false;

		DPRINTF("Starting mplayer slave...");
		if (!process_->start())
		{
			DPRINTF("Could not execute mplayer");
			DEXITMETHOD("MPlayer::play(%s)", (const char *)media->location().toString().utf8());
			return InitializationFailed;
		}
	}
	else
	{
		params_ = "Using recycled mplayer process.\n" + args.join(" ") + "\n";
		messages_ = QString::null;
		playbackWasStarted_ = false;
		errorDetected_ = false;
		quitState_ = QuitNull;

		DPRINTF("Recycling mplayer slave");
		QString command = "loadfile \"" + media->location().toString() + "\"";

		if (mplayerSupportsPausingKeepToogle && isPaused_)
			command = "pausing_keep " + command + "\npause";

		sendCommand(command);
		isPaused_ = false;
	}

	sendCurrentVolume();

	watchdogCheck_ = 0;
	timer_.start(1000);

	emit started();

	isOSDVisible_ = false;
	isTemporaryOSDEnabled_ = false;

	DEXITMETHOD("MPlayer::play(%s)", (const char *)media->location().toString().utf8());
	return PlaybackInitiated;
}

void MPlayer::processReadyReadStdout()
{
	DENTERMETHOD("MPlayer::processReadyReadStdout()");
	if (process_)
		parseLines();
	DEXITMETHOD("MPlayer::processReadyReadStdout()");
}

void MPlayer::processExited()
{
	DENTERMETHOD("MPlayer::processExited()");

	if (process_)
	{
		parseLines();

		currentProcessArgs_.clear();

		DPRINTF("emit errorMessage?");

		if (!messages_.isNull() && !playbackWasStarted_ && (process_->exitStatus() != 0 || errorDetected_))
		{
			DPRINTF("emit errorMessage!");
			emit errorMessage(params_ + "\n" + messages_);
		}
	}

	DEXITMETHOD("MPlayer::processExited()");
}

void MPlayer::sendCommand(const QString &c)
{
	DENTERMETHOD("MPlayer::sendCommand(\"%s\")", (const char *)c.utf8());

	if (!isRunning())
	{
		DEXITMETHOD("MPlayer::sendCommand(\"%s\") MPlayer not running!", (const char *)c.utf8());
		return;
	}

#ifndef WINDOWS
	process_->writeToStdin(c + "\n");
#else
	// work around a bug in Q... Windows Edition 3.3.8, not present in 3.3.7
	QString cmd = c + "\n";
    QByteArray tmp = cmd.local8Bit();
    tmp.resize(cmd.length());
    process_->writeToStdin(tmp);
#endif

	process_->flushStdin();

	DEXITMETHOD("MPlayer::sendCommand(\"%s\")", (const char *)c.utf8());
}

void MPlayer::stop(bool emitQuitSignal)
{
	DENTERMETHOD("MPlayer::stop(%s)", emitQuitSignal ? "true" : "false");

	if (!isRunning())
	{
		DEXITMETHOD("MPlayer::stop() MPlayer not running!");
		return;
	}

	if (mplayerSupportsIdle)
	{
		sendCommand("stop");
		timer_.stop();

		// if playback is paused and we are stopping,
		// keep the isPaused flag, so we can handle it correctly in the play method
		if (!mplayerSupportsPausingKeepToogle)
			isPaused_ = false;

		buffer_ = "";
		playbackWasStarted_ = false;
		errorDetected_ = false;
	}
	else
		quitMplayer(emitQuitSignal);

	DEXITMETHOD("MPlayer::stop()");
}

void MPlayer::quitMplayer(bool emitQuitSignal)
{
	DENTERMETHOD("MPlayer::quitMplayer(%s)", emitQuitSignal ? "true" : "false");

	if (!isRunning())
	{
		DEXITMETHOD("MPlayer::quitMplayer() MPlayer not running!");
		return;
	}

	if (!emitQuitSignal)
		blockSignals(true);

	// timeout timer...
	QTime time;
	time.start();
	bool timedOut = false;

	sendCommand("quit");
	while (process_->isRunning() && !timedOut)
	{
		timedOut = time.elapsed() > 500;
		usleep(10000); // sleep 10 ms
	}

	if (timedOut)
	{
		timedOut = false;
		// try to terminate the process if it hasn't
		// already exited...
		process_->tryTerminate();

		// wait for shutdown, give process time to
		// clean up.
		while (process_->isRunning() && !timedOut)
		{
			timedOut = time.elapsed() > 500;
			usleep(10000); // sleep 10 ms
		}

		// if friendly termination didn't work out,
		// kill the process.
		if (timedOut)
		{
			process_->kill(); // final judgement .
			usleep(10000); // sleep 10 ms
		}
	}

	process_->kill(); // final judgement .
	currentProcessArgs_.clear();

	if (process_)
	{
		scheduleDestructionOfProcess(process_);
		process_ = NULL;
	}

	emit quit(); // also triggers cleanUp();

	if (!emitQuitSignal)
		blockSignals(false);

	DEXITMETHOD("MPlayer::quitMplayer()");
}

void MPlayer::cleanUp()
{
	DENTERMETHOD("MPlayer::cleanUp()");

	if (logStream_.device())
	{
		delete logStream_.device();
		logStream_.unsetDevice();
	}

	timer_.stop();
	isPaused_ = false;
	buffer_ = "";
	currentProcessArgs_.clear();

	DEXITMETHOD("MPlayer::cleanUp()");
}

void MPlayer::parseLines()
{
	if (parsing_)
		return;

	DENTERMETHOD("MPlayer::parseLines()");

	int newPlaytime = -1;

	while (process_ && process_->canReadLineStdout())
	{
		parsing_ = true;
		QString line = process_->readLineStdout();

		DPRINTF("LINE stdout: %s", (const char*)(line.utf8()));

		if (!playbackWasStarted_)
			messages_ += line + "\n";

		if (logStream_.device())
		{
			logStream_ << line << "\n";
			logStream_.device()->flush();
		}

		if (line.startsWith("PTS=") && !mplayerHasGetTimePos)
		{
			if (!isSendingPTS_)
			{
				DPRINTF("mplayer is sending PTS.");
				isSendingPTS_ = true;
			}

			int playtime = line.mid(4).toInt();

			playbackWasStarted_ = true;

			newPlaytime = playtime;
		}
		else if (line.startsWith("ANS_TIME_POSITION="))
		{
			QString positionString = line.mid(18);
			int posDot = positionString.find(".");
			int playtime = 0;

			if (posDot > -1)
			{
				playtime = positionString.left(posDot).toInt();
				if (positionString.mid(posDot + 1, 1).toInt() > 5)
					++playtime;
			}
			else
				playtime = positionString.toInt();

			playbackWasStarted_ = true;
			watchdogCheck_ = 0;

			newPlaytime = playtime;
		}
		else if (mplayerSupportsIdle && line.find("EOF code: 1") != -1)
		{
			quitState_ = QuitEOF;
			QTimer::singleShot(1, this, SLOT(pollMPlayer()));
			newPlaytime = -1;
		}
		else if (line.startsWith("Exiting..."))
		{
			DPRINTF("OMG, I read Exiting!");

			// ここですぐにemitしてもいい気がする
			if (line.find("Quit") != -1)
			{
				quitState_ = QuitUser;
			}
			else if (line.find("End of file") != -1)
			{
				quitState_ = QuitEOF;
			}

			QTimer::singleShot(1, this, SLOT(pollMPlayer()));

			newPlaytime = -1;
		}
		else if (line.startsWith("Unknown option") ||
				line.startsWith("FATAL") ||
				line.startsWith("Cannot find codec") ||
				line.startsWith("File not found ") ||
				line.startsWith("Failed to open ") ||
				line.startsWith("- MPlayer crashed") ||
				line.startsWith("MPlayer interrupted by signal"))
		{
			playbackWasStarted_ = false;
			errorDetected_ = true;
			newPlaytime = -1;
		}
		else if (line.startsWith("Seek failed")) {
			// -hr-mp3-seekのとき、末尾以降にシークしようとするとSeek failedの
			// メッセージを吐きながら帰ってこなくなる。mplayer側の修正が必要。
			stop();
		}

		emit output(line);
	}

	parsing_ = false;

	if (newPlaytime > -1)
	{
		newPlaytime_ = newPlaytime;
		QTimer::singleShot(1, this, SLOT(updatedValues()));
	}

	DEXITMETHOD("MPlayer::parseLines()");
}

void MPlayer::updatedValues()
{
	if (newPlaytime_ > -1)
	{
		DENTERMETHOD("currentPlayTime(%d)", newPlaytime_);
		emit currentPlayTime(newPlaytime_);
		DEXITMETHOD("currentPlayTime(%d)", newPlaytime_);
		newPlaytime_ = -1;
	}
}

bool MPlayer::isPaused()
{
	return isPaused_;
}

void MPlayer::pause()
{
	if (!isRunning())
		return;

	sendCommand("pause");

	if (isPaused_) {
		isPaused_ = false;
		emit resumed();
	} else {
		isPaused_ = true;
		emit paused();
	}
}

void MPlayer::seekRelativeSeconds(int seconds)
{
	QString command;
	sendCommand(command.sprintf("seek %d 0", seconds));
	if (isPaused_)
	    sendCommand("pause");

	updatePosition();
}

void MPlayer::seekAbsolutePercentage(int percent)
{
	QString command;
	sendCommand(command.sprintf("seek %d 1", percent));
	if (isPaused_)
	    sendCommand("pause");

	updatePosition();
}

void MPlayer::seekAbsoluteSeconds(int seconds)
{
	QString command;
	sendCommand(command.sprintf("seek %d 2", seconds));
	if (isPaused_)
	    sendCommand("pause");

	updatePosition();
}

void MPlayer::osdShowProgressBar(int seconds)
{
	QString command;
	sendCommand(command.sprintf("osd_show_progbar %d", seconds));
}

void MPlayer::toggleOSD()
{
	if (isTemporaryOSDEnabled_)
		return;

	if (isOSDVisible_)
	{
		isOSDVisible_ = false;
		sendCommand("osd 0");
	}
	else
	{
		isOSDVisible_ = true;
		sendCommand("osd 3");
	}
}

void MPlayer::enableTemporaryOSD()
{
	if (isTemporaryOSDEnabled_)
		return;

	savedOSDVisiblity_ = isOSDVisible_;
	if (!isOSDVisible_)
		toggleOSD();

	isTemporaryOSDEnabled_ = true;
}

void MPlayer::disableTemporaryOSD()
{
	isTemporaryOSDEnabled_ = false;

	if (savedOSDVisiblity_ != isOSDVisible_)
		toggleOSD();
}

void MPlayer::setBrightness(int value)
{
	QString args;
	sendCommand(args.sprintf("brightness %d", value));
}

void MPlayer::pollMPlayer()
{
	DENTERMETHOD("MPlayer::pollMPlayer()");

	if (!isRunning())
	{
		if (quitState_ == QuitEOF)
		{
			DPRINTF("QuitEOF");
			cleanUp();
			emit eof();
		}
		else
		{
			DPRINTF("QuitUser");
			emit quit(); // also triggers cleanUp();
		}
	}
	else if (mplayerSupportsIdle)
	{
		if (quitState_ == QuitEOF)
		{
			DPRINTF("EOF");
			quitState_ = QuitNull;
			timer_.stop();
			isPaused_ = false;
			buffer_ = "";
			playbackWasStarted_ = false;
			errorDetected_ = false;

			emit eof();
		}
		else
			updatePosition();
	}
	else
		updatePosition();

	DEXITMETHOD("MPlayer::pollMPlayer()");
}

void MPlayer::updatePosition()
{
	if (!isSendingPTS_ && !isPaused_)
	{
		if (mplayerHasGetTimePos)
		{
			sendCommand("get_time_pos");
			++watchdogCheck_;
		}
		else if (mplayerHasGetPercentPos)
		{
			sendCommand("get_percent_pos");
			++watchdogCheck_;
		}

		if (watchdogCheck_ > 5)
		{
			// For some reason mplayer doesn't send the current position
			DPRINTF("Something went wrong and MPlayer doesn't respond to our position requests, quitting MPlayer.");
			quitMplayer(true);
		}
	}
}

void MPlayer::handleSignal(int sig)
{
	DENTERMETHOD("MPlayer::handleSignal(%d)", sig);

	DPRINTF("Got signal %d.", sig);
	quitMplayer(false);
	exit(-sig);

	DEXITMETHOD("MPlayer::handleSignal(%d)", sig);
}

void MPlayer::systemVolumeChanged(int level)
{
	sendCurrentVolume();
}

void MPlayer::sendCurrentVolume()
{
	if (isRunning() && useSoftVolume())
	{
		DPRINTF("Setting volume to %d...", qSystemVolume.getCurrentLevel());
		sendCommand("volume " + QString::number(qSystemVolume.getCurrentLevel()) + " 1");
	}
}
