/*
 * Copyright (C) 2007-2008 Andre Beckedorf <evilJazz _AT_ katastrophos _DOT_ net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include "transitiontimer.h"

TransitionTimer::TransitionTimer()
	:	transitionTime_(1000),
		timeBetweenSteps_(0),
		startValue_(0),
		currentValue_(0),
		endValue_(0),
		step_(0)
{
	timer_ = new QTimer();
	connect(timer_, SIGNAL(timeout()), this, SLOT(timerTimedOut()));
	time_.start();
}

TransitionTimer::~TransitionTimer()
{
	delete timer_;
}

void TransitionTimer::timerTimedOut()
{
	if (isRunning())
	{
		updateCurrentValue();
		
		emit transitionStep(startValue_, endValue_, currentValue_, step_);
		
		if (step_ == 1)
		{
			stopTimer();
			emit transitionFinished();
			step_ = 0; // reset step value, so we don't trigger it again...
		}
	}
}

void TransitionTimer::stopTimer()
{
	if (isRunning())
	{
		timer_->stop();
		emit transitionStopped();		
	}	
}

void TransitionTimer::transitionTo(float value)
{
	if (value == currentValue_ || (value == endValue_ && isRunning()))
		return;
	
	stopTransition();
	
	startValue_ = currentValue_;
	endValue_ = value;
	
	timer_->start(timeBetweenSteps_);
	time_.restart();
	
	timerTimedOut();	
}

void TransitionTimer::stopTransition()
{
	timerTimedOut();
	stopTimer();
}

float TransitionTimer::currentValue() 
{
	updateCurrentValue();
	return currentValue_;
}

void TransitionTimer::setCurrentValue(float value)
{
	stopTimer();
	currentValue_ = value;
}

void TransitionTimer::updateCurrentValue()
{
	// Don't update the current value if the timer isn't running
	// since time_.elapsed() will yield wrong results...
	if (isRunning())
	{
		step_ = transitionFunction(time_.elapsed(), transitionTime_);
		
		if (step_ > 1)
			step_ = 1;
		
		currentValue_ = startValue_ + (endValue_ - startValue_) * step_;
	}
}

float TransitionTimer::transitionFunction(uint elapsedTime, uint endTime)
{
	float rel = (float)elapsedTime / (float)endTime;
	if (rel < 1)
		return sinf(rel * M_PI / 2);
	else
		return 1;
}
