/*
 * Copyright (C) 2007-2008 Andre Beckedorf <evilJazz _AT_ katastrophos _DOT_ net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <unistd.h>
#include <qapplication.h>
#include <qimage.h>
#include <qdir.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qregexp.h>
#include <kmdcodec.h>

#include "cachedimageprovider.h"
#include "backgroundtasks.h"
#include "debug.h"

/* CachedImage */

CachedImage::CachedImage(CustomCachedImageProvider *owner, QString name, QImage *image, void *data)
	:	owner_(owner), 
		name_(name),
		image_(image),
		data_(data),
		finished_(false),
		readFromCache_(false),
		priority_(TaskProcessingController::TaskPriorityHigh),
		currentTask_(NULL),
		cacheFileName_("")
{
#ifdef DEBUG_ACTIVE_CACHEDIMAGES
	if (owner_)
	{
		owner_->mutexActiveCachedImages_.lock();
		owner_->activeCachedImages_.append(this);
		owner_->mutexActiveCachedImages_.unlock();
	}
#endif
}
	
CachedImage::~CachedImage()
{
	if (image_)
		delete image_;
	
#ifdef DEBUG_ACTIVE_CACHEDIMAGES
	if (owner_)
	{
		owner_->mutexActiveCachedImages_.lock();
		owner_->activeCachedImages_.removeRef(this);
		owner_->mutexActiveCachedImages_.unlock();
	}
#endif	
}

/* CacheImageCache */

CacheImageCache::~CacheImageCache()
{
	clear();
}

void CacheImageCache::deleteItem(Item d)
{
	CachedImage *cachedImage = static_cast<CachedImage *>(d);
	cachedImage->owner()->cachedImageRemovedFromCache(cachedImage);
}

/* ImageLoadTask */

class ImageLoadTask : public Task
{
	friend class CustomCachedImageProvider;
public:
	ImageLoadTask(CustomCachedImageProvider *owner, CachedImage* cachedImage):
		Task(), owner_(owner), cachedImage_(cachedImage)
	{
	}
	
	virtual void run()
	{
		QImage *image = owner_->threadedCreateImage(cachedImage_);
		
		owner_->mutexCacheImageSharedDataAccess_.lock();
		cachedImage_->setImage(image);
		cachedImage_->setFinished(true);
		owner_->mutexCacheImageSharedDataAccess_.unlock();
	}
	
	virtual void notifyTaskFinished()
	{
		Task::notifyTaskFinished();
		owner_->loadTaskFinished(this);
	}
	
private:
	CustomCachedImageProvider *owner_;
	CachedImage *cachedImage_;
};


/* ImageSaveTask */

class ImageSaveTask : public Task
{
	friend class CustomCachedImageProvider;
public:
	ImageSaveTask(CustomCachedImageProvider *owner, CachedImage* cachedImage):
		Task(), owner_(owner), cachedImage_(cachedImage)
	{
	}

	virtual void run()
	{
		owner_->threadedSaveImageToCacheDirectory(cachedImage_);
	}

	virtual void notifyTaskFinished()
	{
		Task::notifyTaskFinished();
		owner_->saveTaskFinished(this);
	}
	
private:
	CustomCachedImageProvider *owner_;
	CachedImage *cachedImage_;
};


/* CustomCachedImageProvider */

CustomCachedImageProvider::CustomCachedImageProvider(QObject *owner, const char *name)
	:	QObject(owner, name),
#ifdef DEBUG_ACTIVE_CACHEDIMAGES	
		mutexActiveCachedImages_(true),
#endif
		owner_(owner),
		imageCacheDirectory_("/tmp/imagecache"),
		imageCache_(),
		mutexCache_(true),
		nullImageNameList_(),
		prefetchedNamesTimestampList_(),
		prefetchNamesList_(),
		prefetching_(false),
		prefetchTimer_(),		
		imagesWaitList_(),
		mutexImagesWaitList_(true),
		mutexCacheImageSharedDataAccess_(true)
{
	imageCache_.setMaxCost(10);
	imageCache_.setAutoDelete(true);
	
	imagesWaitList_.setAutoDelete(false);
	
	prefetchedNamesTimestampList_.setAutoDelete(true);
	connect(&prefetchTimer_, SIGNAL(timeout()), this, SLOT(prefetchTimerTimeOut()));
	
	backgroundTasks_ = new TaskProcessingController(this);
	
#ifdef QTOPIA
	connect(&eventPollTimer_, SIGNAL(timeout()), this, SLOT(eventPollTimerTimedOut()));
#endif
}

CustomCachedImageProvider::~CustomCachedImageProvider()
{
	DENTERMETHOD("CachedImageProvider::~CachedImageProvider");
	
	imagesWaitList_.clear();
	imageCache_.clear();
	
	backgroundTasks_->waitUntilFinished();
	
	delete backgroundTasks_;
	
	DPRINTF("backgroundTasks deleted.");
	DEXITMETHOD("CachedImageProvider::~CachedImageProvider");
}

void CustomCachedImageProvider::setImageCacheDirectory(const QString &directory)
{
	if (directory != imageCacheDirectory_)
	{
		imageCacheDirectory_ = directory;
		
		if (!QDir(directory).exists())
			QDir().mkdir(directory, true);
	}
}

QString CustomCachedImageProvider::buildCacheFilename(const QString &name)
{
	QString cacheFilename = imageCacheDirectory_ + "/";
	
	//cacheFilename += QString(KCodecs::base64Encode(QCString(name), false));
	
	cacheFilename += QString(KMD5(name.utf8()).hexDigest().data());
		
	return cacheFilename;
}

void CustomCachedImageProvider::lock()
{
	mutexCache_.lock();
	mutexImagesWaitList_.lock();
	backgroundTasks_->lockQueues();
}

void CustomCachedImageProvider::unlock()
{
	backgroundTasks_->unlockQueues();
	mutexImagesWaitList_.unlock();
	mutexCache_.unlock();	
}

bool CustomCachedImageProvider::imageExists(const QString &name, bool searchNullImageList)
{
	lock();
	
	bool result = false;
	
	if (imageCache_.find(name, false))
		result = true;
	else if (imagesWaitList_.find(name))
		result = true;
	else if (searchNullImageList && nullImageNameList_.find(name))
		result = true;
	else
		result = QFile::exists(buildCacheFilename(name));
	
	unlock();
	
	return result;
}

bool CustomCachedImageProvider::containsData(void *data)
{
	bool result = false;
	
	mutexCache_.lock();
	
	QCacheIterator<CachedImage> it(imageCache_);
	
	if (it.toFirst())
	{
		while (!it.atLast())
		{
			if (it.current()->data() == data)
			{
				result = true;
				break;
			}
			
			++it;
		}
	}
	
	if (!result)
	{
		mutexImagesWaitList_.lock();
		
		QDictIterator<CachedImage> it(imagesWaitList_);

		while (it.current())
		{
			if (it.current()->data() == data)
			{
				result = true;
				break;
			}

			++it;
		}
		
		mutexImagesWaitList_.unlock();
	}
	
	mutexCache_.unlock();
	
	return result;
}

QImage* CustomCachedImageProvider::loadImageFromCacheDirectory(const QString &cacheFileName)
{
	return new QImage(cacheFileName);
}

QImage *CustomCachedImageProvider::internalGetImage(const QString &name, void *data, bool returnImage, TaskProcessingController::TaskPriority prio)
{
	DMEMSTAT();
	
	mutexCache_.lock();
	
	// try to find image in image memory cache...
	QImage *resultImage = NULL;
	
	if (!nullImageNameList_.find(name))
	{
		CachedImage *cachedImage = imageCache_.find(name, true);
		
		if (!cachedImage)
		{
			DPRINTF("Image '%s' wasn't found in the memory cache, looking in wait list...", (const char *)(name.utf8()));
			
			mutexImagesWaitList_.lock();
			
			cachedImage = imagesWaitList_.find(name);

			if (cachedImage && prio == TaskProcessingController::TaskPriorityHigh)
			{
				// Try to reschedule from low priority to high priority
				// if that is still possible, ie. the task isn't running yet.
				backgroundTasks_->lockQueues();
				mutexCacheImageSharedDataAccess_.lock();
				
				imagesWaitList_.remove(name);
				imageCache_.insert(name, cachedImage);
				
				if (cachedImage->currentTask() && cachedImage->priority() == TaskProcessingController::TaskPriorityLow)
				{
					backgroundTasks_->rescheduleTask(cachedImage->currentTask(), TaskProcessingController::TaskPriorityHigh, true);
					cachedImage->setPriority(TaskProcessingController::TaskPriorityHigh);
				}
				
				mutexCacheImageSharedDataAccess_.unlock();
				backgroundTasks_->unlockQueues();
			}
			
			mutexImagesWaitList_.unlock();
		}
				
		// if not found, look into our image file cache...
		if (!cachedImage)
		{
			DPRINTF("Image '%s' wasn't found in the memory cache...", (const char *)(name.utf8()));
			
			QString cacheFilename = buildCacheFilename(name);
			if (QFile::exists(cacheFilename))
			{
				DPRINTF("Image '%s' exists as cache file, loading image '%s'...", (const char *)(name.utf8()), (const char *)(cacheFilename.utf8()));
				
				// TODO: Loading of cached image might be potentially slow...
				resultImage = loadImageFromCacheDirectory(cacheFilename); 
				cachedImage = new CachedImage(this, name, resultImage, data);
				cachedImage->setFinished(true);
				cachedImage->setCacheFileName(cacheFilename);
				cachedImage->setReadFromCache(true);
				initializeCachedImage(cachedImage);
				
				if (prio == TaskProcessingController::TaskPriorityHigh)
					imageCache_.insert(name, cachedImage);
				else
				{
					mutexImagesWaitList_.lock();
					imagesWaitList_.insert(name, cachedImage);
					mutexImagesWaitList_.unlock();
				}
			}
			else
			{
				DPRINTF("Image '%s' does not exist as cache file '%s', creating image...", (const char *)(name.utf8()), (const char *)(cacheFilename.utf8()));
				
				// Create an empty cached image.
				// The image_ member will be initialized by the threadedCreateImage method below...
				cachedImage = new CachedImage(this, name, NULL, data);
				cachedImage->setFinished(false);
				cachedImage->setCacheFileName(cacheFilename);
				initializeCachedImage(cachedImage);
				
				if (prio == TaskProcessingController::TaskPriorityHigh)
					imageCache_.insert(name, cachedImage);
				else
				{
					mutexImagesWaitList_.lock();
					imagesWaitList_.insert(name, cachedImage);
					mutexImagesWaitList_.unlock();
				}
	
				ImageLoadTask *loadTask = new ImageLoadTask(this, cachedImage);
				
				cachedImage->setCurrentTask(loadTask);
				cachedImage->setPriority(prio);
				
				backgroundTasks_->addTask(loadTask, prio);
	
#ifdef QTOPIA
				if (!eventPollTimer_.isActive())
					eventPollTimer_.start(250);
#endif
				
				if (returnImage)
					resultImage = preloadImage(name, data);
			}
		}
		else if (returnImage)
		{
			// if the current image is already in the cache but hasn't finished
			// loading yet, just return the preloadImage again otherwise
			// return the image in the cache...
			if (!cachedImage->finished())
				resultImage = preloadImage(name, data);
			else
			{
				DPRINTF("Image '%s' was found in the memory cache...", (const char *)(name.utf8()));
				resultImage = cachedImage->image();
			}
		}
	}
	else if (returnImage)
	{
		DPRINTF("Image '%s' was found in null image name list...", (const char *)(name.utf8()));
		resultImage = nullImage(name, data);
	}
	
	mutexCache_.unlock();
	
#ifdef DEBUG_ACTIVE_CACHEDIMAGES	
	debugPrintActiveCachedImages();
#endif
	
	if (returnImage)
		return resultImage;
	else
		return NULL;
}

QImage *CustomCachedImageProvider::getImage(const QString &name, void *data)
{
	return internalGetImage(name, data, true, TaskProcessingController::TaskPriorityHigh);
}

void CustomCachedImageProvider::queue(const QString &name, void *data)
{
	internalGetImage(name, data, false, TaskProcessingController::TaskPriorityLow);
}

void CustomCachedImageProvider::prefetchHint(const QStringList &names)
{
	QTime currentTimestamp = QTime::currentTime();
	QTime expireTimestamp = currentTimestamp.addSecs(-300);
	
	for (QStringList::ConstIterator it = names.begin(); it != names.end(); ++it)
	{
		if (QFile::exists(buildCacheFilename(*it)))
		{
			QTime *timestamp = prefetchedNamesTimestampList_.find(*it);
					
			if (!timestamp || (timestamp && *timestamp < expireTimestamp))
			{
				DPRINTF("Adding %s to prefetch list...", (const char *)(*it).utf8());
				
				prefetchNamesList_.append(*it);
				prefetchedNamesTimestampList_.insert(*it, new QTime(currentTimestamp));
				
				if (!prefetchTimer_.isActive())
				{
					prefetching_ = false;
					prefetchTimer_.start(10);
				}
			}
		}
	}
}

void CustomCachedImageProvider::prefetchTimerTimeOut()
{
	if (prefetching_) return;
	
	prefetching_ = true;
	
	if (!prefetchNamesList_.isEmpty())
	{
		QString name = prefetchNamesList_.first();
		prefetchNamesList_.remove(prefetchNamesList_.begin());
		
		DPRINTF("prefetching %s ...", (const char *)name.utf8());
		
		QFile file(buildCacheFilename(name));

		if (file.open(IO_Raw | IO_ReadOnly))
		{
			char buffer[65536];
			
			int maxFetchBytes = 1024 * 1024; // 1 MB

			QTime time;
			time.start();
			
	        while (!file.atEnd() && file.at() < maxFetchBytes)
	        {
	        	// only read a block any x milliseconds
	        	// in order not to overload the I/O
	        	while (time.elapsed() < 10) 
	        		qApp->processOneEvent();
	        	 
	        	file.readBlock(&buffer[0], 65536);
	        	
	        	time.restart();
	        }

	        file.close();
		}
	}
	else
		prefetchTimer_.stop();
	
	prefetching_ = false;
}

void CustomCachedImageProvider::flushMemoryCache()
{
	mutexCache_.lock();
	imageCache_.clear();
	mutexCache_.unlock();
	
#ifdef DEBUG_ACTIVE_CACHEDIMAGES
	debugPrintActiveCachedImages();
#endif
}

#ifdef DEBUG_ACTIVE_CACHEDIMAGES

void CustomCachedImageProvider::debugPrintActiveCachedImages()
{
	lock();
	qDebug("--- Statistics of cached images for %p ---", this);
	qDebug("activeCachedImages_.count(): %d", activeCachedImages_.count());
	qDebug("imageCache_.count(): %d", imageCache_.count());
	qDebug("imageWaitList_.count(): %d", imagesWaitList_.count());
	qDebug(" ");
	unlock();
}

#endif

void CustomCachedImageProvider::flushCache()
{
	flushMemoryCache();
	// TODO: Delete all entries in the cache directory...
}

void CustomCachedImageProvider::loadTaskFinished(ImageLoadTask *task)
{
	mutexCache_.lock();
	
	DPRINTF("Loading of cache image '%s' finished.", (const char *)(task->cachedImage_->name().utf8()));
	
	mutexCacheImageSharedDataAccess_.lock();
	
	task->cachedImage_->setCurrentTask(NULL);
	
	if (task->cachedImage_->finished())
	{
		if (task->cachedImage_->image())
		{
			emit imageCached(task->cachedImage_);
			
			mutexImagesWaitList_.lock();
			
			// Check if the calling ImageLoadTask is on the wait for destruction list...
			if (imagesWaitList_.find(task->cachedImage_->name()))
			{
				// If yes, add a ImageSaveTask, since it would be a shame to drop the results
				// now that we've wasted so many cycles on it...
				
				DPRINTF("Adding ImageSaveTask for '%s'.", (const char *)(task->cachedImage_->name().utf8()));
				
				// no need to re-add the cached image to imagesWaitList_ 
				// because it's already inserted...
				
				// Add a save task
				ImageSaveTask *saveTask = new ImageSaveTask(this, task->cachedImage_);
				task->cachedImage_->setCurrentTask(saveTask);
				backgroundTasks_->addTask(saveTask);
			}
			mutexImagesWaitList_.unlock();
		}
		else
		{
			DPRINTF("The returned image is null. Adding to null image name list.");
			
			nullImageNameList_.insert(task->cachedImage_->name(), (void *)1);
			emit imageIsNull(task->cachedImage_);
			
			freeCachedImage(task->cachedImage_);
			task->cachedImage_ = NULL;			
		}
	}
	else
	{
		// The task was removed and nothing happened with the cached image
		// so remove it too...
		freeCachedImage(task->cachedImage_);
		task->cachedImage_ = NULL;		
	}

	mutexCacheImageSharedDataAccess_.unlock();
	
	mutexCache_.unlock();
}

void CustomCachedImageProvider::cachedImageRemovedFromCache(CachedImage *cachedImage)
{
	DPRINTF("Cache image '%s' was removed from memory cache.", (const char *)(cachedImage->name().utf8()));
	
	// Got signal from the memory image cache, that it is dropping the cachedImage...
	
	backgroundTasks_->lockQueues();
	mutexCacheImageSharedDataAccess_.lock();
	
	if (cachedImage->finished())
	{
		// Skip adding another save image task or deleting the object if the image 
		// is marked as finished but still has a task assigned (ie. the save image task)
		if (!cachedImage->currentTask())
		{
			// Check whether we need to save the image, that is, check if
			// the current image doesn't already exist in the file cache...
			QString cacheFilename = buildCacheFilename(cachedImage->name());
			if (!QFile::exists(cacheFilename) && !cachedImage->readFromCache())
			{
				mutexCache_.lock();
				mutexImagesWaitList_.lock();
				
				DPRINTF("Adding ImageSaveTask for '%s'.", (const char *)(cachedImage->name().utf8()));

				if (!imagesWaitList_.find(cachedImage->name()))
					imagesWaitList_.insert(cachedImage->name(), cachedImage);

				// Add a save task
				ImageSaveTask *saveTask = new ImageSaveTask(this, cachedImage);
				cachedImage->setCurrentTask(saveTask);
				backgroundTasks_->addTask(saveTask);
				
				mutexImagesWaitList_.unlock();
				mutexCache_.unlock();
			}
			else
			{
				freeCachedImage(cachedImage);
				cachedImage = NULL;
			}
		}
		// This cached image still has a save task assigned so put it onto the wait list...
		else
		{
			mutexImagesWaitList_.lock();
			
			if (!imagesWaitList_.find(cachedImage->name()))
				imagesWaitList_.insert(cachedImage->name(), cachedImage);
			
			mutexImagesWaitList_.unlock();
		}
	}
	else
	{
		if (cachedImage->currentTask())
		{
			// Check, if we can still remove the task from the list...
			if (!backgroundTasks_->taskRunning(cachedImage->currentTask()) &&
				backgroundTasks_->taskEnqueued(cachedImage->currentTask()))
			{
				backgroundTasks_->removeTask(cachedImage->currentTask());
				freeCachedImage(cachedImage);
				cachedImage = NULL;
			}
			// The task is currently running so put it onto the wait list
			// and wait for the loadTaskFinished callback...
			// We'll create the ImageSaveTask there.
			else
			{
				mutexImagesWaitList_.lock();
				
				if (!imagesWaitList_.find(cachedImage->name()))
					imagesWaitList_.insert(cachedImage->name(), cachedImage);
				
				mutexImagesWaitList_.unlock();
			}
		}
		else
		{
			freeCachedImage(cachedImage);
			cachedImage = NULL;			
		}
	}
	
	mutexCacheImageSharedDataAccess_.unlock();
	backgroundTasks_->unlockQueues();
}

void CustomCachedImageProvider::initializeCachedImage(CachedImage *cachedImage)
{
	// descendants do heavy lifting here ;)
}

void CustomCachedImageProvider::finalizeCachedImage(CachedImage *cachedImage)
{
	// descendants do heavy lifting here ;)
}

void CustomCachedImageProvider::saveTaskFinished(ImageSaveTask *task)
{
	mutexCache_.lock();
	mutexImagesWaitList_.lock();
	mutexCacheImageSharedDataAccess_.lock();

	// check whether we actually have an image and the cache file exists...
	if (task->cachedImage_->image() && QFile::exists(task->cachedImage_->cacheFileName()))
		emit imageSaved(task->cachedImage_);
	
	if (imageCache_.find(task->cachedImage_->name(), false))
	{
		DPRINTF("ImageSaveTask for '%s' finished. Not removing image from memory, still in use...", (const char *)(task->cachedImage_->name().utf8()));		

		// if the cached image was rescheduled and put into the cacheImage list again
		// just set the current task to NULL to cause cachedImageRemovedFromCache to
		// clean up and remove the cached image from memory once the cached image is 
		// dropped from the cache list...
		task->cachedImage_->setCurrentTask(NULL);
	}
	else
	{
		// only clean up the cached image if it's in the wait list (ie. it was not rescheduled)
		// or not in any other list...
		DPRINTF("ImageSaveTask for '%s' finished. Removing image from wait list and memory...", (const char *)(task->cachedImage_->name().utf8()));
		freeCachedImage(task->cachedImage_);
		task->cachedImage_ = NULL;
	}

	mutexCacheImageSharedDataAccess_.unlock();
	mutexImagesWaitList_.unlock();
	mutexCache_.unlock();
}

void CustomCachedImageProvider::freeCachedImage(CachedImage *cachedImage)
{
	imageCache_.take(cachedImage->name()); // explicitly take the item, so cachedImageRemovedFromCache doesn't get called...
	imagesWaitList_.remove(cachedImage->name());
	cachedImage->setCurrentTask(NULL);
	finalizeCachedImage(cachedImage);
	delete cachedImage;
}

void CustomCachedImageProvider::eventPollTimerTimedOut()
{
#ifdef QTOPIA
	// Deal with a problem in non-MT Qtopia, where
	// the thread events won't fire by themselfes...

	if (backgroundTasks_->idle())
		eventPollTimer_.stop();
		
	qApp->processEvents();
#endif
}