/***************************************************************************
 *   Copyright (C) 2004 by Steven Scott                                    *
 *   progoth@progoth.com                                                   *
 *                                                                         *
 *   Copyright (C) 2007-2008 Andre Beckedorf                               *
 *   evilJazz _AT_ katastrophos _DOT_ net                                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "qregexp.h"
#include <qurl.h>

#ifdef QTOPIA
#include <qpe/config.h>
#else
#include <config.h>
#endif

#include <qdir.h>

#include "qhttp.h"
#include "kmdcodec.h"
#include "audioscrobbler.h"
#include "debug.h"
#include "qfile.h"
#include "helpers.h"

#define UA_QSCROBBLER "qscrobbler 1.0"

unsigned int AudioScrobbler::MIN_SONG_LENGTH =      30;
unsigned int AudioScrobbler::MAX_SONG_LENGTH = 30 * 60;
unsigned int AudioScrobbler::MIN_PLAY_TIME   =     240;

AudioScrobbler::AudioScrobbler( QObject *parent, const QString &clientName, const QString &clientVersion )
  : QObject(parent), m_buf(NULL), m_subcachechanged(false), m_itemsFetchedWithPostData(0), m_clientName(clientName),
    m_clientVersion(clientVersion),
    m_postaddress(QString::null), m_challenge(QString::null), m_handshake_done(false),
    m_username(QString::null), m_password(QString::null), m_cacheFileName(QString::null), m_submissionPaused(false),
    m_interval(0), m_cachesaveinterval(300000), m_songplaying(false)
{
	qDebug("1");
	m_http = new QHttp();
	connect(m_http, SIGNAL(requestFinished( int, bool )), this, SLOT(httpRequestFinished( int, bool )));

	qDebug("2");
    cacheTimer = new QTimer( this );
    connect( cacheTimer, SIGNAL(timeout()), this, SLOT(saveCacheToDisk()) );
    cacheTimer->start( m_cachesaveinterval );

    qDebug("3");
    songPlayTimer = new QTimer( this );
    connect( songPlayTimer, SIGNAL(timeout()), this, SLOT(newSong()) );

    qDebug("4");
    hsTimer = new QTimer( this );
    connect( hsTimer, SIGNAL(timeout()), this, SLOT(doHandshake()) );

    qDebug("5");
    intervalTimer = new QTimer( this );
    connect( intervalTimer, SIGNAL(timeout()), this, SLOT(submit()) );
}

AudioScrobbler::~AudioScrobbler()
{
    saveCacheToDisk();
    if (m_http)
    	delete m_http;
}

void AudioScrobbler::sendStatus(const QString &status)
{
	m_laststatus = status;
	emit statusMessage(status);
}

void AudioScrobbler::clearCache()
{
	m_subcache.clear();
	m_subcachechanged = true;
	saveCacheToDisk();
}

int AudioScrobbler::play( const QString &artist, const QString &songtitle, const QString &album,
               const QString &musicbrainzid, unsigned int length )
{
    if( songPlayTimer->isActive() ) songPlayTimer->stop();

    //sendStatus(tr("Playing"));

    if( length < MIN_SONG_LENGTH || length > MAX_SONG_LENGTH ) return -1;

    QString art = artist;
    QString st = songtitle;

    if( (artist.isEmpty() || artist.find( QString("various"),0,FALSE) >= 0) && songtitle.find( "/" ) >= 0 )
    {
        QRegExp re( "([^/]+)\\s*/\\s*(.+)" );
        if( songtitle.find( re ) >= 0 )
        {
            art = re.capturedTexts()[1].stripWhiteSpace();
            st = re.capturedTexts()[2].stripWhiteSpace();
        }
    }

    m_currentsong.artist = art;
    m_currentsong.songtitle = st;
    m_currentsong.album = album;
    m_currentsong.mbid = musicbrainzid;
    m_currentsong.length = length;

    m_playtime = QDateTime::currentDateTime();
    m_songplaying = true;
    m_remainingtime = ( (1+length/2) < MIN_PLAY_TIME) ? 1+length/2 : MIN_PLAY_TIME;
    songPlayTimer->start( m_remainingtime * 1000, true );
    m_requiredtime = m_remainingtime;

    emit songAdded(artist, songtitle, album, musicbrainzid, length);

    return m_requiredtime;
}

void AudioScrobbler::pause()
{
    if( !songPlayTimer->isActive() || !m_songplaying ) return;
    songPlayTimer->stop();
    QDateTime cur = QDateTime::currentDateTime();

    m_remainingtime -= m_playtime.secsTo( cur );

    //sendStatus( tr( "Paused" ) );
    m_songplaying = false;
}

void AudioScrobbler::unpause()
{
    if( songPlayTimer->isActive() || m_songplaying ) return;
    m_songplaying = true;
    m_playtime = QDateTime::currentDateTime();
    songPlayTimer->start( m_remainingtime * 1000, true );

    //sendStatus( tr( "Playing" ) );
}

void AudioScrobbler::seek()
{
    if( songPlayTimer->isActive() )
    {
        songPlayTimer->stop();
        //sendStatus( tr( "Seek; submission cancelled" ) );
    }
}

void AudioScrobbler::stop()
{
    if( songPlayTimer->isActive() )
        songPlayTimer->stop();

    m_songplaying = false;
    emit songRemoved();
    //sendStatus( tr( "Stopped" ) );
}

void AudioScrobbler::newSong()
{
    m_subcache.addSubmission( m_currentsong );
    m_subcachechanged = true;
    saveLockedCacheToDisk();
    emit songAddedForSubmission( m_currentsong.artist + " - " + m_currentsong.songtitle );
    submit();
}

void AudioScrobbler::submit()
{
	DENTERMETHOD("AudioScrobbler::submit()");

	if (intervalTimer->isActive() || !m_handshake_done || m_submissionPaused)
	{
		DEXITMETHOD("AudioScrobbler::submit()");
		return;
	}

    if (m_subcache.submissionCount())
        doRequest(m_postaddress, m_subcache.postData(m_username, md5Response(), &m_itemsFetchedWithPostData));

    DEXITMETHOD("AudioScrobbler::submit()");
}

void AudioScrobbler::reinitialize()
{
    if (hsTimer->isActive())
    	hsTimer->stop();

    if (intervalTimer->isActive())
    	intervalTimer->stop();

    m_handshake_done = false;
    cacheTimer->changeInterval(m_cachesaveinterval);

    hsTimer->start(5000, true);
}

void AudioScrobbler::setUsername( const QString &username )
{
	m_username = username;
	m_handshake_done = false;
}

void AudioScrobbler::setPassword( const QString &password )
{
    KMD5 m( password.utf8() );
    m_password = m.hexDigest();
    m_handshake_done = false;
}

void AudioScrobbler::setMD5Password( const QString &passwordMD5 )
{
	m_password = passwordMD5;
	m_handshake_done = false;
}

void AudioScrobbler::setSubmissionPaused(bool pause)
{
	m_submissionPaused = pause;

	if (!pause && (!songPlayTimer->isActive() || !m_songplaying))
		submit();
}

void AudioScrobbler::httpRequestFinished(int res, bool error)
{
	DENTERMETHOD("AudioScrobbler::httpRequestFinished(%d, %s)", res, error ? "true" : "false");

	if (!error)
	{
	    QString response = m_buf->buffer();

		sendStatus( "Got response: " + response );

	    //if (!response.isEmpty())
	    //{
	    //	DPRINTF("Response: %s", (const char*)(response.utf8()));
	    	parseResponse(response);
	    //}
	}
	else
	{
		sendStatus( tr("Error while trying to talk to server: ") + m_http->errorString());

		m_handshake_done = false;
        hsTimer->start(120000, true);
	}

    DEXITMETHOD("AudioScrobbler::httpRequestFinished(%d, %s)", res, error ? "true" : "false");
}

QString AudioScrobbler::md5Response() const
{
	KMD5 m( (m_password + m_challenge).utf8() );
    return m.hexDigest();
}

void AudioScrobbler::setInterval( unsigned int interval )
{
    // need to do something here?
    m_interval = interval * 1000;
}

void AudioScrobbler::setCacheSaveInterval( unsigned int interval )
{
    m_cachesaveinterval = interval * 1000;
}

void AudioScrobbler::setCacheFileName(const QString &fileName)
{
	m_cacheFileName = fileName;

	if (m_subcache.count() > 0)
		saveCacheToDisk();
	else
		loadCacheFromDisk();
}

void AudioScrobbler::parseResponse( const QString &response )
{
	DENTERMETHOD("AudioScrobbler::parseResponse(...)");

	DPRINTF("response: %s", (const char *)response.utf8());

	static QRegExp uptodate( "^UPTODATE\\s*(\\S{32})\\s*(\\S+)\\s*(INTERVAL\\s+(\\d+))?", false );
    static QRegExp update( "^UPDATE\\s+(\\S+)\\s*(\\S{32})\\s*(\\S+)\\s*(INTERVAL\\s+(\\d+))?", false );
    static QRegExp failed( "^FAILED\\s+([^\\n]*)\\s*(INTERVAL\\s+(\\d+))?", false );
    static QRegExp baduser( "^BADUSER\\s*(INTERVAL\\s+(\\d+))?", false );
    static QRegExp badauth( "^BADAUTH\\s*(INTERVAL\\s+(\\d+))?", false );
    static QRegExp ok( "^OK\\s*(INTERVAL\\s+(\\d+))?", false );

    if( !m_handshake_done )
    {
        if( response.find( uptodate ) >= 0 )
        {
            m_handshake_done = true;
            QStringList re = uptodate.capturedTexts();
            m_challenge = re[1];
            m_postaddress = re[2];
            if( re.count() > 3 )
                setInterval( re[5].toUInt() );
            //hsTimer->stop();
            sendStatus( tr("Successful handshake") );
            if( !m_interval ) intervalTimer->start( 2 * 1000, true );
        }
        else if( response.find( update ) >= 0 )
        {
            m_handshake_done = true;
            QStringList re = update.capturedTexts();
            m_challenge = re[2];
            m_postaddress = re[3];
            if( re.count() > 4 )
                setInterval( re[6].toUInt() );
            //hsTimer->stop();
            sendStatus( tr("Update needed: %1").arg( re[1] ) );
            if( !m_interval ) intervalTimer->start( 2 * 1000, true );
        }
        else if( response.find( failed ) >= 0 )
        {
            QStringList re = failed.capturedTexts();

            if( re.count() > 2 )
                setInterval( re[3].toUInt() );

            hsTimer->start( 10 * 60 * 1000, true );
            sendStatus( tr("Handshake failed: %1").arg( re[1] ) );
        }
        else if( response.find( baduser ) >= 0 )
        {
            QStringList re = baduser.capturedTexts();
            if( re.count() > 1 )
                setInterval( re[2].toUInt() );
            // let's do something to not submit again until the user is reset

            intervalTimer->start( 24 * 60 * 60 * 1000, true );
            sendStatus( tr("Error: Bad username") );
        }
        else
        {
            sendStatus( tr("Could not parse server response: %1").arg(response) );
            hsTimer->start( 120000, true );
        }
    }
    else // we've already handshaken
    {
        if( response.find( failed ) >= 0 )
        {
        	m_handshake_done = false;
        	hsTimer->start( 120000, true );

        	QStringList re = failed.capturedTexts();
            if( re.count() > 2 )
                setInterval( re[3].toUInt() );

            sendStatus( tr("Submission failed: %1").arg( re[1] ) );
        }
        else if( response.find( badauth ) >= 0 )
        {
        	m_handshake_done = false;
        	QStringList re = badauth.capturedTexts();
            if( re.count() > 1 )
                setInterval( re[2].toUInt() );
            sendStatus( tr("Bad password") );
        }
        else if( response.find( ok ) >= 0 )
        {
            QStringList re = ok.capturedTexts();
            if( re.count() > 1 )
                setInterval( re[2].toUInt() );

            DPRINTF("m_itemsFetchedWithPostData: %d", m_itemsFetchedWithPostData);
            m_subcache.removeFromTop(m_itemsFetchedWithPostData);
            //m_subcache.clear();
            m_subcachechanged = true;
            saveLockedCacheToDisk();
            sendStatus( tr("Submit succeeded") );
        }
        else
        {
            sendStatus( tr("Can't parse server response: %1").arg(response) );
            intervalTimer->start( 2 * 60 * 1000, true );
        }
//        m_cache_mutex.unlock();
    }

    if( m_interval && !intervalTimer->isActive() )
    	intervalTimer->start( m_interval, true );

    DEXITMETHOD("AudioScrobbler::parseResponse(...)");
}

void AudioScrobbler::saveCacheToDisk()
{
//	m_cache_mutex.lock();
    saveLockedCacheToDisk();
//  m_cache_mutex.unlock();
}

void AudioScrobbler::saveLockedCacheToDisk()
{
	if (m_subcachechanged && m_cacheFileName != "")
	{
		DPRINTF("Saving changed cache content...");
		QFile f(m_cacheFileName);
	    f.open(IO_WriteOnly);
	    QDataStream s(&f);
	    s << m_subcache;
	    f.close();
	    m_subcachechanged = false;
	    emit cacheSaved();
	}
}

void AudioScrobbler::loadCacheFromDisk()
{
    if (QFile(m_cacheFileName).exists())
    {
	    QFile f(m_cacheFileName);
	    f.open(IO_ReadOnly);
	    QDataStream s(&f);
	    m_subcache.clear();
	    s >> m_subcache;
	    f.close();
    }
}

void AudioScrobbler::doHandshake()
{
    DENTERMETHOD("AudioScrobbler::doHandshake()");

	if (!m_handshake_done && m_username != "" && m_password != "")
	{
        DPRINTF("Initializing handshake...");
		doRequest( QUrl(handshakeAddress() + m_username) );
	}

    DEXITMETHOD("AudioScrobbler::doHandshake()");
}

void AudioScrobbler::doRequest( const QUrl &address, const QString &postdata )
{
	DENTERMETHOD("AudioScrobbler::doRequest(...)");
	DPRINTF("address: %s", (const char*)(address.toString().utf8()));
	DPRINTF("postdata: %s", (const char*)(postdata.utf8()));

	sendStatus(tr("Firing request"));

	m_http->abort();

    QCString tmp;
    if (!postdata.isEmpty())
        tmp = postdata.utf8();

    if (m_buf)
    	delete m_buf;
    m_buf = new QBuffer();

    QString proxyHost = address.host();
   	int proxyPort = 80;

   	getHTTPProxySettings(proxyHost, proxyPort);

    m_http->setHost(proxyHost, proxyPort);

    if (postdata.isEmpty())
    {
    	DPRINTF("Using GET method...");

    	QHttpRequestHeader header("GET", address.toString());
    	header.setValue("Connection", "Keep-Alive");
    	header.setValue("Host", address.host());
    	header.setValue("User-Agent", UA_QSCROBBLER);
    	m_http->request(header, (QIODevice *) 0, m_buf);

    	//m_http->get(address.toString(), &m_buf);
    }
    else
    {
    	DPRINTF("Using POST method...");

    	QHttpRequestHeader header("POST", address.toString());
    	header.setValue("Connection", "Keep-Alive");
    	header.setValue("Host", address.host());
    	header.setValue("User-Agent", UA_QSCROBBLER);
    	header.setValue("Content-Type", "application/x-www-form-urlencoded");
    	m_http->request(header, postdata.utf8(), m_buf);

    	//m_http->post(address.toString(), postdata.utf8(), &m_buf);
    }

    DEXITMETHOD("AudioScrobbler::doRequest(...)");
}
