#include <math.h>
#include <qpainter.h>

#include "coverartflow.h"
#include "coverartprovider.h"
#include "debug.h"
#include "mediadatabase.h"
#include "sqlite3.h"


/* CoverArtImageProvider */

CoverArtImageProvider::CoverArtImageProvider(QObject *owner, const char *name)
	:	ReflectionImageProvider(owner, name)
{

}

QImage* CoverArtImageProvider::threadedCreateImage(CachedImage *cachedImage)
{
	DENTERMETHOD("CoverArtImageProvider::threadedCreateImage(CachedImage *cachedImage)");

	QImage srcImage;
	QImage *dstImage = NULL;

	CoverArtItem *item = static_cast<CoverArtItem *>(cachedImage->data());

	DPRINTF("CoverArtItem '%s' - '%s' - '%s' ", (const char *)(item->artist.utf8()), (const char *)(item->album.utf8()), (const char *)(item->filename.utf8()));

	bool imageWasLoaded = gCoverArtProvider.loadCoverDirectly(srcImage, item->artist, item->album, item->filename);

	if (imageWasLoaded && !srcImage.isNull())
	{
		DMEMSTAT();

		DPRINTF("Scaling image '%s'...", (const char *)(cachedImage->name().utf8()));

		QImage scaledImage = prepareImage(srcImage, refSize().width(), refSize().height() - reflectionHeight(), reflectionHeight());

		DPRINTF("Assigning scaled image '%s'...", (const char *)(cachedImage->name().utf8()));
		dstImage = new QImage(scaledImage);
	}
	else
		DPRINTF("Failed loading image '%s'", (const char *)(cachedImage->name().utf8()));

	DEXITMETHOD("CoverArtImageProvider::threadedCreateImage(CachedImage *cachedImage)");

	return dstImage;
}

void CoverArtImageProvider::initializeCachedImage(CachedImage *cachedImage)
{
	CoverArtItem *item = static_cast<CoverArtItem *>(cachedImage->data());
	CoverArtItem *newItem = new CoverArtItem();

	// Clone item
	*newItem = *item;

	cachedImage->setData(newItem);
}

void CoverArtImageProvider::finalizeCachedImage(CachedImage *cachedImage)
{
	// Delete cloned item
	delete static_cast<CoverArtItem *>(cachedImage->data());
}

CoverArtItem CoverArtImageProvider::createCoverArtItemFromMedia(const Media &info)
{
	CoverArtItem result;
	result.playlistID = 0;
	result.album = info.album();
	result.artist = info.artist();
	result.key = createKey(info.artist(), info.album());
	result.filename = info.location().toString();
	return result;
}


/* CoverArtFlow */

CoverArtFlow::CoverArtFlow(ReflectionImageProvider *imageProvider, LayeredPaintBox *parent)
	: 	ReflectionImageFlow(imageProvider, parent),
		PlayListViewExtension(),
		coverArtItems_(),
		playlistIDLookupDict_(),
		active_(false)
{
	coverArtItems_.setAutoDelete(true);

	setTransitionTime(500);
	setVerticalOffset(ReflectionHeight / 4);
	//setConstantVerticalOffset(-80);
	setReflectionHeight(ReflectionHeight);
	setSlideSize(QSize(SlideWidthHeight, SlideWidthHeight + ReflectionHeight));

	connect(&preloadImagesTimer_, SIGNAL(timeout()), this, SLOT(preloadImages()));
	connect(&focusAlbumTimer_, SIGNAL(timeout()), this, SLOT(focusCurrentAlbum()));
}

CoverArtFlow::~CoverArtFlow()
{

}

void CoverArtFlow::setPlayListView(PlayList *playList)
{
	if (playListView())
		disconnect(playListView(), SIGNAL(currentChanged(QListViewItem *)), this, SLOT(playListCurrentChanged(QListViewItem *)));

	PlayListViewExtension::setPlayListView(playList);

	if (playListView())
		connect(playListView(), SIGNAL(currentChanged(QListViewItem *)), this, SLOT(playListCurrentChanged(QListViewItem *)));
}

void CoverArtFlow::extensionGraphExecutionFinished()
{
	if (active() && itemCount() > 0)
	{
		if (playListView()->currentItem())
		{
			PlayListItem *item = static_cast<PlayListItem *>(playListView()->currentItem());
			jumpTo((int)playlistIDLookupDict_[item->id()]);
		}
		else
			jumpTo(0);
	}
}

void CoverArtFlow::playListCurrentChanged(QListViewItem *lvi)
{
	if (active() && itemCount() > 0)
	{
		if (lvi)
		{
			PlayListItem *item = static_cast<PlayListItem *>(lvi);
			transitionTo((int)playlistIDLookupDict_[item->id()]);
		}
		else
			transitionTo(0);
	}
}

void CoverArtFlow::transitionFinished()
{
	DENTERMETHOD("CoverArtFlow::transitionFinished()");

	CustomPerspectiveImageFlow::transitionFinished();
	focusAlbumTimer_.start(500, true);

	DEXITMETHOD("CoverArtFlow::transitionFinished()");
}

void CoverArtFlow::focusCurrentAlbum()
{
	int itemIndex = floor(currentValue());
	int itemIndexViaPlaylist = -1;

	// Try to get the item index via the currently selected playlist item...
	if (playListView()->currentItem())
		itemIndexViaPlaylist = (int)playlistIDLookupDict_[static_cast<PlayListItem *>(playListView()->currentItem())->id()];

	// Only select the first song if the particular group isn't selected yet
	if (itemIndex != itemIndexViaPlaylist)
	{
		// Make sure the album is visible in the list
		// This is a workaround to avoid having the first song
		// of this particular group as last item in the visible area...
		if (itemIndex < coverArtItems_.count() - 1)
		{
			PlayListItem *playlistItem = playListView()->findItemByID(coverArtItems_.at(itemIndex + 1)->playlistID);
			playListView()->ensureItemVisible(playlistItem);
		}
		else
			playListView()->ensureItemVisible(playListView()->lastItem());

		PlayListItem *playlistItem = playListView()->findItemByID(coverArtItems_.at(itemIndex)->playlistID);
		playListView()->ensureItemVisible(playlistItem);
		playListView()->setCurrentItem(playlistItem);
	}
}

void CoverArtFlow::mouseDoubleClickEvent(QMouseEvent *e)
{
	focusCurrentAlbum();
	playListView()->triggerPlaybackIntent();
}

void CoverArtFlow::paintOverlay(QPainter &p, const QPixmap &pixmap)
{
	ReflectionImageFlow::paintOverlay(p, pixmap);

	p.setPen(QColor(255, 255, 255));
	p.setBrush(NoBrush);

	int itemIndex = floor(currentValue());

	if (itemIndex >= 0 && itemIndex < coverArtItems_.count())
	{
		CoverArtItem *item = coverArtItems_.at(itemIndex);

		//int boxHeight = floor((reflectionHeight() - verticalOffset()) * scale());
		int boxHeight = 40;
#ifdef QTOPIA
		int fontHeight = 15; // TODO: FIXME!
#else
		int fontHeight = p.fontInfo().pixelSize();
#endif

		QRect rect;

		if (boxHeight < fontHeight * 2)
		{
			rect = QRect(0, height() - fontHeight - 2 - 40, width(), fontHeight + 2);
			p.drawText(rect, Qt::AlignCenter, item->artist + " - " + item->album);
		}
		else
		{
			rect = QRect(0, height() - boxHeight - 40, width(), boxHeight);
			p.drawText(rect, Qt::AlignCenter, item->artist + "\n" + item->album);
		}
	}

	if (imageProvider() && imageProvider()->isWorking())
	{
		p.drawText(5, 25, "Loading...");
	}
}

void CoverArtFlow::setActive(bool value)
{
	active_ = value;

	if (active_)
		execute();
	else
	{
		coverArtItems_.clear();
		playlistIDLookupDict_.clear();
		imageProvider()->flushMemoryCache();
	}
}

void CoverArtFlow::execute()
{
	if (!active())
		return;

	DENTERMETHOD("CoverArtFlow::execute()");
	DTIMERINIT(timer);

	QString query =
		"SELECT artist, media_location.location || media.filename FROM media "
		"LEFT JOIN media_location ON media_location.location_id = media.location_id "
		"WHERE media_id = ?1;";

	sqlite3_stmt *vm2;
	sqlite3_prepare_v2(playListView()->mediaDatabase()->db(), query.utf8(), -1, &vm2, 0);

	playListView()->mediaDatabase()->dbDebug(query);

	query = "SELECT playlist_id, media_id, album FROM " + dbSafeString(inputViewSource()) + ";";
	sqlite3_stmt *vm;
	sqlite3_prepare_v2(playListView()->mediaDatabase()->db(), query.utf8(), -1, &vm, 0);

	playListView()->mediaDatabase()->dbDebug(query);

	CoverArtItem *item = NULL;

	coverArtItems_.clear();
	playlistIDLookupDict_.clear();

	setItemCount(0);

	QString previousAlbum = QString::null;

	while (sqlite3_step(vm) == SQLITE_ROW)
	{
		QString currentAlbum = QString::fromUtf8((const char *)sqlite3_column_text(vm, 2));
		int playlistID = sqlite3_column_int(vm, 0);

		// Build groups by album
		if (previousAlbum != currentAlbum)
		{
			int mediaID = sqlite3_column_int(vm, 1);
			sqlite3_bind_int(vm2, 1, mediaID);

			if (sqlite3_step(vm2) == SQLITE_ROW)
			{
				item = new CoverArtItem();

				item->playlistID = playlistID;
				item->album = currentAlbum;
				item->artist = QString::fromUtf8((const char *)sqlite3_column_text(vm2, 0));
				item->key = CoverArtImageProvider::createKey(item->artist, currentAlbum);
				item->filename = QString::fromUtf8((const char *)sqlite3_column_text(vm2, 1));

				coverArtItems_.append(item);

				previousAlbum = currentAlbum;
			}

			sqlite3_reset(vm2);
		}

		if (coverArtItems_.count() > 1)
			playlistIDLookupDict_.insert(playlistID, (int *)(coverArtItems_.count() - 1));
	}

	sqlite3_finalize(vm);
	sqlite3_finalize(vm2);

	setItemCount(coverArtItems_.count());

	preloadImages();

	DTIMERPRINT(timer, "Grouping albums...");
	DEXITMETHOD("CoverArtFlow::execute()");
}

QImage CoverArtFlow::getImage(int index)
{
	if (index >= 0 && index < coverArtItems_.count())
	{
		CoverArtItem *item = coverArtItems_.at(index);
		if (item)
		{
			QImage *image = imageProvider()->getImage(item->key, item);

			if (image != NULL)
				return *image;
			else
				return NULL;
		}
		else
			return NULL;
	}
	else
		return NULL;
}

void CoverArtFlow::preloadImages()
{
	imageProvider()->lock();

	QStringList keys;

	for (CoverArtItem *item = coverArtItems_.first(); item != NULL; item = coverArtItems_.next())
	{
		if (!imageProvider()->imageExists(item->key))
			imageProvider()->queue(item->key, item);

		keys.append(item->key);
	}

	imageProvider()->prefetchHint(keys);

	imageProvider()->unlock();
}

void CoverArtFlow::imageWasCached(CachedImage *image)
{
	ReflectionImageFlow::imageWasCached(image);

	if (!preloadImagesTimer_.isActive())
		preloadImagesTimer_.start(1000, true);
}

void CoverArtFlow::reset()
{
	imageProvider()->resetNullImageNameList();
	imageProvider()->flushMemoryCache();
}

QString CoverArtFlow::outputViewSource()
{
	return inputViewSource();
}


/* ScrollBarSkinSupport */

ScrollBarSkinSupport::ScrollBarSkinSupport(Skin::PartsType scrollBarType, Skin::PartsType scrollBarSliderType, const SkinManager *skinmgr, PositionedLayer *owner)
	:	QObject(),
		scrollBarType_(scrollBarType),
		scrollBarSliderType_(scrollBarSliderType),
		owner_(owner)
{
	setSkin(skinmgr->skin());
	connect(skinmgr, SIGNAL(skinChanged(const Skin *)), this, SLOT(setSkin(const Skin *)));
}

void ScrollBarSkinSupport::setSkin(const Skin *skin)
{
	skin_ = skin;
	scrollBarParts_ = skin->lookup(scrollBarType_);
	scrollBarSliderParts_ = skin->lookup(scrollBarSliderType_);
}

int ScrollBarSkinSupport::buttonSize()
{
	SkinSections *partsSections = static_cast<SkinSections *>(scrollBarParts_->data());
	return partsSections->at(0)->hitWidth();
}

int ScrollBarSkinSupport::minSliderWidth()
{
	SkinSections *partsSections = static_cast<SkinSections *>(scrollBarParts_->data());
	return partsSections->at(0)->width() + partsSections->at(2)->width();
}

void ScrollBarSkinSupport::paintThreeSectionedParts(QPainter &p, const QRect &dstRect, const SkinParts *parts)
{
	// TODO: Optimize the crap below...

	SkinSections *partsSections = static_cast<SkinSections *>(parts->data());

	SkinSection *leftSection = partsSections->at(0);
	SkinSection *midSection = partsSections->at(1);
	SkinSection *rightSection = partsSections->at(2);

	// Left part
	p.drawPixmap(
		dstRect.left(), dstRect.top(),
		parts->image(),
		leftSection->x(), 0, leftSection->width(), parts->image().height()
	);

	// Mid part
	if (midSection->width() > 0)
	{
		uint drawnWidth = 0;
		uint xToDrawTo = dstRect.left() + midSection->x();

		int drawWidth = dstRect.width() - leftSection->width() - rightSection->width() - 1;

		if (drawWidth > 0)
			while (drawnWidth < drawWidth /* && xToDrawTo < owner_->width() */)
			{
				drawnWidth += midSection->width();

				uint widthToDraw = midSection->width();
				if (drawnWidth > drawWidth)
					widthToDraw -= drawnWidth - drawWidth;

				p.drawPixmap(
					xToDrawTo, dstRect.top(),
					parts->image(),
					midSection->x(), 0, widthToDraw, parts->image().height()
				);

				xToDrawTo += widthToDraw;
			}
	}

	// Right part
	p.drawPixmap(
		QMAX(dstRect.left() + leftSection->width(), dstRect.right() - rightSection->width()), dstRect.top(),
		parts->image(),
		rightSection->x(), 0, rightSection->width(), parts->image().height()
	);
}

void ScrollBarSkinSupport::paintOverlay(QPainter &p, const QRect &geometry, const QRect &slider)
{
	paintThreeSectionedParts(p, geometry, scrollBarParts_);
	paintThreeSectionedParts(p, slider, scrollBarSliderParts_);
}

/* SkinScrollBar */

SkinScrollBar::SkinScrollBar(Skin::PartsType scrollBarType, Skin::PartsType scrollBarSliderType, const SkinManager *skinmgr, LayeredPaintBox *parent)
	:	ScrollBar(parent),
		skinSupport_(scrollBarType, scrollBarSliderType, skinmgr, this)
{
	connect(skinmgr, SIGNAL(skinChanged(const Skin *)), this, SLOT(setSkin(const Skin *)));
}

void SkinScrollBar::paintOverlay(QPainter &p, const QPixmap &pixmap)
{
	skinSupport_.paintOverlay(p, geometry(), rectSlider());
}

/* SkinFlowScrollBar */

SkinFlowScrollBar::SkinFlowScrollBar(Skin::PartsType scrollBarType, Skin::PartsType scrollBarSliderType, const SkinManager *skinmgr, CustomFlow *flow, LayeredPaintBox *parent)
	:	FlowScrollBar(flow, parent),
		skinSupport_(scrollBarType, scrollBarSliderType, skinmgr, this)
{
	connect(skinmgr, SIGNAL(skinChanged(const Skin *)), this, SLOT(setSkin(const Skin *)));
}

void SkinFlowScrollBar::paintOverlay(QPainter &p, const QPixmap &pixmap)
{
	if (itemCount() <= 1)
		return;

	skinSupport_.paintOverlay(p, geometry(), rectSlider());
}
