/* -*- mode: c++; tab-width: 4; c-basic-offset: 4 -*- */
/*
 * Copyright (C) 2005 Atmark <atmarkat _AT_ msn _DOT_ com>
 *                    AGAWA Koji <i _AT_ atty _DOT_ jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <qapplication.h>
#ifdef QTOPIA
#include <qpe/config.h>
#else
#include <config.h>
#endif
#include <qfile.h>
#include <qtextstream.h>
#include <math.h>

#include "debug.h"
#include "configuration.h"
#include "action.h"
#include "addonmanager.h"

GlobalConfiguration & GlobalConfiguration::singleton()
{
	static GlobalConfiguration instance;
	return instance;
}

void GlobalConfiguration::read()
{
	appBinaryFilename_ = QFileInfo(QString(qApp->argv()[0])).absFilePath();
	appPath_ = QFileInfo(appBinaryFilename_).dirPath(true) + "/";

#ifdef WINDOWS
	appVolumePath_ = appPath_.left(3);
#else
	appVolumePath_ = "/"; // TODO: figure our mount point of volume...
#endif

	QString appGlobalIniFile = appPath_ + QFileInfo(appBinaryFilename_).baseName() + ".ini";

	DPRINTF("appGlobalIniFile: %s", (const char*)appGlobalIniFile.latin1());

	if (QFile::exists(appGlobalIniFile))
	{
		Config cf(appGlobalIniFile, Config::File);
		cf.setGroup("Application");
		selfcontainedMode_ = cf.readBoolEntry("selfcontained", false);
	}
	else
		selfcontainedMode_ = false;

	DPRINTF("selfcontainedMode: %s", selfcontainedMode_ ? "true" : "false");

	if (selfcontainedMode_)
	{
		configPath_ = appPath_ + "data/";
		skinPath_ = appPath_ + "skins/";
	}
	else
	{
#ifdef OSX
		skinPath_ = appPath_ + "../Resources/skins/";
		configPath_ = QDir::homeDirPath() + "/Library/Application Support/" + QString(APPNAME) + "/";
#else
		configPath_ = QDir::homeDirPath() + "/" + QString(CONFIGPATHNAME) + "/";

#ifdef QTOPIA
		skinPath_ =  "/opt/QtPalmtop/share/quasar/skins/";
#else
		QString path("/usr/share/quasar/skins");
		if (QDir(path).exists())
			skinPath_ = path + "/";
		else
		{
			path = "/usr/local/share/quasar/skins";
			if (QDir(path).exists())
				skinPath_ = path + "/";
			else
			{
				path = appPath_ + "skins/";
				if (QDir(path).exists())
					skinPath_ = path + "/";
				else
					skinPath_ = appPath_ + "../skins/";
			}
		}
#endif
#endif
	}

	QDir().mkdir(configPath_);

	configFilename_ = configPath_ + QString(CONFIGNAME);
	databaseFilename_ = configPath_ + QString(MEDIADBNAME);

	DPRINTF("configPath_: %s", (const char*)configPath_.latin1());
	DPRINTF("skinPath_: %s", (const char*)skinPath_.latin1());

	DPRINTF("configName_: %s", (const char*)configFilename_.latin1());
	DPRINTF("databaseFilename_: %s", (const char*)configFilename_.latin1());
}

QString GlobalConfiguration::getRelativeFilePath(const QString &input) const
{
	if (isRunningSelfcontained())
	{
		QString result;

		if (input.startsWith(configPath_))
		{
			result = "%CONFIG%/" + input.right(input.length() - configPath_.length());
		}
		else if (input.startsWith(appPath_))
		{
			result = "%APP%/" + input.right(input.length() - appPath_.length());
		}
#ifdef WINDOWS
		else if (input.startsWith(appVolumePath_))
		{
			result = "%VOL%/" + input.right(input.length() - appVolumePath_.length());
		}
#endif
		else
			result = input;

		DPRINTF("GlobalConfiguration::makeRelativeString(%s) result = %s", (const char *)input.latin1(), (const char *)result.latin1());

		return result;
	}
	else
		return input;
}

QString GlobalConfiguration::resolveRelativeFilePath(const QString &input) const
{
	QString result;

	if (input.startsWith("%CONFIG%"))
		result = configPath_ + input.mid(9);
	else if (input.startsWith("%APP%"))
		result = appPath_ + input.mid(6);
	else if (input.startsWith("%VOL%"))
		result = appVolumePath_ + input.mid(6);
	else
		result = input;

	DPRINTF("GlobalConfiguration::resolveRelativeString(%s) result = %s", (const char *)input.latin1(), (const char *)result.latin1());

	return result;
}

Configuration & Configuration::singleton()
{
	static Configuration instance;
	return instance;
}

/*!
  各設定項目のデフォルト値はここで定義します。
  設定ダイアログの「デフォルトに戻す」なども適切に処理されます。
 */
Configuration::Configuration()
	: lastDirectory(),
	  lastPlaylistDirectory(),
	  lastPlaylistFile(),
	  playOrder(PlayList::ORDER_NORMAL),
	  infoMode(SkinManager::NormalMode),
	  inputMode(PlayList::Select),
	  isBackendLoggingEnabled(false),
	  backendLogFileName("/tmp/quasar-log.txt"),
	  changePlayInfoLayoutInPortaitMode(false),
#ifdef LOWPERF
	  playlistAutoTriggerFilterInterval(500),
#else
	  playlistAutoTriggerFilterInterval(10),
#endif
	  skin1("default"),
	  skin2("default"),
	  automaticallySwitchToPlayInfoEnabled(false),
	  isAskQuitingEnabled(false),
	  registerAsRemoteConMaster(false),
	  qcopBroadcastingEnabled(false),
	  createTemporaryPlaybackStatusFile(false),
	  restorePreviousSessionOnRestartEnabled(true),
	  isOverviewVisible(false),
#ifdef QTOPIA
	  isBatteryPlusEnabled(false),
	  batteryPlusMode4Video(),
	  batteryPlusMode4Audio(),
#endif
	  mplayerPath("mplayer"),
	  isOverlayEnabled(true),
	  overviewSplitterPosition(120),
	  isCoverArtFlowVisible(false),
	  coverArtFlowSplitterPosition(120),
#ifdef QTOPIA
	  isTouchModeEnabled(false),
#else
	  isTouchModeEnabled(false),
#endif
	  isKineticScrollingEnabled(false),
	  isFullScreenEnabled(true),
	  isDoubleBufferingEnabled(true),
	  isCacheEnabled(true),
	  cacheSize(256),
	  isPrefetchFileEnabled(true),
	  framedropMethod(FRAMEDROP_NORMAL),
	  isAdditionalOptionsEnabled(false),
	  additionalOptions(""),
	  machineCategory_(MACHINE_GENERIC),
	  machineName_("")
{
	for (int i = 0; i < HOTKEY_CONTEXT_MAX; ++i)
		hotKeyMap[i] = gActionManager.defaultHotKey(static_cast<HotKeyContext>(i));

	setDefaultFormats();
	detectMachine();
	// GENERIC機種ではデフォルトでオーバレイを有効にしない
	if (machineCategory() == MACHINE_GENERIC) {
		isOverlayEnabled = false;
		isDoubleBufferingEnabled = false;
	}

	coverArtFlowCacheDirectory = qGlobalConfig.configPath() + "covercache";
	coverArtDirectory = qGlobalConfig.configPath() + "coverart";
}

Configuration::~Configuration()
{
}

//! 機種を判別します。
void Configuration::detectMachine()
{
	// まず、bvddが動いているかを確認し、動いていればSL-C3000系であると判断する
	// bvddが動いていなければ、SL-C3000系であってもGENERIC機種とする
	QFile modulesFile("/proc/modules");
	if (modulesFile.open(IO_ReadOnly)) {
		QTextStream s(&modulesFile);
		QString line;
		while ((line = s.readLine()) != QString::null) {
			if (line.startsWith("bvdd "))
				machineCategory_ = MACHINE_SLC3000;
		}
	}

	QFile productFile("/proc/deviceinfo/product");
	if (productFile.open(IO_ReadOnly)) {
		QTextStream s(&productFile);
		s >> machineName_;
		if (machineName_ == "SL-C860" || machineName_ == "SL-C760" ||
			machineName_ == "SL-C750" || machineName_ == "SL-C700")
			machineCategory_ = MACHINE_SLC700;
	} else
		machineName_ = "Unknown";
}

bool Configuration::isValidConfig() const
{
	Config cf(qGlobalConfig.configFilename(), Config::File);
	cf.setGroup(CONFIGGROUPNAME);

	bool isValid = false;
	if (cf.readNumEntry("CONFIG_VERSION", -1) == CONFIGVERSION)
		isValid = true;

	return isValid;
}

int Configuration::readDatabaseVersion() const
{
	Config cf(qGlobalConfig.configFilename(), Config::File);
	cf.setGroup(CONFIGGROUPNAME);
	return cf.readNumEntry("DB_VERSION", -1);
}

float Configuration::readAppVersion() const
{
	Config cf(qGlobalConfig.configFilename(), Config::File);
	cf.setGroup(CONFIGGROUPNAME);
	int appVersion = cf.readNumEntry("APP_VERSION", -1);
	return (appVersion > -1 ? (float)appVersion / 1000000 : -1);
}

// 設定ファイル内のキー名は基本的に変数名をそのまま使います。
void Configuration::write() const
{
	DPRINTF("Configuration::write()");
	Config cf(qGlobalConfig.configFilename(), Config::File);

	cf.setGroup(CONFIGGROUPNAME);
	cf.writeEntry("CONFIG_VERSION", CONFIGVERSION);
	cf.writeEntry("APP_VERSION", APPVERSION_MAJOR * 1000000 + APPVERSION_MINOR * (int)powf(10, 6 - QString::number(APPVERSION_MINOR).length()));
	cf.writeEntry("DB_VERSION", DBVERSION);
	cf.writeEntry("lastDirectory", qGlobalConfig.getRelativeFilePath(lastDirectory));
	cf.writeEntry("lastPlaylistDirectory", qGlobalConfig.getRelativeFilePath(lastPlaylistDirectory));
	cf.writeEntry("lastPlaylistFile", qGlobalConfig.getRelativeFilePath(lastPlaylistFile));
	cf.writeEntry("PlayOrder", playOrder);
	cf.writeEntry("InfoMode", infoMode);
	cf.writeEntry("InputMode", inputMode);
	cf.writeEntry("isBackendLoggingEnabled", isBackendLoggingEnabled);
	cf.writeEntry("backendLogFileName", qGlobalConfig.getRelativeFilePath(backendLogFileName));

	cf.writeEntry("changePlayInfoLayoutInPortaitMode", changePlayInfoLayoutInPortaitMode);
	cf.writeEntry("playlistAutoTriggerFilterInterval", playlistAutoTriggerFilterInterval);

	cf.writeEntry("skin1", skin1);
	cf.writeEntry("skin2", skin2);
	cf.writeEntry("automaticallySwitchToPlayInfoEnabled", automaticallySwitchToPlayInfoEnabled);
	cf.writeEntry("isAskQuitingEnabled", isAskQuitingEnabled);
	cf.writeEntry("registerAsRemoteConMaster", registerAsRemoteConMaster);
	cf.writeEntry("qcopBroadcastingEnabled", qcopBroadcastingEnabled);
	cf.writeEntry("createTemporaryPlaybackStatusFile", createTemporaryPlaybackStatusFile);
	cf.writeEntry("restorePreviousSessionOnRestartEnabled", restorePreviousSessionOnRestartEnabled);

	cf.writeEntry("isOverviewVisible", isOverviewVisible);

	cf.writeEntry("overviewSplitterPosition", overviewSplitterPosition);

	cf.writeEntry("isCoverArtFlowVisible", isCoverArtFlowVisible);
	cf.writeEntry("coverArtFlowSplitterPosition", coverArtFlowSplitterPosition);

	cf.writeEntry("isTouchModeEnabled", isTouchModeEnabled);
	cf.writeEntry("isKineticScrollingEnabled", isKineticScrollingEnabled);

#ifdef QTOPIA
	cf.setGroup("BatteryPlus");
	cf.writeEntry("isBatteryPlusEnabled", isBatteryPlusEnabled);
	cf.writeEntry("batteryPlusMode4Video", batteryPlusMode4Video);
	cf.writeEntry("batteryPlusMode4Audio", batteryPlusMode4Audio);
#endif

	cf.setGroup("MPlayer");
	cf.writeEntry("mplayerPath", qGlobalConfig.getRelativeFilePath(mplayerPath));
	cf.writeEntry("isFullScreenEnabled", isFullScreenEnabled);
	cf.writeEntry("isDoubleBufferingEnabled", isDoubleBufferingEnabled);
	cf.writeEntry("framedropMethod", framedropMethod);
	cf.writeEntry("isCacheEnabled", isCacheEnabled);
	cf.writeEntry("cacheSize", cacheSize);
	cf.writeEntry("isPrefetchFileEnabled", isPrefetchFileEnabled);
	cf.writeEntry("isAdditionalOptionsEnabled", isAdditionalOptionsEnabled);
	cf.writeEntry("additionalOptions", additionalOptions);

	cf.setGroup("Formats");
	cf.writeEntry("ambiguousFormatExtensions", ambiguousFormatExtensions_, ',');
	cf.writeEntry("fileExtensions", formatFileExtensions, '^');
	cf.writeEntry("mplayerArguments", formatMPlayerArguments, '^');

	cf.setGroup("CoverArt");
	cf.writeEntry("coverArtDirectory", qGlobalConfig.getRelativeFilePath(coverArtDirectory));
	cf.writeEntry("coverArtFlowCacheDirectory", qGlobalConfig.getRelativeFilePath(coverArtFlowCacheDirectory));

	for (int context = 0; context < HOTKEY_CONTEXT_MAX; ++context) {
		cf.setGroup(QString("HotKey_%1").arg(context));
		cf.writeEntry("count", static_cast<int>(hotKeyMap[context].count()));
		int i = 0;
		for (HotKeyMap::ConstIterator it = hotKeyMap[context].begin();
			 it != hotKeyMap[context].end();
			 ++it, ++i)
		{
			// writes entry: current index = key_value,accel_id
			cf.writeEntry(QString::number(i),
						  QString("%1,%2").arg(it.key()).arg(it.data()));
		}
	}

	qAddOnManager.writeToConfig(cf);
}

// Modified version of Config::readListEntry that supports empty entries in a list too
QStringList Configuration::readListEntry(Config &cf, const QString &key, const QChar &sep, bool allowEmptyEntries)
{
    QString s = cf.readEntry(key);
    if (s.isEmpty())
    	return QStringList();
    else
    	return QStringList::split(sep, s, allowEmptyEntries);
}

// 設定ファイル内のキー名は基本的に変数名をそのまま使います。
void Configuration::read()
{
	Config cf(qGlobalConfig.configFilename(), Config::File);

	cf.setGroup(CONFIGGROUPNAME);

	if (cf.readNumEntry("CONFIG_VERSION") != CONFIGVERSION)
		return;

	lastDirectory = qGlobalConfig.resolveRelativeFilePath(cf.readEntry("lastDirectory", lastDirectory));
	lastPlaylistDirectory = qGlobalConfig.resolveRelativeFilePath(cf.readEntry("lastPlaylistDirectory", lastPlaylistDirectory));
	lastPlaylistFile = qGlobalConfig.resolveRelativeFilePath(cf.readEntry("lastPlaylistFile", lastPlaylistFile));
	playOrder = static_cast<PlayList::PlayOrder>(cf.readNumEntry("PlayOrder", playOrder));
	infoMode = static_cast<SkinManager::InfoMode>(cf.readNumEntry("InfoMode", infoMode));
	inputMode = static_cast<PlayList::InputMode>(cf.readNumEntry("InputMode", inputMode));

	isBackendLoggingEnabled = cf.readBoolEntry("isBackendLoggingEnabled", isBackendLoggingEnabled);
	backendLogFileName = qGlobalConfig.resolveRelativeFilePath(cf.readEntry("backendLogFileName", backendLogFileName));

	changePlayInfoLayoutInPortaitMode = cf.readBoolEntry("changePlayInfoLayoutInPortaitMode", changePlayInfoLayoutInPortaitMode);
	playlistAutoTriggerFilterInterval = cf.readNumEntry("playlistAutoTriggerFilterInterval", playlistAutoTriggerFilterInterval);

	skin1 = cf.readEntry("skin", skin1);
	skin1 = cf.readEntry("skin1", skin1);
	skin2 = cf.readEntry("skin2", skin2);
	automaticallySwitchToPlayInfoEnabled = cf.readBoolEntry("automaticallySwitchToPlayInfoEnabled", automaticallySwitchToPlayInfoEnabled);
	isAskQuitingEnabled = cf.readBoolEntry("isAskQuitingEnabled", isAskQuitingEnabled);
	registerAsRemoteConMaster = cf.readBoolEntry("registerAsRemoteConMaster", registerAsRemoteConMaster);
	qcopBroadcastingEnabled = cf.readBoolEntry("qcopBroadcastingEnabled", qcopBroadcastingEnabled);
	createTemporaryPlaybackStatusFile = cf.readBoolEntry("createTemporaryPlaybackStatusFile", createTemporaryPlaybackStatusFile);
	restorePreviousSessionOnRestartEnabled = cf.readBoolEntry("restorePreviousSessionOnRestartEnabled", restorePreviousSessionOnRestartEnabled);

	isOverviewVisible = cf.readBoolEntry("isOverviewVisible", isOverviewVisible);

	overviewSplitterPosition = cf.readNumEntry("overviewSplitterPosition", overviewSplitterPosition);

	isCoverArtFlowVisible = cf.readBoolEntry("isCoverArtFlowVisible", isCoverArtFlowVisible);
	coverArtFlowSplitterPosition = cf.readNumEntry("coverArtFlowSplitterPosition", coverArtFlowSplitterPosition);

	isTouchModeEnabled = cf.readBoolEntry("isTouchModeEnabled", isTouchModeEnabled);
	isKineticScrollingEnabled = cf.readBoolEntry("isKineticScrollingEnabled", isKineticScrollingEnabled);

#ifdef QTOPIA
	cf.setGroup("BatteryPlus");
	isBatteryPlusEnabled = cf.readBoolEntry("isBatteryPlusEnabled", isBatteryPlusEnabled);
	batteryPlusMode4Video = cf.readEntry("batteryPlusMode4Video", batteryPlusMode4Video);
	batteryPlusMode4Audio = cf.readEntry("batteryPlusMode4Audio", batteryPlusMode4Audio);
#endif

	cf.setGroup("MPlayer");
	mplayerPath = qGlobalConfig.resolveRelativeFilePath(cf.readEntry("mplayerPath", mplayerPath));
	isFullScreenEnabled = cf.readBoolEntry("isFullScreenEnabled", isFullScreenEnabled);
	isDoubleBufferingEnabled = cf.readBoolEntry("isDoubleBufferingEnabled", isDoubleBufferingEnabled);
	framedropMethod = static_cast<FramedropMethod>(cf.readNumEntry("framedropMethod", framedropMethod));
	isCacheEnabled = cf.readBoolEntry("isCacheEnabled", isCacheEnabled);
	cacheSize = cf.readNumEntry("cacheSize", cacheSize);
	isPrefetchFileEnabled = cf.readBoolEntry("isPrefetchFileEnabled", isPrefetchFileEnabled);
	isAdditionalOptionsEnabled = cf.readBoolEntry("isAdditionalOptionsEnabled", isAdditionalOptionsEnabled);
	additionalOptions = cf.readEntry("additionalOptions", additionalOptions);

	cf.setGroup("Formats");

	ambiguousFormatExtensions_ = readListEntry(cf, "ambiguousFormatExtensions", ',');

	// use our modified readListEntry, because we need to consider empty entries too
	formatFileExtensions = readListEntry(cf, "fileExtensions", '^', true);
	formatMPlayerArguments = readListEntry(cf, "mplayerArguments", '^', true);

	if (formatFileExtensions.count() == 0 || formatMPlayerArguments.count() == 0)
		setDefaultFormats();
	else
	{
		// last entries are probably empty, so if they are, just delete them...
		// this circumvents an issue in the writeEntry implementation for QStringLists...
		if (formatFileExtensions.last().isEmpty())
		{
			formatFileExtensions.remove(formatFileExtensions.fromLast());
			formatMPlayerArguments.remove(formatMPlayerArguments.fromLast());
		}

		updateValidExtensionsList();
	}

	cf.setGroup("CoverArt");
	coverArtDirectory = qGlobalConfig.resolveRelativeFilePath(cf.readEntry("coverArtDirectory", coverArtDirectory));
	coverArtFlowCacheDirectory = qGlobalConfig.resolveRelativeFilePath(cf.readEntry("coverArtFlowCacheDirectory", coverArtFlowCacheDirectory));

	for (int context = 0; context < HOTKEY_CONTEXT_MAX; ++context) {
		cf.setGroup(QString("HotKey_%1").arg(context));
		int count = cf.readNumEntry("count", 0);
		if (count > 0) {
			hotKeyMap[context].clear();
			for (int i = 0; i < count; ++i) {
				QString value(cf.readEntry(QString::number(i)));
				if (!value.isEmpty()) {
					QStringList args(QStringList::split(',', value));
					// adds key_value -> accel_id
					hotKeyMap[context].insert(args[0].toInt(), args[1].toInt());
				}
			}
		}
	}

	qAddOnManager.readFromConfig(cf);
}

QString Configuration::mplayerBinLocation()
{
#ifdef OSX
	QString t(qGlobalConfig.appPath() + "../Resources/" + mplayerPath);
	if (QFile::exists(t))
		return t;
#else
	QString t(qGlobalConfig.appPath() + mplayerPath);

	#ifdef WINDOWS
	if (!t.endsWith(".exe", false))
		t += ".exe";
	#endif

	if (QFile::exists(t))
		return t;
#endif

	return mplayerPath;
}

void Configuration::setDefaultFormats()
{
	formatFileExtensions.clear();
	formatFileExtensions.append("ogg,oga");
	formatFileExtensions.append("flac");
	formatFileExtensions.append("mp3");
	formatFileExtensions.append("m4a,m4p,m4b,aac");
	formatFileExtensions.append("mp4,m4v");
	formatFileExtensions.append("avi,divx");
	formatFileExtensions.append("mkv");
	formatFileExtensions.append("mov");
	formatFileExtensions.append("wma");
	formatFileExtensions.append("wmv");
	formatFileExtensions.append("asf");
	formatFileExtensions.append("mpg,mpeg");

	formatMPlayerArguments.clear();
#ifdef LOWPERF
	formatMPlayerArguments.append("-ac vorbis");
#else
	formatMPlayerArguments.append("");
#endif
	formatMPlayerArguments.append("");
#ifdef LOWPERF
	formatMPlayerArguments.append("-ac mad -hr-mp3-seek");
#else
	formatMPlayerArguments.append("");
#endif
	formatMPlayerArguments.append("");
	formatMPlayerArguments.append("");
	formatMPlayerArguments.append("");
	formatMPlayerArguments.append("");
	formatMPlayerArguments.append("");
	formatMPlayerArguments.append("");
	formatMPlayerArguments.append("");
	formatMPlayerArguments.append("");
	formatMPlayerArguments.append("");

	updateValidExtensionsList();

	ambiguousFormatExtensions_.clear();
	// TagLib will scan both file types and most likely
	// will detect the audio track. That's why we need
	// to scan for video tracks using MPlayer...
	ambiguousFormatExtensions_.append("mp4");
	ambiguousFormatExtensions_.append("m4v");
}

void Configuration::updateValidExtensionsList()
{
	validExtensionsList_.clear();
	for (int i = 0; i < formatFileExtensions.count(); ++i)
	{
		QStringList exts = QStringList::split(',', formatFileExtensions[i], false);
		for (int j = 0; j < exts.count(); ++j)
			validExtensionsList_ << exts[j];
	}
}

const QString Configuration::mplayerArguments(QString fileExtension)
{
	fileExtension = fileExtension.lower();

	for (int i = 0; i < formatFileExtensions.count(); ++i)
		if (formatFileExtensions[i].contains(fileExtension, false))
			return formatMPlayerArguments[i];

	return QString::null;
}
