/*
 * Copyright (C) 2006-2007	Andre Beckedorf
 * 					 		<evilJazz _AT_ katastrophos _DOT_ net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <qstring.h>
#include <qdatastream.h>
#include <qtextcodec.h>

#include "media.h"
#include "mediaidentifier.h"
#include "mediadatabase.h"
#include "mplayer.h"
#include "quasar.h"
#include "configuration.h"
#include "debug.h"

#define EMPTY_TRACKNO ""

/* class MediaLocation */

MediaLocation::MediaLocation()
	: location_("")
{
}

MediaLocation::MediaLocation(const QString& url)
	: location_(url)
{
}

bool MediaLocation::isLocalFile() const
{
	return !location_.startsWith("http://") && !location_.startsWith("https://");
}

QString MediaLocation::fileName() const
{
	int index = location_.findRev('/');
	if (index > -1)
		return location_.right(location_.length() - index - 1);
	else
		return location_;
}

QString MediaLocation::dirPath() const
{
	int index = location_.findRev('/');
	if (index > -1)
		return location_.left(index);
	else
		return location_;
}

MediaLocation::operator QString() const
{
    return toString();
}


MediaAudio::MediaAudio(Media *parent)
	:	codec(""),
		bitrate(0),
		sampleRate(0),
		channels(0),
		owner(parent)
{
}

/* class MediaAudio */

bool MediaAudio::save()
{
	// insert audio related information for the media
	if (owner->isAudio() || owner->isVideo())
	{
		sqlite3_stmt *vm = owner->owningMediaDatabase()->stmtInsertMediaAudio;

		sqlite3_bind_int(vm, 1, owner->mediaid_);
		sqlite3_bind_text(vm, 2, codec.utf8(), -1, SQLITE_TRANSIENT);
		sqlite3_bind_int(vm, 3, bitrate);
		sqlite3_bind_int(vm, 4, sampleRate);
		sqlite3_bind_int(vm, 5, channels);

		sqlite3_step(vm);
		DHOP(owner->owningMediaDatabase()->dbDebug("stmtInsertMediaAudio"));
		sqlite3_reset(vm);

		return true;
	}
	else
		return false;
}

bool MediaAudio::load()
{
	DHPRINTF("MediaAudio::load() media_id = %d", owner->mediaid_);

	if (owner->mediaid_ == 0)
		return false;

	// load audio info...
	if (owner->isAudio() || owner->isVideo())
	{
		sqlite3_stmt *vm = owner->owningMediaDatabase()->stmtSelectMediaAudioByMediaID;
		sqlite3_bind_int(vm, 1, owner->mediaid_);
		if (sqlite3_step(vm) == SQLITE_ROW)
		{
			codec = QString::fromUtf8((const char *)sqlite3_column_text(vm, 1));
			bitrate = sqlite3_column_int(vm, 2);
			sampleRate = sqlite3_column_int(vm, 3);
			channels = sqlite3_column_int(vm, 4);

			sqlite3_reset(vm);

			return true;
		}
		else
		{
			DHOP(owner->owningMediaDatabase()->dbDebug());
			sqlite3_reset(vm);
			return false;
		}
	}
	else
		return false;
}

/* class MediaVideo */

MediaVideo::MediaVideo(Media *parent)
	: 	codec(""),
		bitrate(0),
		width(0),
		height(0),
		FPS(0),
		aspectRatio(0),
		owner(parent)
{
}

bool MediaVideo::save()
{
	// insert video related information for the media
	if (owner->isVideo())
	{
		sqlite3_stmt *vm = owner->owningMediaDatabase()->stmtInsertMediaVideo;

		sqlite3_bind_int(vm, 1, owner->mediaid_);
		sqlite3_bind_text(vm, 2, codec.utf8(), -1, SQLITE_TRANSIENT);
		sqlite3_bind_int(vm, 3, bitrate);
		sqlite3_bind_int(vm, 4, width);
		sqlite3_bind_int(vm, 5, height);
		sqlite3_bind_double(vm, 6, FPS);
		sqlite3_bind_double(vm, 7, aspectRatio);

		sqlite3_step(vm);
		DHOP(owner->owningMediaDatabase()->dbDebug("stmtInsertMediaVideo"));
		sqlite3_reset(vm);

		return true;
	}
	else
		return false;
}

bool MediaVideo::load()
{
	DHPRINTF("MediaVideo::load() media_id = %d", owner->mediaid_);

	if (owner->mediaid_ == 0)
		return false;

	// load video info...
	if (owner->isVideo())
	{
		sqlite3_stmt *vm = owner->owningMediaDatabase()->stmtSelectMediaVideoByMediaID;
		sqlite3_bind_int(vm, 1, owner->mediaid_);
		if (sqlite3_step(vm) == SQLITE_ROW)
		{
			codec = QString::fromUtf8((const char *)sqlite3_column_text(vm, 1));
			bitrate = sqlite3_column_int(vm, 2);
			width = sqlite3_column_int(vm, 3);
			height = sqlite3_column_int(vm, 4);
			FPS = sqlite3_column_double(vm, 5);
			aspectRatio = sqlite3_column_double(vm, 6);

			sqlite3_reset(vm);

			return true;
		}
		else
		{
			DHOP(owner->owningMediaDatabase()->dbDebug());
			sqlite3_reset(vm);
			return false;
		}
	}
	else
		return false;
}

/* class MediaMetadataExt */

MediaMetadataExt::MediaMetadataExt(Media *parent)
	: owner(parent)
{
}

bool MediaMetadataExt::save()
{
	//DPRINTF("MediaMetadataExt::save() media_id = %d", owner->mediaid_);
	sqlite3_stmt *vm;

	// delete all metadata tags for mediaid
	vm = owner->owningMediaDatabase()->stmtDeleteMediaMetadataExtByMediaID;
	sqlite3_bind_int(vm, 1, owner->mediaid_);
	sqlite3_step(vm);
	DHOP(owner->owningMediaDatabase()->dbDebug("stmtInsertMediaMetaDataExt"));
	sqlite3_reset(vm);

	// insert new metadata tags
	vm = owner->owningMediaDatabase()->stmtInsertMediaMetadataExt;
	for (MetadataMap::ConstIterator it = map.begin(); it != map.end(); ++it)
	{
		for (MetadataEntry::ConstIterator it2 = it.data().begin(); it2 != it.data().end(); ++it2)
		{
			sqlite3_bind_int(vm, 1, owner->mediaid_);
			sqlite3_bind_text(vm, 2, it.key(), -1, SQLITE_TRANSIENT);
			sqlite3_bind_text(vm, 3, (*it2).utf8(), -1, SQLITE_TRANSIENT);
			sqlite3_step(vm);
			DHOP(owner->owningMediaDatabase()->dbDebug("stmtInsertMediaMetaDataExt"));
			sqlite3_reset(vm);
		}
	}
	return true;
}

bool MediaMetadataExt::load()
{
	DHPRINTF("MediaMetadataExt::load() media_id = %d", owner->mediaid_);

	if (owner->mediaid_ == 0)
		return false;

	sqlite3_stmt *vm = owner->owningMediaDatabase()->stmtSelectMediaMetadataExtByMediaID;
	sqlite3_bind_int(vm, 1, owner->mediaid_);
	while (sqlite3_step(vm) == SQLITE_ROW)
	{
		QCString tag((const char *)sqlite3_column_text(vm, 1));
		QString value = QString::fromUtf8((const char *)sqlite3_column_text(vm, 2));

		//DPRINTF("%s = %s", (const char *)tag, value.latin1());

		MetadataMap::ConstIterator it = map.find(tag);
		if (it == map.end())
			map.insert(tag, QValueList<QString>());

		map[tag].append(value);
	}

	sqlite3_reset(vm);

	return true;
}


/* Media */

Media::Media(MediaDatabase *parent)
	: parent_(parent),
	  mediaid_(0),
	  mediaType_(MEDIATYPE_UNKNOWN),
	  locationHash_(0),
	  location_(),
	  fileSize_(0),
	  lastModified_(),
	  trackNo_(""),
	  title_(""),
	  artist_(""),
	  album_(""),
	  playTime_(""),
	  genre_(""),
	  length_(0),
	  year_(0),
	  compilation_(0),
	  lastError_("")
{
	listeners_.setAutoDelete(false);
}

Media::~Media()
{
	MediaListener *listener;
	for (listener = listeners_.first(); listener != 0; listener = listeners_.next())
		listener->mediaDestroyed(this);
}

void Media::initializeFromLocation(const MediaLocation &location)
{
	DENTERMETHOD("Media::initializeFromLocation(%s)", (const char*)(location.toString().utf8()));

	location_ = location;
	locationHash_ = elfHashString(location.toString());

	identify(location_);

	if (location_.isLocalFile() && QFile::exists(location_.toString()))
	{
		QFileInfo fileInfo = QFileInfo(location_.toString());
		lastModified_ = fileInfo.lastModified();
		fileSize_ = fileInfo.size();
		updateDisplayInformation(location_);
		DHOP(dumpInfo());
	}

	DEXITMETHOD("Media::initializeFromLocation(%s)", (const char*)(location.toString().utf8()));
}

bool Media::loadByID(int mediaid)
{
	DHENTERMETHOD("Media::loadByID(%d)", mediaid);

	sqlite3_stmt *vm = owningMediaDatabase()->stmtSelectMediaIDByMediaID;
	sqlite3_bind_int(vm, 1, mediaid);
	DHOP(owningMediaDatabase()->dbDebug("stmtSelectMediaIDByMediaID"));

	bool retval = loadFromDBVM(vm);

	DHEXITMETHOD("Media::loadByID(%d)", mediaid);
	return retval;
}

bool Media::loadFromDBVM(sqlite3_stmt *vm)
{
	int retval = false;

	if (sqlite3_step(vm) == SQLITE_ROW)
	{
		DHPRINTF("loading from database...");

		mediaid_ = sqlite3_column_int(vm, 0);
		DPRINTF("mediaid = %d", (int)mediaid_);

		loadMedia();

		retval = true;
	}
	sqlite3_reset(vm);

	return retval;
}

bool Media::loadMedia()
{
	DPRINTF("Media::loadMedia() media_id = %d", mediaid_);

	if (mediaid_ == 0)
		return false;

	bool retval = false;
	sqlite3_stmt *vm;

	// load meta data...
	vm = owningMediaDatabase()->stmtSelectMediaMetadataAndMediaFileInfoByMediaID;
	// SELECT hash, location, filename, type, filesize, modified_date, playtime, track, title, album, artist, genre, year, compilation FROM media LEFT JOIN media_location ON media.location_id = media_location.location_id WHERE media_id = ?1;

	sqlite3_bind_int(vm, 1, mediaid_);
	DOP(owningMediaDatabase()->dbDebug("sqlite3_bind_int(vm, 1, owner->mediaid_);"));

	int errNum = sqlite3_step(vm);
	DPRINTF("sqlite3_step errNum: %d", errNum);
	if (errNum == SQLITE_ROW)
	{
		locationHash_ = sqlite3_column_int(vm, 0);

		// TODO: resolve location by locationID and fill location with constructed data...
		//int locationID = sqlite3_column_int(vm, 1);
		QString filename = QString::fromUtf8((const char *)sqlite3_column_text(vm, 2));
		location_ = MediaLocation(QString::fromUtf8((const char *)sqlite3_column_text(vm, 1)) + filename);

		mediaType_ = (MediaType)sqlite3_column_int(vm, 3);
		fileSize_ = sqlite3_column_int(vm, 4);
		DHPRINTF("fileSize = %d", (int)fileSize);

		QString lastMod = QString::fromUtf8((const char *)sqlite3_column_text(vm, 5));
		DHPRINTF("lastMod = %s", (const char *)lastMod.latin1());

		dbParseTimestamp(lastMod, lastModified_);
		DPRINTF("parsed lastMod -> _lastModified = %s", (const char *)lastModified_.toString().latin1());

		length_  = sqlite3_column_int(vm, 6);
		trackNo_ = QString::fromUtf8((const char *)sqlite3_column_text(vm, 7));
		title_   = QString::fromUtf8((const char *)sqlite3_column_text(vm, 8));

		if (title_.isEmpty())
			title_ = filename;

		album_   = QString::fromUtf8((const char *)sqlite3_column_text(vm, 9));
		artist_  = QString::fromUtf8((const char *)sqlite3_column_text(vm, 10));
		genre_   = QString::fromUtf8((const char *)sqlite3_column_text(vm, 11));
		year_    = sqlite3_column_int(vm, 12);
		compilation_ = sqlite3_column_int(vm, 13);
		updatePlaytime();

		retval = true;
	}
	else
	{
		DOP(owningMediaDatabase()->dbDebug("Media::loadMedia()"));
	}
	sqlite3_reset(vm);

	return retval;
}

MediaAudio * Media::mediaAudio(bool autoNew) const
{
	MediaDatabase *mediaDatabase = owningMediaDatabase();
	Media *self = const_cast<Media*>(this);
	MediaAudio *mediaaudio = mediaDatabase->cacheMediaAudio_->find((int)self);

	if (mediaaudio)
	{
		DHPRINTF("Media::mediaAudio() existing : %d", (int)mediaaudio);
		return mediaaudio;
	}
	else if (autoNew)
	{
		mediaaudio = new MediaAudio(self);
		if (!mediaDatabase->cacheMediaAudio_->insert((int)self, mediaaudio))
		{
			// hell will most likely break loose, if we enter here...
			// should not happen.
			delete mediaaudio;
			mediaaudio = NULL;
		}
		else
			mediaaudio->load();

		DHPRINTF("Media::mediaAudio() new : %d", (int)mediaaudio);
		return mediaaudio;
	}
	else
		return NULL;
}

MediaVideo * Media::mediaVideo(bool autoNew) const
{
	MediaDatabase *mediaDatabase = owningMediaDatabase();
	Media *self = const_cast<Media*>(this);
	MediaVideo *mediavideo = mediaDatabase->cacheMediaVideo_->find((int)self);

	if (mediavideo)
		return mediavideo;
	else if (autoNew)
	{
		mediavideo = new MediaVideo(self);
		if (!mediaDatabase->cacheMediaVideo_->insert((int)self, mediavideo))
		{
			// hell will most likely break loose, if we enter here...
			// should not happen.
			delete mediavideo;
			mediavideo = NULL;
		}
		else
			mediavideo->load();

		return mediavideo;
	}
	else
		return NULL;
}

MediaMetadataExt * Media::mediaMetadataExt(bool autoNew) const
{
	MediaDatabase *mediaDatabase = owningMediaDatabase();
	Media *self = const_cast<Media*>(this);
	MediaMetadataExt *mediametadata = mediaDatabase->cacheMediaMetadataExt_->find((int)self);

	if (mediametadata)
	{
		DHPRINTF("Media::mediaMetadataExt() existing : %d", (int)mediametadata);
		return mediametadata;
	}
	else if (autoNew)
	{
		mediametadata = new MediaMetadataExt(self);
		if (!mediaDatabase->cacheMediaMetadataExt_->insert((int)self, mediametadata))
		{
			// hell will most likely break loose, if we enter here...
			// should not happen.
			delete mediametadata;
			mediametadata = NULL;
		}
		else
			mediametadata->load();

		DHPRINTF("Media::mediaMetadataExt() new : %d", (int)mediametadata);
		return mediametadata;
	}
	else
		return NULL;
}

void Media::saveToDB()
{
	DHENTERMETHOD("Media::saveToDB()");

	sqlite3_stmt *vm;
	MediaDatabase *mediaDatabase = owningMediaDatabase();

	DHPRINTF("Location: %s", location().toString().latin1());

	if (mediaid_)
 	{
		DHPRINTF("updating existing media by mediaid: %d", mediaid_);
		vm = mediaDatabase->stmtDeleteMediaByMediaID;
		sqlite3_bind_int(vm, 1, mediaid_);
		sqlite3_step(vm);
		sqlite3_reset(vm);

		DHOP(mediaDatabase->dbDebug("stmtDeleteMediaByMediaID"));

		vm = mediaDatabase->stmtInsertMedia;
		sqlite3_reset(vm);
		sqlite3_bind_int(vm, 1, mediaid_);
	}
	else
	{
		vm = mediaDatabase->stmtInsertMedia;
		sqlite3_reset(vm);

		DHPRINTF("inserting new media...");
		sqlite3_bind_null(vm, 1);
	}

	sqlite3_step(vm);
	DHOP(mediaDatabase->dbDebug("stmtInsertMedia"));
	sqlite3_reset(vm);

	if (!mediaid_)
		mediaid_ = mediaDatabase->lastInsertID();

	DHPRINTF("new media ID is %d", mediaid_);
	DHOP(dumpInfo());

	// Save metadata...
	saveMediaMetadata();

	// Save file info
	saveMediaFileInfo();

	if (mediaType_ != MEDIATYPE_UNKNOWN)
	{
		MediaAudio *mediaaudio = mediaAudio(false);
		if (mediaaudio)
			mediaaudio->save();

		MediaVideo *mediavideo = mediaVideo(false);
		if (mediavideo)
			mediavideo->save();

		MediaMetadataExt *mediametadata = mediaMetadataExt(false);
		if (mediametadata)
			mediametadata->save();
	}

	DHEXITMETHOD("Media::saveToDB()");
}

bool Media::saveMediaFileInfo()
{
	sqlite3_stmt *vm;

	// insert meta data for the media
	vm = owningMediaDatabase()->stmtInsertMediaFileInfo;
	// UPDATE media SET hash = ?2, location_id = ?3, filename = ?4, type = ?5, filesize = ?6, modified_date = ?7 WHERE media_id = ?1;

	sqlite3_bind_int(vm, 1, mediaid_);

	locationHash_ = elfHashString(location_.toString());
	sqlite3_bind_int(vm, 2, locationHash_);

	// splitting the location string manually into path and filename
	// since QUrl::fileName() and QUrl::dirPath() behave in an unwanted manner
	// when the location string contains a question mark or ampersand.
	QString path = location_.toString();
	int splitPos = path.findRev('/');
	QString filename = path.mid(splitPos + 1);
	path = path.left(splitPos + 1);

	int locationID = 0;

	// try to resolve existing entry for the current location...
	sqlite3_stmt *vm2 = owningMediaDatabase()->stmtSelectLocationIDByLocation;
	sqlite3_bind_text(vm2, 1, path.utf8(), -1, SQLITE_TRANSIENT);
	if (sqlite3_step(vm2) == SQLITE_ROW)
	{
		// if found, use the ID...
		locationID = sqlite3_column_int(vm2, 0);
	}
	else
	{
		// ... otherwise add new item and use the new assigned ID.
		sqlite3_stmt *vm3 = owningMediaDatabase()->stmtInsertMediaLocation;
		sqlite3_bind_null(vm3, 1);
		sqlite3_bind_text(vm3, 2, path.utf8(), -1, SQLITE_TRANSIENT);
		sqlite3_step(vm3);
		locationID = sqlite3_last_insert_rowid(owningMediaDatabase()->db());
		sqlite3_reset(vm3);
	}
	sqlite3_reset(vm2);

	sqlite3_bind_int(vm, 3, locationID);

	sqlite3_bind_text(vm, 4, filename.utf8(), -1, SQLITE_TRANSIENT);

	sqlite3_bind_int(vm, 5, mediaType_);
	sqlite3_bind_int(vm, 6, fileSize_);
	QString lastMod = dbFormatDate(lastModified_);
	sqlite3_bind_text(vm, 7, lastMod.utf8(), -1, SQLITE_TRANSIENT);

	sqlite3_step(vm);
	DHOP(owningMediaDatabase()->dbDebug("stmtInsertMediaFileInfo"));
	sqlite3_reset(vm);

	return true;
}

bool Media::saveMediaMetadata()
{
	sqlite3_stmt *vm;

	// insert meta data for the media
	vm = owningMediaDatabase()->stmtInsertMediaMetaData;
	// UPDATE media SET playtime = ?2, track = ?3, title = ?4, album = ?5, artist = ?6, genre = ?7, year = ?8, compilation = ?9 WHERE media_id = ?1;

	sqlite3_bind_int(vm, 1, mediaid_);
	sqlite3_bind_int(vm, 2, length_);
	sqlite3_bind_text(vm, 3, (trackNo_ == EMPTY_TRACKNO ? QString("") : trackNo_).utf8(), -1, SQLITE_TRANSIENT);
	sqlite3_bind_text(vm, 4, title_.utf8(), -1, SQLITE_TRANSIENT);
	sqlite3_bind_text(vm, 5, album_.utf8(), -1, SQLITE_TRANSIENT);
	sqlite3_bind_text(vm, 6, artist_.utf8(), -1, SQLITE_TRANSIENT);
	sqlite3_bind_text(vm, 7, genre_.utf8(), -1, SQLITE_TRANSIENT);
	sqlite3_bind_int(vm, 8, year_);
	sqlite3_bind_int(vm, 9, compilation_);
	sqlite3_step(vm);
	DHOP(owningMediaDatabase()->dbDebug("stmtInsertMediaMetaData"));
	sqlite3_reset(vm);

	return true;
}

void Media::identify(const MediaLocation &location)
{
	MediaIdentifier::singleton().identify(this, location);
}

void Media::updateDisplayInformation(const MediaLocation &location)
{
	QString tracknumber = metadata("TRACK");
	if (tracknumber.isNull() || tracknumber.isEmpty() || tracknumber == "0")
		trackNo_ = EMPTY_TRACKNO;
	else
		trackNo_ = tracknumber;

	QString title = metadata("TITLE");
	if (title.isNull() || title.isEmpty())
		if (location.isLocalFile())
			title_ = location.fileName();
		else
			title_ = location.toString();
	else
		title_ = title;

	artist_ = metadata("ARTIST");
	album_ = metadata("ALBUM");
	genre_ = metadata("GENRE");

	QString year = metadata("DATE");
	if (!year.isNull() && !year.isEmpty())
		year_ = year.toInt();

	// TODO: compilation info ZERO for now...
	compilation_ = 0;
	updatePlaytime();
}

void Media::updatePlaytime()
{
	playTime_ = QString().sprintf("%3d:%02d", length_ / 60, length_ % 60);
}

QString Media::metadata(const QCString& key, int pos) const
{
	MediaMetadataExt * mediametadata = mediaMetadataExt();

	MetadataMap::ConstIterator it = mediametadata->map.find(key);

	if (it == mediametadata->map.end())
		return QString::null;

	if (pos < 0 || pos >= it.data().count())
		return it.data().last();

	return *(it.data().at(pos));
}

QStringList Media::metadataStringList() const
{
	QStringList list;
	MediaMetadataExt * mediametadata = mediaMetadataExt();

	for (MetadataMap::ConstIterator it = mediametadata->map.begin(); it != mediametadata->map.end(); ++it)
		for (MetadataEntry::ConstIterator it2 = it.data().begin(); it2 != it.data().end(); ++it2)
			list << QString("%1 = %2").arg(it.key()).arg((*it2).latin1());

	return list;
}

void Media::dumpInfo() const
{
	DENTERMETHOD("Media::dumpInfo()");

	DPRINTF("--------------------------------");
	DPRINTF("Media ID: <%d>", mediaid_);
	DPRINTF("Location: <%s>", location().toString().latin1());
	DPRINTF("Last-Modified: <%s>", lastModified().toString().latin1());
	DPRINTF("Filesize: %d", fileSize());

	DPRINTF("Length: %d", length());

	DPRINTF("Type: %s", (isAudio() ? "audio" : (isVideo() ? "video" : "no idea...")));

	DPRINTF("Track: %s", trackNo().latin1());
	DPRINTF("Artist: %s", artist().latin1());
	DPRINTF("Title: %s", title().latin1());
	DPRINTF("Album: %s", album().latin1());
	DPRINTF("Playtime: %s", playTime().latin1());

	DPRINTF("--------------------------------");

	if (isVideo())
		DPRINTF("Video: Codec=<%s> Bitrate=%d Size=%dx%d Fps=%f Aspect=%f ",
			   mediaVideo()->codec.latin1(), mediaVideo()->bitrate, mediaVideo()->width, mediaVideo()->height, mediaVideo()->FPS, mediaVideo()->aspectRatio);

	if (isVideo() || isAudio())
		DPRINTF("Audio: Codec=<%s> Bitrate=%d Nch=%d",
			   mediaAudio()->codec.latin1(), mediaAudio()->bitrate, mediaAudio()->channels);

	DPRINTF("Metadata:");
	for (MetadataMap::ConstIterator it = mediaMetadataExt()->map.begin(); it != mediaMetadataExt()->map.end(); ++it) {
		for (MetadataEntry::ConstIterator it2 = it.data().begin(); it2 != it.data().end(); ++it2)
			DPRINTF("  %s=%s", static_cast<const char *>(it.key()),
					static_cast<const char *>((*it2).local8Bit()));
	}
	DPRINTF("--------------------------------");

	DEXITMETHOD("Media::dumpInfo()");
}
