/*
 * Copyright (C) 2006-2007	Andre Beckedorf
 * 					 		<evilJazz _AT_ katastrophos _DOT_ net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <cstdio>
#include <cstdlib>
#include <unistd.h>

#include <qstring.h>
#include <qdatastream.h>
#include <qtextcodec.h>
#include <qregexp.h>

#ifdef WINDOWS
#include "wincompat.h"
#endif

#include "mediaidentifier.h"
#include "mediadatabase.h"
#include "mplayer.h"
#include "configuration.h"
#include "debug.h"

#include <qbuffer.h>

#include <fileref.h>
#include <tstring.h>
#include <tmap.h>
#include <tstringlist.h>
#include <tbytevector.h>
#include <tag.h>
#include <mpegfile.h>
#include <attachedpictureframe.h>
#include <oggfile.h>
#include <vorbisfile.h>
#include <flacfile.h>
#include <oggflacfile.h>

#include <mp4file.h>
#include <mp4itunestag.h>

#include <kmdcodec.h>

bool containsUTF8(const QString &input)
{
	if (!input.isNull())
	{
		const QChar *c = input.unicode();

		// The following code was blatantly stolen from MythTV's util.cpp and adapted for use with QString/QChar.
		// It's covered by GPLv2.
		while (*c++)
		{
			ushort uc = c->unicode();

			// ASCII is < 0x80.
			// 0xC2..0xF4 is probably UTF-8.
			// Anything else probably ISO-8859-1 (Latin-1, Unicode)
			if (uc > 0xC1 && uc < 0xF5)
			{
				int bytesToCheck = 2; 			// Assume  0xC2-0xDF (2 byte sequence)

				if (uc > 0xDF) 					// Maybe   0xE0-0xEF (3 byte sequence)
					++bytesToCheck;

				if (uc > 0xEF) 					// Matches 0xF0-0xF4 (4 byte sequence)
					++bytesToCheck;

				while (bytesToCheck--)
				{
					++c;

					if (c->isNull()) 			// String ended in middle
						return false; 			// Not valid UTF-8

					uc = c->unicode();

					if (uc < 0x80 || uc > 0xBF) // Bad UTF-8 sequence
						break; 					// Keep checking in outer loop
				}

				if (!bytesToCheck) 				// Have checked all the bytes in the sequence
					return true; 				// Hooray! We found valid UTF-8!
			}
		}
	}

	return false;
}

QString reinterpretUTF8(const QString &input)
{
	return QString::fromUtf8(input.latin1(), input.length());
}


MediaIdentifier & MediaIdentifier::singleton()
{
	static MediaIdentifier instance;
	return instance;
}


MediaIdentifier::MediaIdentifier()
	: media_(NULL),
	  fileScanningFinished_(false),
	  identifyProcess_(NULL),
	  tempLine_("")
{
	connect(&killIdentifyProcessTimer_, SIGNAL(timeout()), this, SLOT(killIdentifyProcessTimerTimedOut()));
}

MediaIdentifier::~MediaIdentifier()
{
	DPRINTF("Killing mplayer slave because we're quitting...");
	cleanUpIdentifyProcess();
}

void MediaIdentifier::cleanUpIdentifyProcess()
{
	if (identifyProcess_)
	{
		identifyProcess_->tryTerminate();
		usleep(1000000); // wait 1 sec
		if (identifyProcess_->isRunning())
		{
			identifyProcess_->kill();
			usleep(500000); // wait 500 ms
		}

		delete identifyProcess_;
		identifyProcess_ = NULL;
	}
}

void MediaIdentifier::killIdentifyProcessTimerTimedOut()
{
	killIdentifyProcessTimer_.stop();
	DPRINTF("Killing mplayer slave since we no longer need it...");
	cleanUpIdentifyProcess();
}

void MediaIdentifier::sendCommand(const QString &command)
{
	identifyProcess_->writeToStdin(command + "\n");
	identifyProcess_->flushStdin();
}

void MediaIdentifier::identify(Media *media, const MediaLocation &location)
{
	bool tagsRead = false;
	bool audioPropertiesRead = false;

	QString extension = QFileInfo(location.toString()).extension(false);

	media_ = media;
	fileScanningFinished_ = false;

	if (location.isLocalFile() && QFile::exists(location.toString()))
		loadTags(location, tagsRead, audioPropertiesRead);

	if (!tagsRead || !audioPropertiesRead || qConfig.isAmbiguousFormatExtension(extension))
		identifyUsingMPlayer(media, location);

	media_ = NULL;
}

void MediaIdentifier::identifyUsingMPlayer(Media *media, const MediaLocation &location)
{
	// kill mplayer slave after 10 seconds, if no other identification request
	// happens within that period...
	killIdentifyProcessTimer_.changeInterval(10000);

	// perhaps we are retrying after a timeout?
	// when this happens, media_ is still set, so use this information to
	// recognize this state...
	bool mediaWasSet = media_;

	media_ = media;
	fileScanningFinished_ = false;

	if (!identifyProcess_ || !identifyProcess_->isRunning())
	{
		QStringList args;
		args << qConfig.mplayerBinLocation() << "-vo" << "null" << "-ao" << "null" << "-v" << "-nocache" << "-slave" << "-identify";
		args << location.toString();

		identifyProcess_ = new QProcess(args);

		DPRINTF("Starting mplayer slave...");
		identifyProcess_->start();

		DPRINTF("Sending pause...");
		sendCommand("pause");
	}
	else
	{
		DPRINTF("Mplayer slave still running, inputing new file...");
		sendCommand("pausing loadfile \"" + location.toString() + "\" 0");
	}

	DPRINTF("Waiting for file scanning to finish...");

	// timeout timer...
	QTime time;
	time.start();
	bool timedOut = false;

	while ((identifyProcess_->isRunning() || identifyProcess_->canReadLineStdout()) && !fileScanningFinished_ && !timedOut)
	{
		timedOut = time.elapsed() > 5000;
		parseLines();
		usleep(1000); // sleep 1 ms
	}

	if (timedOut)
	{
		DPRINTF("Timeout while scanning file...");
		cleanUpIdentifyProcess();

		if (!mediaWasSet)
		{
			DPRINTF("Retrying again...");
			identifyUsingMPlayer(media, location);
		}

		return;
	}

	media_ = NULL;
}

void MediaIdentifier::parseLines()
{
	if (!media_)
		return;

	while (identifyProcess_->canReadLineStdout())
	{
		QString line = identifyProcess_->readLineStdout();

		DPRINTF("line: %s", (const char *)line.utf8());

		if (line.startsWith("ID_"))
		{
			// ID_*
			line = line.mid(3);
			if (line.startsWith("LENGTH="))
			{
				line = line.mid(7);
				int posDot = line.find(".");
				media_->length_ = (posDot > -1 ? line.left(posDot).toInt() : line.toInt());
			}
			else if (line.startsWith("AUDIO_"))
			{
				// ID_AUDIO_*
				line = line.mid(6);

				// avoid stepping down from video to audio...
				if (media_->mediaType_ == Media::MEDIATYPE_UNKNOWN)
					media_->mediaType_ = Media::MEDIATYPE_AUDIO;

				if (line.startsWith("CODEC="))
					media_->mediaAudio()->codec = line.mid(6);
				else if (line.startsWith("BITRATE="))
					media_->mediaAudio()->bitrate = line.mid(8).toInt();
				else if (line.startsWith("RATE="))
					media_->mediaAudio()->sampleRate = line.mid(5).toInt();
				else if (line.startsWith("NCH="))
					media_->mediaAudio()->channels = line.mid(4).toInt();
			}
			else if (line.startsWith("VIDEO_"))
			{
				// ID_VIDEO_*
				line = line.mid(6);

				media_->mediaType_ = Media::MEDIATYPE_VIDEO;

				if (line.startsWith("FORMAT="))
					media_->mediaVideo()->codec = line.mid(7);
				else if (line.startsWith("BITRATE="))
					media_->mediaVideo()->bitrate = line.mid(8).toInt();
				else if (line.startsWith("WIDTH="))
					media_->mediaVideo()->width = line.mid(6).toInt();
				else if (line.startsWith("HEIGHT="))
					media_->mediaVideo()->height = line.mid(7).toInt();
				else if (line.startsWith("FPS="))
					media_->mediaVideo()->FPS = line.mid(4).toFloat();
				else if (line.startsWith("ASPECT="))
					media_->mediaVideo()->aspectRatio = line.mid(7).toFloat();
			}
			else if (line.startsWith("CLIP_INFO_NAME"))
			{
				tempLine_ = "";

				int posEq = line.find('=');
				if (posEq > -1)
					tempLine_ = line.mid(posEq + 1);
			}
			else if (line.startsWith("CLIP_INFO_VALUE") && tempLine_.length() > 0)
			{
				int posEq = line.find('=');
				if (posEq > -1)
				{
					line = line.mid(posEq + 1);

					if (containsUTF8(line))
						line = reinterpretUTF8(line);

					addMediadata(tempLine_, line);
				}
			}
		}
		else if (line.startsWith("OggVorbisComment: "))
		{
			media_->mediaType_ = Media::MEDIATYPE_AUDIO;
			line = line.mid(18);
			int sepIndex = line.find('=');
			if (sepIndex > 0)
			{
				QString tag(line.left(sepIndex));
				QString value(line.mid(sepIndex + 1));

				if (containsUTF8(value))
					value = reinterpretUTF8(value);

				addMediadata(tag, value);
			}
		}
		else if (line.startsWith("SLAVE: Couldn't open demuxer"))
		{
			media_->mediaType_ = Media::MEDIATYPE_UNKNOWN;
		}
		else if (line.startsWith("Starting playback"))
		{
			//qDebug("Got finish signal...");
			fileScanningFinished_ = true;
		}
	}
}

void MediaIdentifier::addMediadata(const QString &tag, const QString &value)
{
	MediaMetadataExt * mediametadata = media_->mediaMetadataExt();

	// Resolve common aliases...
	QCString key = QCString(tag.upper());
	if (key == "NAME")
		key = "TITLE";
	else if (key == "TRACKNUMBER")
		key = "TRACK";

	// Check if the key exists. If not, add it.
	MetadataMap::Iterator it = mediametadata->map.find(key);
	if (it == mediametadata->map.end())
		it = mediametadata->map.insert(key, MetadataEntry());

	// Try to avoid duplicates...
	if (!it.data().contains(value))
		it.data().append(value);
}

bool MediaIdentifier::loadTags(const MediaLocation &location, bool &tagsWereReadable, bool &audioPropertiesWereReadable)
{
	TagLib::FileRef f(QFile::encodeName(location.toString()));

	tagsWereReadable = false;
	audioPropertiesWereReadable = false;

	if (!f.isNull() && f.tag())
	{
		TagLib::Tag *tag = f.tag();

		if (!tag->title().isNull())	addMediadata("TITLE", TStringToQString(tag->title()));
		if (!tag->album().isNull())	addMediadata("ALBUM", TStringToQString(tag->album()));
		if (!tag->artist().isNull())	addMediadata("ARTIST", TStringToQString(tag->artist()));
		if (!tag->genre().isNull())	addMediadata("GENRE", TStringToQString(tag->genre()));
		if (!tag->comment().isNull())	addMediadata("COMMENT", TStringToQString(tag->comment()));

		addMediadata("TRACK", QString::number(tag->track()));

		tagsWereReadable = true;
	}

	if (!f.isNull() && f.audioProperties())
	{
		TagLib::AudioProperties *properties = f.audioProperties();

		media_->mediaType_ = Media::MEDIATYPE_AUDIO;
		media_->length_ = properties->length();

		MediaAudio *audio = media_->mediaAudio();

		QString extension = QFileInfo(location.toString()).extension().lower();
		QString codec = "";

		// TODO: This lookup is by no means accurate. Improve!
		if (extension.startsWith("ogg"))
			codec = "Vorbis";
		else if (extension.startsWith("mp3"))
			codec = "MP3";
		else if (extension.startsWith("fla") || extension.startsWith("oga"))
			codec = "FLAC";
		else if (extension.startsWith("mpc"))
			codec = "Musepack";
		else if (extension.startsWith("aac"))
			codec = "AAC";
		else if (extension.startsWith("mp4") || extension.startsWith("m4"))
			codec = "MP4";
		else if (extension.startsWith("wma") || extension.startsWith("asf"))
			codec = "Windows Media Audio";
		else if (extension.startsWith("wv"))
			codec = "WavPack";
		else if (extension.startsWith("wav"))
			codec = "WAVE";

		audio->codec = codec;
		audio->bitrate = properties->bitrate() * 1000;
		audio->sampleRate = properties->sampleRate();
		audio->channels = properties->channels();

		audioPropertiesWereReadable = true;
	}

	return false;
}

void MediaIdentifier::dumpXiphComment(TagLib::Ogg::XiphComment *xiphComment)
{
	DENTERMETHOD("MediaIdentifier::dumpXiphComment()");

	TagLib::Ogg::FieldListMap::ConstIterator it = xiphComment->fieldListMap().begin();

	for (; it != xiphComment->fieldListMap().end(); ++it)
	{
		TagLib::String fieldName = (*it).first;
		TagLib::StringList values = (*it).second;

		TagLib::StringList::ConstIterator valuesIt = values.begin();

		for (; valuesIt != values.end(); ++valuesIt)
			qDebug("%s = %s", fieldName.toCString(), (*valuesIt).toCString());
	}

	DEXITMETHOD("MediaIdentifier::dumpXiphComment()");
}

bool MediaIdentifier::loadCoverArtFromXiphComment(TagLib::Ogg::XiphComment *xiphComment, QImage &coverArtImage)
{
	bool result = false;

	if (xiphComment->contains("COVERART"))
	{
		DPRINTF("xiphComment contains COVERART.");

		TagLib::ByteVector coverArtMD5Data = xiphComment->fieldListMap()["COVERART"].front().data(TagLib::String::Latin1);

		// create a shallow copy QByteArray wrapper...
		QByteArray coverArtMD5DataInput;
		coverArtMD5DataInput.setRawData(coverArtMD5Data.data(), coverArtMD5Data.size());

		QByteArray coverArtRawDataOutput;
		KCodecs::base64Decode(coverArtMD5DataInput, coverArtRawDataOutput);

		coverArtMD5DataInput.resetRawData(coverArtMD5Data.data(), coverArtMD5Data.size());

		result = coverArtImage.loadFromData(coverArtRawDataOutput);
		DPRINTF("%s", result ? "Read image successfully." : "Could not decode image.");
	}

	return result;
}

bool MediaIdentifier::loadCoverArtFromID3v2(TagLib::ID3v2::Tag *id3v2Tag, QImage &coverArtImage)
{
	bool result = false;

	TagLib::ID3v2::FrameList frameList = id3v2Tag->frameList("APIC");

	if (!frameList.isEmpty())
	{
		TagLib::ID3v2::AttachedPictureFrame *attachedPictureFrame = static_cast<TagLib::ID3v2::AttachedPictureFrame *>(frameList.front());

		if (attachedPictureFrame->picture().size() > 0)
		{
			DPRINTF("ID3v2 header contains APIC frame.");
			result = coverArtImage.loadFromData((const uchar *)attachedPictureFrame->picture().data(), attachedPictureFrame->picture().size());
			DPRINTF("%s", result ? "Read image successfully." : "Could not decode image.");
		}
	}

	return result;
}

bool MediaIdentifier::loadCoverArtFromFile(const QString &filename, QImage &coverArtImage)
{
	bool result = false;

	//if (!QFile(filename).exists())
	//	return false;

	QString extension = QFileInfo(filename).extension(false);

	// Check for a cover art image in ID3v2 APIC frame
	if (extension.find("mp3", 0, false) != -1)
	{
		TagLib::MPEG::File f(QFile::encodeName(filename));

		if (f.isValid() && f.ID3v2Tag())
			result = loadCoverArtFromID3v2(f.ID3v2Tag(), coverArtImage);
	}
	// Check for a cover art image in Ogg Container in XiphComment
	else if ((extension.find("ogg", 0, false) != -1) ||
			 (extension.find("oga", 0, false) != -1))
	{
		TagLib::Ogg::Vorbis::File f(QFile::encodeName(filename));

		result = f.isValid() && f.tag() && loadCoverArtFromXiphComment(f.tag(), coverArtImage);

		DHOP(dumpXiphComment(f.tag()));

		if (!result)
		{
			TagLib::Ogg::FLAC::File f(QFile::encodeName(filename));

			if (f.isValid() && f.tag())
				result = loadCoverArtFromXiphComment(f.tag(), coverArtImage);
		}
	}
	// Check for a cover art image either via ID3v2 APIC frame or in XiphComment
	else if (extension.find("flac", 0, false) != -1)
	{
		TagLib::FLAC::File f(QFile::encodeName(filename));

		if (f.isValid())
		{
			result = f.ID3v2Tag() && loadCoverArtFromID3v2(f.ID3v2Tag(), coverArtImage);

			if (!result)
				result = f.xiphComment() && loadCoverArtFromXiphComment(f.xiphComment(), coverArtImage);
		}
	}
	// Check for a M4A iTunes cover art image
	else if ((extension.find("m4a", 0, false) != -1) ||
			(extension.find("aac", 0, false) != -1) ||
			(extension.find("m4b", 0, false) != -1) ||
			(extension.find("m4p", 0, false) != -1) ||
			(extension.find("mp4", 0, false) != -1) ||
			(extension.find("m4v", 0, false) != -1) ||
			(extension.find("mp4v", 0, false) != -1))
	{
		TagLib::MP4::File f(QFile::encodeName(filename));

		if (f.isValid() && f.tag())
		{
			TagLib::MP4::Tag *tag = static_cast<TagLib::MP4::Tag *>(f.tag());

			if (!tag->cover().isNull() && !tag->cover().isEmpty())
			{
				DPRINTF("M4A file contains iTunes covr atom.");
				result = coverArtImage.loadFromData((const uchar *)tag->cover().data(), tag->cover().size());
				DPRINTF("%s", result ? "Read image successfully." : "Could not decode image.");
			}
		}
	}

	return result;
}
