/*
 * Copyright (C) 2007-2008 Andre Beckedorf <evilJazz _AT_ katastrophos _DOT_ net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <qpainter.h>
#include "reflectionimageflow.h"
#include "imagefx.h"
#include "debug.h"

QImage prepareImage(QImage image, int maxWidth, int maxHeight, int reflectionHeight)
{
	DTIMERINIT(test);

	if (image.isNull())
		return QImage();

	QSize outputSize = ImageFX::optimalFitSize(image.size(), QSize(maxWidth, maxHeight + reflectionHeight));
	QImage scaledImage;

	if (outputSize.width() > 0 && outputSize.height() > 0)
	{
		DTIMERINIT(test2);
		scaledImage = image.smoothScale(outputSize.height(), outputSize.width());
		DTIMERPRINT(test2, "smooth scale");

		DTIMERSTART(test2);
		scaledImage = ImageFX::createReflection(scaledImage, reflectionHeight, 120, 0);

		QImage blendImage(scaledImage.width(), scaledImage.height(), 32);
		blendImage.setAlphaBuffer(false);
		blendImage.fill(0x00000000);

		uint *srcBits;
		uint *dstBits;

		for (int y = 0; y < scaledImage.height(); ++y)
		{
			srcBits = (uint *)scaledImage.scanLine(y);
			dstBits = (uint *)blendImage.scanLine(y);

			int x = scaledImage.width();
			while (--x)
			{
				*dstBits = blendARGB(*srcBits, *dstBits);
				++srcBits;
				++dstBits;
			}
		}

		scaledImage = blendImage;

		DTIMERPRINT(test2, "create reflection");
	}

	DTIMERPRINT(test, "cover art preparation");

	return scaledImage;
}


/*  ReflectionImageProvider  */

ReflectionImageProvider::ReflectionImageProvider(QObject *owner, const char *name)
	:	CustomScaledCachedImageProvider(owner, name),
		preloadImageFilename_(QString::null),
		refSize_(300, 400),
		reflectionHeight_(100)
{
	invalidatePreloadImage();
}

QImage* ReflectionImageProvider::threadedCreateImage(CachedImage *cachedImage)
{
	DPRINTF("threadedCreateImage");
	//DTIMERINIT(loadtimer);
	DPRINTF("Loading image '%s'...", (const char *)(cachedImage->name().utf8()));
	QImage srcImage;

	QImage *dstImage = NULL;

	bool imageWasLoaded = srcImage.load(cachedImage->name());

	if (imageWasLoaded && !srcImage.isNull())
	{
		//DMEMSTAT();

		//DPRINTF("Scaling image '%s'...", (const char *)(cachedImage->name().utf8()));
		//QImage scaledImage = srcImage.smoothScale(300, 300);

		QImage scaledImage = prepareImage(srcImage, refSize_.width(), refSize_.height() - reflectionHeight_, reflectionHeight_);

		//DPRINTF("Assigning scaled image '%s'...", (const char *)(cachedImage->name().utf8()));
		dstImage = new QImage(scaledImage);

		//DPRINTF("Internal saving image '%s' to '%s'...", (const char *)(cachedImage->name().utf8()), (const char *)(cachedImage->cacheFileName().utf8()));
		//dstImage->save(cachedImage->cacheFileName(), "JPEG", 70);
	}
	else
		DPRINTF("Failed loading image '%s'", (const char *)(cachedImage->name().utf8()));

	//DPRINTF("Timer threadedCreateImage...");
	//DTIMERPRINT(loadtimer, "threadedCreateImage");
	return dstImage;
}

QImage* ReflectionImageProvider::preloadImage(const QString &name, void *data)
{
	DPRINTF("preloadImage");

	if (preloadImage_.isNull())
		updatePreloadImage();

	return &preloadImage_;
}

void ReflectionImageProvider::setRefSize(QSize size)
{
	refSize_ = size;
	invalidatePreloadImage();
}

void ReflectionImageProvider::setReflectionHeight(int height)
{
	reflectionHeight_ = height;
	invalidatePreloadImage();
}

void ReflectionImageProvider::setPreloadImageFilename(const QString &filename)
{
	preloadImageFilename_ = filename;
	invalidatePreloadImage();
}

void ReflectionImageProvider::invalidatePreloadImage()
{
	preloadImage_.reset();
}

void ReflectionImageProvider::updatePreloadImage()
{
	DENTERMETHOD("CoverImageProvider::updatePreloadImage()");

	bool imageWasLoaded = false;

	if (!preloadImageFilename_.isNull())
		imageWasLoaded = preloadImage_.load(preloadImageFilename_);

	if (imageWasLoaded && !preloadImage_.isNull())
	{
		preloadImage_ = prepareImage(preloadImage_, refSize_.width(), refSize_.height() - reflectionHeight_, reflectionHeight_);
	}
	else
	{
		preloadImage_.create(refSize_.width(), refSize_.height(), 32);
		preloadImage_.fill(0x00FFFFFF);
	}

	DEXITMETHOD("CoverImageProvider::updatePreloadImage()");
}


/*  ReflectionImageFlow  */

ReflectionImageFlow::ReflectionImageFlow(ReflectionImageProvider *imageProvider, LayeredPaintBox *parent)
	:	CustomPerspectiveImageFlow(parent),
		imageProvider_(imageProvider)
{
	connect(imageProvider_, SIGNAL(imageCached(CachedImage *)), this, SLOT(imageWasCached(CachedImage *)));
	connect(imageProvider_, SIGNAL(imageSaved(CachedImage *)), this, SLOT(imageSaved(CachedImage *)));
	connect(imageProvider_, SIGNAL(imageIsNull(CachedImage *)), this, SLOT(imageIsNull(CachedImage *)));
	connect(imageProvider_, SIGNAL(scaleChangedAndFlushed()), this, SLOT(scaleWasChangedAndCacheWasFlushed()));
	connect(&invalidateTimer_, SIGNAL(timeout()), this, SLOT(invalidateView()));
}

void ReflectionImageFlow::setFileNames(QStringList imageFileNames)
{
	imageFileNames_ = imageFileNames;
	setItemCount(imageFileNames_.count());
	invalidate();
}

void ReflectionImageFlow::setSlideSize(QSize size)
{
	CustomPerspectiveImageFlow::setSlideSize(size);
	imageProvider()->setRefSize(size);
}

QImage ReflectionImageFlow::getImage(int index)
{
	//DPRINTF("%d", imageFileNames_.count());

	if (index >= imageFileNames_.count())
		return NULL;

	//DPRINTF("getImage(%d): %s", index, (const char *)imageFileNames_[index].utf8());

	return *imageProvider()->getImage(imageFileNames_[index], NULL);
}

void ReflectionImageFlow::updateElementCountAndMidIndex()
{
	//qDebug("updateElementCountAndMidIndex()");
	CustomPerspectiveImageFlow::updateElementCountAndMidIndex();

	//qDebug("elementCount(): %d  -  memoryCacheSize(): %d", elementCount(), imageProvider_.memoryCacheSize());

	// prevent performance problem, when more items are visible
	// than can be cached by simply avoiding this situation...
	//if (elementCount() > imageProvider_.memoryCacheSize())

	// TODO: Refine with more intelligent algorithm...
	int overSize = 8 * elementCount();
	imageProvider()->setMemoryCacheSize(elementCount() + overSize);
}

void ReflectionImageFlow::resized()
{
	if (height() < 200)
		setAntialiasMode(CustomPerspectiveImageFlow::AntialiasAlways);
	else
		setAntialiasMode(CustomPerspectiveImageFlow::AntialiasStaticSceneOnly);

	CustomPerspectiveImageFlow::resized();

	imageProvider()->setScale(scale());
}

void ReflectionImageFlow::imageWasCached(CachedImage *cachedImage)
{
	triggerInvalidateTimer();
}

void ReflectionImageFlow::imageSaved(CachedImage *cachedImage)
{
	triggerInvalidateTimer();
}

void ReflectionImageFlow::imageIsNull(CachedImage *cachedImage)
{
	triggerInvalidateTimer();
}

void ReflectionImageFlow::scaleWasChangedAndCacheWasFlushed()
{
	triggerInvalidateTimer();
}

void ReflectionImageFlow::triggerInvalidateTimer()
{
	if (!invalidateTimer_.isActive())
		invalidateTimer_.start(300, true);
}

void ReflectionImageFlow::invalidateView()
{
	invalidate();
}
