#ifndef COVERARTDOWNLOADER_H_
#define COVERARTDOWNLOADER_H_

#include <qwidget.h>
#include <qdialog.h>
#include <qhttp.h>
#include <qlist.h>
#include <qintcache.h>
#include <qlistview.h>
#include <qprogressbar.h>
#include <qmap.h>
#include <qlabel.h>
#include <qhttp.h>
#include <qhgroupbox.h>

#include "addonmanager.h"
#include "configurationdialog.h"
#include "coverartconfig.h"
#include <qlineedit.h>
#include <qvbox.h>

#include "paintbox.h"
#include "scrollbar.h"
#include "customperspectiveimageflow.h"

#include "quasar.h"
#include "configuration.h"
#include "helpers.h"
#include "playlist.h"
#include "skin.h"
#include "skinmanager.h"

#define COVERART_DEFAULT_URL "http://coverart.katastrophos.net/query.php"

class QSplitter;
class AlbumListViewItem;

class CoverArtSearchResult
{
public:
	QString title;
	QString detailsURL;
	QString thumbnailImageURL;
	QString imageURL;

	AlbumListViewItem *owner;
};

class AlbumListViewItem : public QCheckListItem
{
public:
	AlbumListViewItem(QListView *parent) : QCheckListItem(parent, "", CheckBox), selectedResultIndex(0) { coverArtSearchResults.setAutoDelete(true); }

	QString artist() const { return text(0); }
	QString album() const { return text(1); }

	QList<CoverArtSearchResult> coverArtSearchResults;
	int selectedResultIndex;
};

class HttpGrabber : public QObject
{
	Q_OBJECT
public:
	HttpGrabber(QObject *parent = 0, const char *name = 0);
	virtual ~HttpGrabber();

	void setTimeout(uint msec) { timeout_ = msec; }
	uint timeout() { return timeout_; }
	bool download(const QString &url, void *userData = NULL);
	bool isBusy() { return busy_; };
	void abort();

signals:
	void finished(HttpGrabber *, const QString &url, void *userData, bool error, bool timedOut);
	void dataAvailable(HttpGrabber *, const QString &url, void *userData, QByteArray *data);

private slots:
	void timerTimeout();
	void httpRequestFinished(int res, bool error);

private:
	QTimer timer_;
	QHttp http_;
	QString url_;
	void *userData_;
	uint timeout_;
	bool busy_;
	int downloadRes_;
	uint lastBytesAvailable_;
};

class HttpGrabberList : public QList<HttpGrabber>
{
public:
	void abortAll()
	{
		for (HttpGrabber *g = first(); g != NULL; g = next())
			g->abort();
	}

	int busyGrabbersCount()
	{
		int result = 0;
		for (HttpGrabber *g = first(); g != NULL; g = next())
			if (g->isBusy())
				++result;

		return result;
	}

	HttpGrabber *getIdleGrabber()
	{
		HttpGrabber *g = first();
		while (g != NULL && g->isBusy())
			g = next();
		return g;
	}
};

class DownloadedCoverArtImageFlow : public CustomPerspectiveImageFlow
{
	Q_OBJECT
public:
	DownloadedCoverArtImageFlow(LayeredPaintBox *parent);
	QList<QImage> &images() { return images_; }
	void invalidate() { CustomPerspectiveImageFlow::invalidate(); }
	void setAlbumListViewItem(AlbumListViewItem *albumListViewItem);

protected:
	QImage getImage(int index);
	void paintOverlay(QPainter &p, const QPixmap &pixmap);
	void transitionFinished();

private:
	QList<QImage> images_;
	AlbumListViewItem *albumListViewItem_;
};

class CoverArtDownloader : public QDialog
{
	Q_OBJECT
public:
	CoverArtDownloader(PlayList* playList, SkinManager* skinManager, QWidget *parent = 0, const char *name = 0, bool modal = FALSE);

signals:
	void closing();

protected slots:
	virtual void accept();
	virtual void reject();

protected:
	void closeEvent(QCloseEvent *);
	void done(int r);
	void showEvent(QShowEvent *);

private slots:
	void imageFlowViewBoxRearrangeLayers();

	void scanForMissingCoverArt();

	void listViewSelectionChanged(QListViewItem *);
	void loadFlowView(AlbumListViewItem *);

	void searchFinished(HttpGrabber *grabber, const QString &url, void *userData, bool error, bool timedOut);
	void searchDataAvailable(HttpGrabber *grabber, const QString &url, void *userData, QByteArray *data);

	void downloadFinished(HttpGrabber *grabber, const QString &url, void *userData, bool error, bool timedOut);
	void downloadDataAvailable(HttpGrabber *grabber, const QString &url, void *userData, QByteArray *data);

private:
	PlayList *playList_;
	SkinManager* skinManager_;
	QListView *listView_;
	QLabel *progressLabel_;
	QSplitter *splitter_;
	QProgressBar *progress_;
	LayeredPaintBox *imageFlowViewBox_;
	DownloadedCoverArtImageFlow *imageFlowView_;
	FlowScrollBar *flowScrollBar_;
	QPushButton *startDownloadButton_;
	QPushButton *cancelButton_;

	QList<AlbumListViewItem> albumListItems_;

	int currentAlbumIndex_;
	bool searchingOnline_;
	HttpGrabberList searchGrabbers_;

	AlbumListViewItem *currentAlbumListViewItem_;
	QList<CoverArtSearchResult> *coverArtSearchResultsToDownload_;
	QList<CoverArtSearchResult> downloadList_;
	int currentResultIndex_;
	bool downloadingImageFiles_;

	HttpGrabberList downloadGrabbers_;
	QIntCache<QByteArray> resultImageDataCache_;

	void handleCoverArtSearchResultImageData(CoverArtSearchResult *coverArtSearchResult, QByteArray *imageData);
	void requestNextCoverArt();
	void abortOnlineSearching();
	void updateProgressBar();
	void downloadNextResultImageData();

	bool eventFilter(QObject *object, QEvent *e);
};


class CoverArtDownloaderAddOn : public AddOn
{
	Q_OBJECT
public:
	CoverArtDownloaderAddOn()
		:	AddOn(),
			coverArtWebServiceURL_(COVERART_DEFAULT_URL),
			dialog_(0)
	{
		qDebug("Registering CoverArtDownloaderAddOn");
	}

	virtual ~CoverArtDownloaderAddOn()
	{
	}

	virtual void hookMainWindow(Quasar *mainWindow)
	{
		mainWindow_ = mainWindow;

		int insertIndex = mainWindow->toolsMenu_->indexOf(mainWindow_->idToolsMenuRescanForCoverArt_) + 1;
		qDebug("insertIndex: %d", insertIndex);
		mainWindow->toolsMenu_->insertItem(tr("Cover Art Downloader..."), this, SLOT(openCoverArtDownloader()), 0, -1, insertIndex);
	}

	virtual void hookConfigurationDialog(ConfigurationDialog *cd)
	{
		cd_ = cd;

		QHGroupBox *coverArtDownloaderGroup = new QHGroupBox(tr("Cover Art Download Settings"), cd_->coverArtConfig->vbox());
		coverArtDownloadWebServiceURL_ = new QLineEdit(coverArtDownloaderGroup);
		connect(coverArtDownloadWebServiceURL_, SIGNAL(textChanged(const QString&)),
				this, SLOT(setCoverArtDownloadWebServiceURL(const QString&)));

		connect(cd_, SIGNAL(settingStatesByConfiguration(const Configuration&)),
				this, SLOT(setStatesByConfiguration(const Configuration&)));

		connect(cd_, SIGNAL(settingDefaultSettings()),
				this, SLOT(setDefaultSettings()));
	}

	virtual void writeToConfig(Config &cf)
	{
		cf.setGroup("CoverArt");
		cf.writeEntry("coverArtWebServiceURL", coverArtWebServiceURL_);
	}

	virtual void readFromConfig(Config &cf)
	{
		cf.setGroup("CoverArt");
		coverArtWebServiceURL_ = cf.readEntry("coverArtWebServiceURL", coverArtWebServiceURL_);
	}

	const QString &coverArtWebServiceURL() { return coverArtWebServiceURL_; }

private slots:
	void openCoverArtDownloader()
	{
		if (!dialog_)
		{
			dialog_ = new CoverArtDownloader(&mainWindow_->playListManager_->currentPlayList(), mainWindow_->skinManager_);
			connect(dialog_, SIGNAL(closing()), this, SLOT(closeCoverArtDownloader()));
			loadWidgetPosition(dialog_, qGlobalConfig.configFilename(), "CoverArtDownloader");
		}

#ifdef QTOPIA
		dialog_->showMaximized();
#else
		dialog_->show();
		dialog_->setWindowState(dialog_->windowState() & ~WindowMinimized | WindowActive);
#endif
	}

	void closeCoverArtDownloader()
	{
		mainWindow_->rescanForCoverArt();
		saveWidgetPosition(dialog_, qGlobalConfig.configFilename(), "CoverArtDownloader");
		dialog_ = NULL;
	}

	void setCoverArtDownloadWebServiceURL(const QString& url)
	{
		coverArtWebServiceURL_ = url;
	}

	void setStatesByConfiguration(const Configuration& config)
	{
		coverArtDownloadWebServiceURL_->setText(coverArtWebServiceURL_);
	}

	void setDefaultSettings()
	{
		coverArtDownloadWebServiceURL_->setText(COVERART_DEFAULT_URL);
	}

private:
	Quasar *mainWindow_;

	ConfigurationDialog *cd_;
	QLineEdit *coverArtDownloadWebServiceURL_;

	CoverArtDownloader *dialog_;

	QString coverArtWebServiceURL_;
};

#endif /*COVERARTDOWNLOADER_H_*/
