/***************************************************************************
 *   Copyright (C) 2004 by Steven Scott                                    *
 *   progoth@progoth.com                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include "kascachelist.h"

#include <qstring.h>
#include <qdatetime.h>
#include <qtextcodec.h>
#include <time.h>
#include <debug.h>

#include <qurl.h>


/* taken directly from kurl.cpp */

static QTextCodec * codecForHint( int encoding_hint /* not 0 ! */ )
{
    return QTextCodec::codecForMib( encoding_hint );
}

// encoding_offset:
// 0 encode both @ and /
// 1 encode @ but not /
// 2 encode neither @ or /
static QString encode( const QString& segment, int encoding_offset, int encoding_hint )
{
  const char *encode_string = "/@<>#\"&%?={}|^~[]\'`\\:+";
  encode_string += encoding_offset;

  QCString local;
  if (encoding_hint==0)
    local = segment.local8Bit();
  else
  {
      QTextCodec * textCodec = codecForHint( encoding_hint );
      if (!textCodec)
          local = segment.local8Bit();
      else
          local = textCodec->fromUnicode( segment );
  }

  int old_length = local.length();

  if ( !old_length )
    return segment.isNull() ? QString::null : QString(""); // differentiate null and empty

  // a worst case approximation
  QChar *new_segment = new QChar[ old_length * 3 + 1 ];
  int new_length = 0;

  for ( int i = 0; i < old_length; i++ )
  {
    // 'unsave' and 'reserved' characters
    // according to RFC 1738,
    // 2.2. URL Character Encoding Issues (pp. 3-4)
    // WABA: Added non-ascii
    unsigned char character = local[i];
    if ( (character <= 32) || (character >= 127) ||
         strchr(encode_string, character) )
    {
      new_segment[ new_length++ ] = '%';

      unsigned int c = character / 16;
      c += (c > 9) ? ('A' - 10) : '0';
      new_segment[ new_length++ ] = c;

      c = character % 16;
      c += (c > 9) ? ('A' - 10) : '0';
      new_segment[ new_length++ ] = c;

    }
    else
      new_segment[ new_length++ ] = local[i];
  }

  QString result = QString(new_segment, new_length);
  delete [] new_segment;
  return result;
}

QString encode_string_no_slash(const QString &str, int encoding_hint = 0)
{
   return encode(str, 0, encoding_hint);
}

/*KASCacheList::KASCacheList(QObject *parent, const char *name)
 : QStringList(parent, name)
{
}*/


/*KASCacheList::~KASCacheList()
{
}*/

/**
 * gets the number of cached audioscrobbler entries
 */
int KASCacheList::submissionCount( void ) const
{
    return count() / 6;
}

void KASCacheList::addSubmission( const ScrobSongData &in )
{
    addSubmission( in.artist, in.songtitle, in.album, in.mbid, in.length );
}

void KASCacheList::addSubmission( const QString &artist, const QString &songtitle, const QString &album,
                                  const QString &mbid, int seconds )
{
    append(artist);
    append(songtitle);
    append(album);
    append(mbid );
    append(QString::number( seconds ) );
    
    time_t a_current_time;
	struct tm a_current_time_tm;

	time(&a_current_time);
	
#ifdef WINDOWS
	a_current_time_tm = *gmtime(&a_current_time);
#else
	gmtime_r(&a_current_time, &a_current_time_tm);
#endif
    
	QString time = QString().sprintf("%04d-%02d-%02d %02d:%02d:%02d",
			a_current_time_tm.tm_year + 1900,
			a_current_time_tm.tm_mon + 1,
			a_current_time_tm.tm_mday,
			a_current_time_tm.tm_hour,
			a_current_time_tm.tm_min,
			a_current_time_tm.tm_sec);
	
	DPRINTF("time: %s", (const char*)time.utf8());
    
    append(time);
}

const ScrobSongData KASCacheList::getSubmission(int index) const
{
    ScrobSongData song;
    
    song.artist = (*this)[0 + index * 6];
    song.songtitle = (*this)[1 + index * 6];
    song.album = (*this)[2 + index * 6];
    song.mbid = (*this)[3 + index * 6];
    song.length = (*this)[4 + index * 6].toInt();
    song.time = (*this)[5 + index * 6];

    return song;
}

void KASCacheList::removeFromTop(int count)
{
	count = count * 6;
	for (int i = 0; i < count; i++)
		this->remove(this->begin());
}

QCString KASCacheList::postData( const QString &username, const QString &md5response, int *fetchedItems) const
{
    QString ret;
    ret += "u=" + encode_string_no_slash( username ) + "&s=" + md5response;
    int total = submissionCount();
    
    // if more than 10, only process 10...
    if (total > 10)
    	total = 10;
    
    if (fetchedItems)
    	*fetchedItems = total;
    
    for( int i = 0; i < total; i++ )
    {
        QString n = QString::number( i );
        
        ret += "&a[" + n + "]=" + encode_string_no_slash((*this)[0 + i * 6])
            +  "&t[" + n + "]=" + encode_string_no_slash((*this)[1 + i * 6])
            +  "&b[" + n + "]=" + encode_string_no_slash((*this)[2 + i * 6])
            +  "&m[" + n + "]=" + (*this)[3 + i * 6]
            +  "&l[" + n + "]=" + (*this)[4 + i * 6]
            +  "&i[" + n + "]=" + encode_string_no_slash((*this)[5 + i * 6]);
    }
    
    return ret.utf8();
}
