/*
 * playlistaspectoverviewfilter.cpp
 *
 *  Created on: 20.04.2010
 *      Author: darkstar
 */

#include "playlistaspectoverviewfilter.h"
#include "sqlite3.h"
#include "debug.h"
#include "compathack.h"

PlayListAspectOverviewFilter::PlayListAspectOverviewFilter()
	:	QObject(),
	 	PlayListAspectExtension(),
	 	users_(),
	 	isDirty_(false),
	 	updateFrom_(Input),
		updatingGenres_(false),
		updatingArtists_(false),
		updatingAlbums_(false),
		selectedGenresFilter(""),
		selectedArtistsFilter(""),
		selectedAlbumsFilter("")
{

}

PlayListAspectOverviewFilter::~PlayListAspectOverviewFilter()
{

}

void PlayListAspectOverviewFilter::reset()
{

}

void PlayListAspectOverviewFilter::registerUser(void *user)
{
	bool activeBefore = isActive();

#ifdef QT4
	if (!users_.contains(user))
		users_.append(user);
#else
	if (!users_.containsRef(user))
		users_.append(user);
#endif

	if (!activeBefore)
		update();
}

void PlayListAspectOverviewFilter::unregisterUser(void *user)
{
#ifdef QT4
	users_.removeOne(user);
#else
	users_.removeRef(user);
#endif
}

bool PlayListAspectOverviewFilter::isActive()
{
	return users_.count() > 0;
}

bool PlayListAspectOverviewFilter::isFiltering()
{
	return selectedGenresFilter.length() > 0 || selectedArtistsFilter.length() > 0 || selectedAlbumsFilter.length() > 0;
}

void PlayListAspectOverviewFilter::execute()
{
	DENTERMETHOD;

	// if the overview widget is invisible and no genre, artist or album filter is set, we can just set
	// it into dirty state and thus speed up processing. Once the overview widget is shown again, we check for isDirty
	// in the showEvent method and update the lists' content accordingly...
	if (!isActive() &&
		(genreList_.count() > 1 && genreList_.at(0)->selected) &&
		(artistList_.count() > 1 && artistList_.at(0)->selected) &&
		(albumList_.count() > 1 && albumList_.at(0)->selected))
	{
		isDirty_ = true;
	}
	else
	{
		if (isDirty_ || updateFrom_ == Input)
		{
			isDirty_ = false;

			QString query = "DROP TABLE IF EXISTS " + identifier() + "_unfiltered;"
				"CREATE TEMPORARY TABLE " + identifier() + "_unfiltered AS "
					"SELECT DISTINCT artist, album, genre FROM " + inputViewSource() + ";";

			sqlite3_exec(playListAspect()->playList()->mediaDatabase()->db(), query.toUtf8(), NULL, NULL, NULL);

			updateFilterFromInput();
		}
		else if (updateFrom_ == GenreList)
			updateFilterFromGenreList();
		else if (updateFrom_ == ArtistList)
			updateFilterFromArtistList();
		else if (updateFrom_ == AlbumList)
			updateFilterFromAlbumList();

		updateFrom_ = Input;
	}

	DEXITMETHOD;
}

void PlayListAspectOverviewFilter::updateFromGenreList()
{
	updateFrom_ = GenreList;
	update();
	updateFrom_ = Input;
}

void PlayListAspectOverviewFilter::updateFromArtistList()
{
	updateFrom_ = ArtistList;
	update();
	updateFrom_ = Input;
}

void PlayListAspectOverviewFilter::updateFromAlbumList()
{
	updateFrom_ = AlbumList;
	update();
	updateFrom_ = Input;
}

const QString PlayListAspectOverviewFilter::outputViewSource()
{
	if (isFiltering())
		return playListAspect()->prefix() + "overviewfilter";
	else
		return inputViewSource();
}

const QString PlayListAspectOverviewFilter::identifier()
{
	return playListAspect()->prefix() + "overview";
}

void PlayListAspectOverviewFilter::saveExtensionState(QStringList &dst)
{
	dst.append("genre:" + buildFilter(genreList_, tr("(Undefined genre)")));
	dst.append("artist:" + buildFilter(artistList_, tr("(Unknown artist)")));
	dst.append("album:" + buildFilter(albumList_, tr("(Unknown album)")));
}

void PlayListAspectOverviewFilter::loadExtensionState(QStringList &src)
{
	if (src.count() == 3)
	{
		selectedGenresFilter = src[0].mid(src[0].indexOf(":") + 1);
		selectedArtistsFilter = src[1].mid(src[1].indexOf(":") + 1);
		selectedAlbumsFilter = src[2].mid(src[2].indexOf(":") + 1);
		isDirty_ = true;
	}
}

int PlayListAspectOverviewFilter::selectionCount(OverviewFilterList &list)
{
	int selectedCount = 0;

	for (int i = 1; i < list.count(); ++i)
		if (list.at(i)->selected)
			++selectedCount;

	return selectedCount;
}

QString PlayListAspectOverviewFilter::buildFilter(OverviewFilterList &list, const QString &textForUnknownEntry)
{
	QString filter = "";
	int selectedCount = 0;

	if (list.count() > 1 && !list.at(0)->selected)
		for (int i = 1; i < list.count(); ++i)
			if (list.at(i)->selected)
			{
				++selectedCount;
				if (list.at(i)->text == textForUnknownEntry)
					filter += (selectedCount > 1 ? QString(", ") : QString("")) + "NULL, \"\"";
				else
					filter += (selectedCount > 1 ? QString(", ") : QString("")) + "\"" + dbSafeString(list.at(i)->text) + "\"";
			}

	return filter;
}

void PlayListAspectOverviewFilter::createFilteredTable(const QString &selectedFilter, const QString &srctable, const QString &dsttable, const QString &what, const QString &wherecol)
{
	QString query =
		"DROP TABLE IF EXISTS " + dsttable + ";"
		"CREATE TEMPORARY TABLE " + dsttable + " AS "
			"SELECT DISTINCT " + what + " FROM " + srctable +
			(selectedFilter.length() > 0 ? " WHERE " + wherecol + " IN (" + selectedFilter + ");" : QString(";"));

	DPRINTF("query : %s", (const char*)query.toUtf8());
	sqlite3_exec(playListAspect()->playList()->mediaDatabase()->db(), query.toUtf8(), NULL, NULL, NULL);
}

void PlayListAspectOverviewFilter::doUpdateOnList(OverviewFilterList &list, const QString &selectedFilter, const QString &colname, const QString &srctable, const QString &unknown, const QString &one, const QString &more)
{
	//	NOTE: This query might yield incorrect results if the optimization settings at compile time are too low. Probably a bug in SQLite...
	QString query = "SELECT DISTINCT " + colname + ", " + (selectedFilter.length() > 0 ? colname + " IN (" + selectedFilter + ")" : QString("0")) +
					" AS selected FROM " + srctable + " ORDER BY " + colname + " ASC;";

	DPRINTF("query : %s", (const char*)query.toUtf8());

#ifdef QT4
	qDeleteAll(list);
#endif
	list.clear();

	sqlite3_stmt *vm;
	sqlite3_prepare_v2(playListAspect()->playList()->mediaDatabase()->db(), query.toUtf8(), -1, &vm, 0);

	OverviewFilterListEntry *entry;
	int selcount = 0;

	while (sqlite3_step(vm) == SQLITE_ROW)
	{
		entry = new OverviewFilterListEntry(
			QString::fromUtf8((const char *)sqlite3_column_text(vm, 0)),
			sqlite3_column_int(vm, 1)
		);

		if (entry->selected)
			++selcount;

		list.append(entry);
	}
	sqlite3_finalize(vm);

	entry = new OverviewFilterListEntry(
		list.count() != 1 ? more.arg(list.count()) : one,
		selcount == 0
	);
	list.insert(0, entry);

	if (list.count() > 1 && (list.at(1)->text.isEmpty()))
		list.at(1)->text = unknown;
}

void PlayListAspectOverviewFilter::updateFilterFromInput()
{
	DENTERMETHOD;

	DTIMERINIT(time);

	if (selectedGenresFilter.isEmpty())
		selectedGenresFilter = buildFilter(genreList_, tr("(Undefined genre)"));

	doUpdateOnList(genreList_, selectedGenresFilter, "genre", identifier() + "_unfiltered", tr("(Undefined genre)"), tr("All (1 genre)"), tr("All (%1 genres)"));

	DTIMERPRINT(time, "updateFromInput");

	emit updatedGenreList();

	updateFilterFromGenreList();

	DEXITMETHOD;
}

void PlayListAspectOverviewFilter::updateFilterFromGenreList()
{
	DENTERMETHOD;

	DTIMERINIT(time);

	selectedGenresFilter = buildFilter(genreList_, tr("(Undefined genre)"));
	createFilteredTable(selectedGenresFilter, identifier() + "_unfiltered", identifier() + "_genre_filtered", "artist, album", "genre");

	if (selectedArtistsFilter.isEmpty())
		selectedArtistsFilter = buildFilter(artistList_, tr("(Unknown artist)"));

	doUpdateOnList(artistList_, selectedArtistsFilter, "artist", identifier() + "_genre_filtered", tr("(Unknown artist)"), tr("All (1 artist)"), tr("All (%1 artists)"));

	DTIMERPRINT(time, "updateFromGenreList");

	emit updatedArtistList();

	updateFilterFromArtistList();

	DEXITMETHOD;
}

void PlayListAspectOverviewFilter::updateFilterFromArtistList()
{
	DENTERMETHOD;

	DTIMERINIT(time);

	selectedArtistsFilter = buildFilter(artistList_, tr("(Unknown artist)"));
	createFilteredTable(selectedArtistsFilter, identifier() + "_genre_filtered", identifier() + "_artist_filtered", "album", "artist");

	if (selectedAlbumsFilter.isEmpty())
		selectedAlbumsFilter = buildFilter(albumList_, tr("(Unknown album)"));

	doUpdateOnList(albumList_, selectedAlbumsFilter, "album", identifier() + "_artist_filtered", tr("(Unknown album)"), tr("All (1 album)"), tr("All (%1 albums)"));

	DTIMERPRINT(time, "updateFromArtistList");

	emit updatedAlbumList();

	updateFilterFromAlbumList();

	DEXITMETHOD;
}

void PlayListAspectOverviewFilter::updateFilterFromAlbumList()
{
	DENTERMETHOD;

	selectedAlbumsFilter = buildFilter(albumList_, tr("(Unknown album)"));

	if (isFiltering())
	{
		QString query =
			"DROP TABLE IF EXISTS " + outputViewSource() + ";"
			"CREATE TEMPORARY TABLE " + outputViewSource() + " AS "
				"SELECT * FROM " + inputViewSource() + " WHERE ";

		if (selectedGenresFilter.length() > 0)
		{
			query += "genre IN (" + selectedGenresFilter + ")";
			if (selectedArtistsFilter.length() > 0 || selectedAlbumsFilter.length() > 0)
				query += " AND ";
		}

		if (selectedArtistsFilter.length() > 0)
		{
			query += "artist IN (" + selectedArtistsFilter + ")";
			if (selectedAlbumsFilter.length() > 0)
				query += " AND ";
		}

		if (selectedAlbumsFilter.length() > 0)
			query += "album IN (" + selectedAlbumsFilter + ")";

		query += ";";

		DTIMERINIT(time);
		sqlite3_exec(playListAspect()->playList()->mediaDatabase()->db(), query.utf8(), NULL, NULL, NULL);
		// the schema changed, so we need to finalize and re-prepare statements...
		playListAspect()->playList()->mediaDatabase()->ensurePreparedStatementsAreValid();
		DTIMERPRINT(time, (const char*)query.toUtf8());
	}
	else // no overview filtering...
	{
		QString query = "DROP TABLE IF EXISTS " + playListAspect()->prefix() + "overviewfilter" + ";";
		DTIMERINIT(time);
		sqlite3_exec(playListAspect()->playList()->mediaDatabase()->db(), query.utf8(), NULL, NULL, NULL);
		// the schema changed, so we need to finalize and re-prepare statements...
		playListAspect()->playList()->mediaDatabase()->ensurePreparedStatementsAreValid();
		DTIMERPRINT(time, (const char*)query.toUtf8());
	}

	DEXITMETHOD;
}
