/*
 * Copyright (C) 2007 Andre Beckedorf <evilJazz _AT_ katastrophos _DOT_ net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <qapplication.h>
#include <qimage.h>
#include <qdir.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qregexp.h>

#include "coverartprovider.h"
#include "mediaidentifier.h"

#ifdef QT4
#include <QWidget>
#include <QDesktopWidget>
#include <QProgressDialog>
#else
#include "qprogressdialog.h"
#endif

#include "debug.h"
#include "compathack.h"

CoverArtProvider & CoverArtProvider::singleton()
{
	static CoverArtProvider instance;
	return instance;
}


CoverArtProvider::CoverArtProvider()
	:	QObject(),
		sourceDirectory_("")
{
	validExtensionsList_ = QStringList() << "jpeg" << "jpg" << "png" << "bmp" << "tif" << "tiff";
}

CoverArtProvider::~CoverArtProvider()
{
}

void CoverArtProvider::setCoverArtDirectory(const QString &directory)
{
	if (directory != sourceDirectory_)
	{
		sourceDirectory_ = directory;
		rescan();
	}
}

void CoverArtProvider::rescan()
{
	coverArtImageFilenames_.clear();
	simplifiedCoverArtImageFilenames_.clear();

	if (!sourceDirectory_.isEmpty() && QDir(sourceDirectory_).exists())
	{
#ifdef QT4
		QProgressDialog progress(tr("Please wait..."), tr("Cancel"), 0, NULL);
#else
		QProgressDialog progress(tr("Please wait..."), tr("Cancel"), 0, NULL, "progress", true);
#endif

		progress.setWindowTitle(tr("Scanning for cover art images..."));

		progress.setMinimumDuration(0);
		progress.setFixedWidth(qMin((int)(qApp->desktop()->width() * 0.9), 640));
		progress.setAutoClose(false);
		progress.setAutoReset(false);
		progress.show();
		qApp->processEvents();

		QString root = QFileInfo(sourceDirectory_).absoluteFilePath();

		fillFileList(coverArtImageFilenames_, root, true);

		int rootLength = root.length();

		for (QStringList::Iterator it = coverArtImageFilenames_.begin(); it != coverArtImageFilenames_.end(); ++it )
			simplifiedCoverArtImageFilenames_.append(simplifyString((*it).mid(rootLength).toLower()));

		progress.hide();
		qApp->processEvents();
	}
}

void CoverArtProvider::fillFileList(QStringList &filelist, const QString &path, const bool subdir)
{
	QDir dir(path);
	if (!dir.exists()) return;

	dir.setFilter(subdir ? QDir::Files | QDir::Dirs : QDir::Files);
	dir.setSorting(QDir::Unsorted);

	QString fileName;

#ifdef QT4
	QFileInfoList list = dir.entryInfoList();
	QListIterator<QFileInfo> it(list);

	while (it.hasNext())
	{
		const QFileInfo *info = &it.next();
#else
	QFileInfo *info;

	const QFileInfoList *list = dir.entryInfoList();
	QFileInfoListIterator it(*list);

	for (; (info = it.current()); ++it)
	{
#endif
		if (info->isFile())
		{
			// only add the file, if its extension is known...
			if (validExtension(info->suffix()))
				filelist.append(info->absoluteFilePath());
		}
		else
		{
			fileName = info->fileName();
			if (fileName != "." && fileName != "..")
				fillFileList(filelist, info->absoluteFilePath(), subdir);
		}
	}
}

QString CoverArtProvider::simplifyString(const QString &input)
{
	QString output = input;
	output.replace(QRegExp("\\/"), " "); // AC/DC -> AC DC
	output.replace(QRegExp("\\-"), " "); // AC-DC -> AC DC
	output.replace(QRegExp("\\_"), " "); // AC_DC -> AC DC
	output.replace(QRegExp("\\(.*\\)"), ""); // remove (Reissue)
	output.replace(QRegExp("\\[.*\\]"), ""); // remove [Reissue]
	output.replace(QRegExp("[,.+&/=:#!?]"), " ");
#ifdef QT4
	output.replace(QRegExp("CD.*[0-9]+", Qt::CaseInsensitive), " ");
#else
	output.replace(QRegExp("CD.*[0-9]+", false), " ");
#endif
	output = output.simplified();

	DPRINTF("simplifyString input: '%s', output: '%s'", (const char*)input.toUtf8(), (const char*)output.toUtf8());
	return output;
}

bool CoverArtProvider::loadCoverDirectly(QImage &dstImage, const QString &artist, const QString &album, const QString &filename)
{
	bool result = false;

	if (!filename.isEmpty())
	{
		result = gMediaIdentifier.loadCoverArtFromFile(filename, dstImage);

		// Check for cover files in the same directory the media file is in...
		if (!result)
		{
			QStringList imageFilenames;
			fillFileList(imageFilenames, QFileInfo(filename).absoluteFilePath(), false);

			if (imageFilenames.count() > 0)
			{
				int i = 0;
				while (i < imageFilenames.count() && !result)
				{
					 result = dstImage.load(imageFilenames[i]);
					 ++i;
				}
			}
		}
	}

	if (!result && !artist.isEmpty() && !album.isEmpty())
	{
		// Check for cover files in a separate tree...
		QString artistMatch = simplifyString(artist.toLower());
		QString albumMatch = simplifyString(album.toLower());

		DPRINTF("artist: %s", (const char*)artist.toUtf8());
		DPRINTF("album: %s", (const char*)album.toUtf8());

		DPRINTF("artistMatch: %s", (const char*)artistMatch.toUtf8());
		DPRINTF("albumMatch: %s", (const char*)albumMatch.toUtf8());

		QStringList matchedFilenames;
		int index = 0;

		for (QStringList::Iterator it = simplifiedCoverArtImageFilenames_.begin(); it != simplifiedCoverArtImageFilenames_.end(); ++it			, ++index)
		{
			if ((*it).lastIndexOf(artistMatch, -1) > -1 && (*it).lastIndexOf(albumMatch, -1) > -1)
			{
				DPRINTF("found match: %s", (const char *)coverArtImageFilenames_[index].toUtf8());
				matchedFilenames.append(coverArtImageFilenames_[index]);
			}
		}

		int i = 0;
		while (i < matchedFilenames.count() && !result)
		{
			 result = dstImage.load(matchedFilenames[i]);
			 ++i;
		}
	}

	return result;
}

QStringList CoverArtProvider::findCoverArtImageFilenames(const QString &artist, const QString &album, const QString &filename)
{
	DENTERMETHODF("[%s, %s, %s]", (const char *)artist.toUtf8(), (const char *)album.toUtf8(), (const char *)filename.toUtf8());

	QStringList result;

	// Check for cover files in the same directory the media file is in...
	if (!filename.isEmpty())
		fillFileList(result, QFileInfo(filename).absoluteFilePath(), false);

	if (!artist.isEmpty() && !album.isEmpty())
	{
		// Check for cover files in a separate tree...
		QString artistMatch = simplifyString(artist.toLower());
		QString albumMatch = simplifyString(album.toLower());
		int index = 0;

		for (QStringList::Iterator it = simplifiedCoverArtImageFilenames_.begin(); it != simplifiedCoverArtImageFilenames_.end(); ++it			, ++index)
		{
			if ((*it).lastIndexOf(artistMatch, -1) > -1 && (*it).lastIndexOf(albumMatch, -1) > -1)
				result.append(coverArtImageFilenames_[index]);
		}
	}

	DEXITMETHODF("[%s, %s, %s]", (const char *)artist.toUtf8(), (const char *)album.toUtf8(), (const char *)filename.toUtf8());

	return result;
}
