#include "helpers.h"

#include <stdlib.h>

#include <qapplication.h>

#ifdef QTOPIA
#include <qobjectlist.h>
#include <qpe/config.h>
#else
#ifdef QT4
#include <QWidget>
#include <QDesktopWidget>
#include <QVector>
#include <QSettings>
#else
#include <qptrlist.h>
#include <qobjectlist.h>
#include <qobject.h>
#include <qmemarray.h>
#include <config.h>
#endif
#endif

#ifdef OSX_Carbon
#include <Carbon/Carbon.h>
//#include <CoreFoundation/CoreFoundation.h>
#include <SystemConfiguration/SCDynamicStoreCopySpecific.h>
#include <SystemConfiguration/SCSchemaDefinitions.h>
#endif

#ifdef WINDOWS
#include <qregexp.h>
#include <windows.h>
#include "winhttp.h"
#endif

#include <qprocess.h>
#include <qurl.h>
#include <qdir.h>

#include "debug.h"
#include "compathack.h"

QString formatFileSize(long long filesize)
{
	QString result;

	if (filesize > 1024 * 1024 * 1024)
		result = QString().sprintf("%.2f GiB", (float)filesize / (1024 * 1024 * 1024));
	else if (filesize > 1024 * 1024)
		result = QString().sprintf("%.2f MiB", (float)filesize / (1024 * 1024));
	else
		result = QString().sprintf("%.1f KiB", (float)filesize / 1024);

	return result;
}

QString formatBitrate(long long bitrate)
{
	QString result;

	if (bitrate > 1000 * 1000 * 1000)
		result = QString().sprintf("%.2f Gbit/s", (float)bitrate / (1000 * 1000 * 1000));
	else if (bitrate > 1000 * 1000)
		result = QString().sprintf("%.2f Mbit/s", (float)bitrate / (1000 * 1000));
	else if (bitrate > 1000)
		result = QString().sprintf("%.1f kbit/s", (float)bitrate / 1000);
	else
		result = QString().sprintf("%.1f bit/s", (float)bitrate);

	return result;
}

QString formatPlayTime(int lengthInSeconds, bool useShortestFormIfPossible)
{
	uint absLengthInSeconds = (lengthInSeconds < 0) ? -lengthInSeconds : lengthInSeconds;

	uint playtimeMins = absLengthInSeconds / 60;
	uint playtimeSecsRem = absLengthInSeconds % 60;
	uint playtimeHours = playtimeMins / 60;
	uint playtimeMinsRem = playtimeMins % 60;
	uint playtimeDays = playtimeHours / 24;
	uint playtimeHoursRem = playtimeHours % 24;

	QString result;

	if (playtimeDays > 0)
		result = QString().sprintf("%d:%02d:%02d:%02d", playtimeDays, playtimeHoursRem, playtimeMinsRem, playtimeSecsRem);
	else if (playtimeHoursRem > 0)
		result = QString().sprintf("%d:%02d:%02d", playtimeHoursRem, playtimeMinsRem, playtimeSecsRem);
	else if (playtimeMinsRem > 0 || !useShortestFormIfPossible)
		result = QString().sprintf("%d:%02d", playtimeMinsRem, playtimeSecsRem);
	else
		result = QString().sprintf("%d", playtimeSecsRem);

	if (lengthInSeconds < 0)
		result = "-" + result;

	return result;
}

void setStyleRecursively(QWidget *w, QStyle *s)
{
#ifdef QT4
	w->setPalette(QPalette());
#else
	w->unsetPalette();
#endif

    w->setStyle(s);

#ifdef QT4
    const QList<QObject*> children = w->children();
    if (children.isEmpty())
    	return;
#else
    const QObjectList *children = w->children();
    if (!children)
        return;
#endif

#ifdef QTOPIA
    QObjectListIt childit(*children);
#else
#ifdef QT4
    QListIterator<QObject*> childit(children);
#else
    QPtrListIterator<QObject> childit(*children);
#endif
#endif

#ifdef QT4


#else
    QObject *child;
    while ((child = childit.current()) != 0)
    {
        ++childit;

        if (child->isWidgetType() && !child->isA("QPopupMenu"))
        	setStyleRecursively((QWidget *) child, s);
    }
#endif
}

void saveWidgetPosition(QWidget *widget, const QString &configName, const QString &groupName)
{
#ifdef QT4
	QSettings cf(configName, QSettings::IniFormat);
	cf.beginGroup(groupName);
	cf.setValue("geometry", widget->saveGeometry());
	cf.endGroup();
#else
#ifndef QTOPIA
	Config cf(configName, Config::File);
	cf.setGroup(groupName);

	bool wasMaximized = widget->isMaximized();

	cf.writeEntry("isMaximized", wasMaximized);

	if (wasMaximized)
		widget->showNormal();

	cf.writeEntry("x", widget->pos().x());
	cf.writeEntry("y", widget->pos().y());

	cf.writeEntry("width", widget->size().width());
	cf.writeEntry("height", widget->size().height());

#if defined(WINDOWS) || defined(OSX)
	cf.writeEntry(
		"hadFrame",
		widget->pos().x() != widget->geometry().x() || widget->pos().y() != widget->geometry().y()
	);
#endif

	if (wasMaximized)
		widget->showMaximized();
#endif
#endif
}

void loadWidgetPosition(QWidget *widget, const QString &configName, const QString &groupName, int defX, int defY, int defWidth, int defHeight)
{
#ifdef QT4
	QSettings cf(configName, QSettings::IniFormat);
	cf.beginGroup(groupName);
	widget->restoreGeometry(cf.value("geometry").toByteArray());
	cf.endGroup();
#else
#ifndef QTOPIA
	Config cf(configName, Config::File);
	cf.setGroup(groupName);

	QSize size(cf.readNumEntry("width", defWidth), cf.readNumEntry("height", defHeight));
	QPoint point(cf.readNumEntry("x", defX), cf.readNumEntry("y", defY));

	QRect screenRect = QApplication::desktop()->availableGeometry(QApplication::desktop()->primaryScreen());

	if (defX == -1 && point.x() == defX)
		point.setX(qMax(0, screenRect.x() + (screenRect.width() - size.width()) / 2));

	if (defY == -1 && point.y() == defY)
		point.setY(qMax(0, screenRect.y() + (screenRect.height() - size.height()) / 2));

	// This is an ugly workaround for when we can't determine the frame offset...
#if defined(WINDOWS) || defined(OSX)
	bool hadFrame = cf.readBoolEntry("hadFrame", true);
	if (!hadFrame && qApp->mainWidget())
	{
		int diffX = qApp->mainWidget()->geometry().x() - qApp->mainWidget()->frameGeometry().x();
		int diffY = qApp->mainWidget()->geometry().y() - qApp->mainWidget()->frameGeometry().y();

		point.setX(point.x() - diffX);
		point.setY(point.y() - diffY);
	}
#endif

	// Use a two step approach to avoid flickering and resizing and
	// work around an issue with some WMs which causes the windows position
	// to be wrongly reported if we save later on and the user didn't change
	// the position or size of the frame.

	// first resize and positioning so Qt knows what we are up to
	widget->resize(size);
	widget->move(point);
	// show window, so the WM is prepared...
	widget->showNormal();
	// finally set again, so the WM explicitly gets it...
	widget->resize(size);
	widget->move(point);

	if (cf.readBoolEntry("isMaximized", false))
		widget->showMaximized();
#endif
#endif
}

#ifdef WINDOWS

static HMODULE winHTTPModule = 0;
typedef BOOL (STDAPICALLTYPE * pfnWinHttpGetDefaultProxyConfiguration)(WINHTTP_PROXY_INFO*);
typedef BOOL (STDAPICALLTYPE * pfnWinHttpGetIEProxyConfigForCurrentUser)(WINHTTP_CURRENT_USER_IE_PROXY_CONFIG*);
static pfnWinHttpGetDefaultProxyConfiguration GetDefaultProxyConfiguration = 0;
static pfnWinHttpGetIEProxyConfigForCurrentUser GetIEProxyConfigForCurrentUser = 0;

bool getHTTPProxySettings(QString &host, int &port)
{
	bool result = false;
	WINHTTP_PROXY_INFO proxyInfo;
	WINHTTP_CURRENT_USER_IE_PROXY_CONFIG userProxyInfo;
	QString proxies;

	if (!winHTTPModule)
	{
		winHTTPModule = LoadLibrary(L"winhttp.dll");
		GetDefaultProxyConfiguration = (pfnWinHttpGetDefaultProxyConfiguration)GetProcAddress(winHTTPModule, "WinHttpGetDefaultProxyConfiguration");
		GetIEProxyConfigForCurrentUser = (pfnWinHttpGetIEProxyConfigForCurrentUser)GetProcAddress(winHTTPModule, "WinHttpGetIEProxyConfigForCurrentUser");
	}

	if (GetIEProxyConfigForCurrentUser(&userProxyInfo))
	{
    	if (!userProxyInfo.fAutoDetect && userProxyInfo.lpszProxy != NULL)
    	{
    		DPRINTF("User Proxy is set.");
			proxies = QString::fromUcs2((const unsigned short *)userProxyInfo.lpszProxy);
			result = true;
    	}

    	if (userProxyInfo.lpszAutoConfigUrl != NULL)
        	GlobalFree((void *)userProxyInfo.lpszAutoConfigUrl);

    	if (userProxyInfo.lpszProxy != NULL)
        	GlobalFree((void *)userProxyInfo.lpszProxy);

    	if (userProxyInfo.lpszProxyBypass != NULL)
			GlobalFree((void *)userProxyInfo.lpszProxyBypass);
	}

	if (!result && GetDefaultProxyConfiguration(&proxyInfo))
    {
		if (proxyInfo.dwAccessType != WINHTTP_ACCESS_TYPE_NO_PROXY && proxyInfo.lpszProxy != NULL)
		{
			DPRINTF("Global Proxy is set.");
			proxies = QString::fromUcs2((const unsigned short *)proxyInfo.lpszProxy);
			result = true;
		}

    	if (proxyInfo.lpszProxy != NULL)
        	GlobalFree((void *)proxyInfo.lpszProxy);

    	if (proxyInfo.lpszProxyBypass != NULL)
			GlobalFree((void *)proxyInfo.lpszProxyBypass);
    }

	if (result)
	{
		QStringList proxyList = QStringList::split(QRegExp(";| "), proxies, false);
		QString proxy(QString::null);

		if (proxyList.count() == 1)
			proxy = proxyList[0];
		else
			for (QStringList::Iterator it = proxyList.begin(); it != proxyList.end(); ++it)
			{
				if ((*it).startsWith("http="))
				{
					proxy = (*it).remove(0, 5);
					break;
				}
			}

		if (!proxy.isNull())
		{
			DPRINTF("proxy: %s", (const char *)proxy.latin1());

			int i = proxy.findRev(":");
			if (i > -1)
			{
				host = proxy.mid(0, i);
				port = proxy.mid(i + 1).toInt(&result);
			}
			else
				result = false;
		}
	}

    return result;
}

#else

#ifdef OSX_Carbon

static QString CFStringToQString(CFStringRef str)
{
    if(!str)
        return QString();

    CFIndex length = CFStringGetLength(str);
    const UniChar *chars = CFStringGetCharactersPtr(str);
    if (chars)
        return QString(reinterpret_cast<const QChar *>(chars), length);

#ifdef QT4
    QVector<UniChar> buffer(length);
#else
    QMemArray<UniChar> buffer(length);
#endif
    CFStringGetCharacters(str, CFRangeMake(0, length), buffer.data());
    return QString(reinterpret_cast<const QChar *>(buffer.data()), length);
}

bool getHTTPProxySettings(QString &host, int &port)
{
	CFDictionaryRef dict = SCDynamicStoreCopyProxies(NULL);

	CFBooleanRef cfEnabled = NULL;

	if (CFDictionaryGetValueIfPresent(dict, kSCPropNetProxiesHTTPEnable, (const void **)&cfEnabled) && CFBooleanGetValue(cfEnabled))
	{
		CFStringRef cfHost;

		if (CFDictionaryGetValueIfPresent(dict, kSCPropNetProxiesHTTPProxy, (const void **)&cfHost))
		{
			SInt32 proxyPort = 0;
			CFNumberRef cfPort = NULL;

			if (CFDictionaryGetValueIfPresent(dict, kSCPropNetProxiesHTTPPort, (const void **)&cfPort) && CFNumberGetValue(cfPort, kCFNumberSInt32Type, (SInt32 *)&proxyPort))
			{
				port = proxyPort;
				host = CFStringToQString(cfHost);

				return true;
			}
		}
	}

	return false;
}

#else

#ifdef QTOPIA

bool getHTTPProxySettings(QString &host, int &port)
{
	// try to figure out the current proxy settings in Qtopia...
	QString proxyConfig = QDir::homeDirPath() + "/Applications/Network/modules/Proxies.conf";
	if (QFile(proxyConfig).exists())
	{
		Config cfg(proxyConfig, Config::File);
		if (cfg.isValid())
		{
			cfg.setGroup("Properties");

			if (cfg.hasKey("type") && cfg.readNumEntry("type", 0) == 2)
			{
				if (cfg.hasKey("httphost") && cfg.hasKey("httpport"))
				{
					QString proxyHost = cfg.readEntry("httphost", "");
					int proxyPort = cfg.readNumEntry("httpport", 80);

					if (!proxyHost.isEmpty())
					{
						host = proxyHost;
						port = proxyPort;
						return true;
					}
				}
			}
		}
	}

	return false;
}

#else

bool getHTTPProxySettings(QString &host, int &port)
{
	QString env = QString::fromLatin1(getenv("HTTP_PROXY"));

    if (env.isEmpty())
    	env = QString::fromLatin1(getenv("http_proxy"));

    if (env.isEmpty())
    	return false;

    QUrl url(env);

   	port = url.port();
   	host = url.host();

   	return url.isValid();
}

#endif
#endif
#endif

bool isPathAbsolute(const QString &location)
{
	return location.startsWith("/") || (location[1] == ':' && (location[2] == '/' || location[3] == '\\'));
}

QString resolveLocation(const QString &parentDir, const QString &location)
{
	QString result(location);

	if (parentDir.isEmpty())
		return result;

	QString dir = parentDir;
	if (dir[dir.length() - 1] != '/')
		dir.append('/');

	if (!isPathAbsolute(location))
	{
		if (result.startsWith("./"))
			result = dir + result.right(result.length() - 2);
		else if (result.startsWith("../"))
		{
			QDir d(dir);
			while (result.startsWith("../"))
			{
				result.remove(0, 3);
				d.cdUp();
			}

			QString absPath = d.absolutePath();
			if (absPath == "/")
				result = absPath + result;
			else
				result = absPath + "/" + result;
		}
		else // relative to the directory this playlist is in... (obviously the same as ./)
			result = dir + result;
	}

	qDebug("resolveLocation: %s -> %s", (const char *)location.toUtf8(), (const char *)result.toUtf8());
	return result;
}

QString makeLocationRelative(const QString &parentDir, const QString &location)
{
	QString dir = parentDir;
	if (dir[dir.length() - 1] != '/')
		dir.append('/');

	if (location.startsWith(parentDir))
		return location.mid(dir.length());
	else
		return location;
}

void writeToProcess(QProcess *process, const QString &line)
{
#ifdef QT4
	process->write(QString(line + "\n").toUtf8());
	process->waitForBytesWritten(-1);
#else
#ifndef WINDOWS
	process->writeToStdin(line + "\n");
#else
	// work around a bug in Q... Windows Edition 3.3.8, not present in 3.3.7
	QString cmd = line + "\n";
    QByteArray tmp = cmd.local8Bit();
    tmp.resize(cmd.length());
    process->writeToStdin(tmp);
#endif

	process->flushStdin();
#endif
}
