/*
 * PlaybackController.cpp
 *
 *  Created on: 09.07.2010
 *      Author: darkstar
 */

#include <cstdlib>
#include <ctime>

#include <qtimer.h>
#include <qapplication.h>
#include <qmessagebox.h>
#include <textviewdialog.h>

#include "playbackcontroller.h"

#include "mplayer.h"
#include "configuration.h"

#include "debug.h"
#include "compathack.h"

/* Helper */

void swap(int &n1, int &n2)
{
	int n = n1;
	n1 = n2;
	n2 = n;
}

/* PlaybackController */

PlaybackController::PlaybackController()
	:	aspectSource_(NULL),
		playOrder_(PlayOrderNormal),
		randomPlayOrderLUT_(NULL),
		count_(0),
		randomPlayOrderLUTIndex_(0),
		activeItemIndex_(-1),
		activeItem_(0, 0, NULL),
		mplayer_(NULL),
		keyEventCount_(0),
		nextMediaPrefetched_(false),
		currentMediaLength_(0)
{
	DPRINTF("Initialize MPlayer class...");
	mplayer_ = new MPlayer();
	connect(mplayer_, SIGNAL(eof()), this, SLOT(playNextItem()));
	connect(mplayer_, SIGNAL(quit()), this, SLOT(stopCurrentPlayback()));
	connect(mplayer_, SIGNAL(currentPlayTime(int)), this, SLOT(mplayerCurrentPlayTime(int)));
	connect(mplayer_, SIGNAL(errorMessage(QString)), this, SLOT(mplayerErrorMessage(QString)));
	DPRINTF("Done");

	osdTimer_ = new QTimer(this);
	connect(osdTimer_, SIGNAL(timeout()), this, SLOT(osdTimerTimedUpdate()));
}

PlaybackController::~PlaybackController()
{
	delete mplayer_;
	mplayer_ = NULL;
}

void PlaybackController::attachTo(PlayListAspectListViewRenderer *source)
{
	detach();

	if (source)
	{
		aspectSource_ = source;

		connect(aspectSource_, SIGNAL(updateView()), this, SLOT(updateView()));

		activeItem_.setMediaDatabase(aspectSource_->playListAspect()->playList()->mediaDatabase());

		updateView();
	}
}

void PlaybackController::detach()
{
	if (aspectSource_)
	{
		disconnect(aspectSource_, SIGNAL(updateView()), this, SLOT(updateView()));

		activeItemIndex_ = -1;
		count_ = 0;

		if (randomPlayOrderLUT_)
			delete randomPlayOrderLUT_;

		randomPlayOrderLUT_ = NULL;

		activeItem_.setMediaDatabase(NULL);
		activeItem_.setID(0);

		aspectSource_ = NULL;
	}
}

void PlaybackController::updateView()
{
	// try to find the activeItemIndex based on our current cached activeItem...
	count_ = aspectSource_->itemList()->count();
	setActiveItem(&activeItem_);
}

void PlaybackController::setPlayOrder(PlayOrder newOrder)
{
	if (newOrder != playOrder_)
	{
		randomPlayOrderLUTIndex_ = 0;
		playOrder_ = newOrder;
		emit playOrderChanged(playOrder_);
	}
}

void PlaybackController::togglePlayOrder()
{
	PlayOrder newOrder = static_cast<PlayOrder>(playOrder_ + 1);
	if (newOrder >= PlayOrderMax)
		newOrder = PlayOrderNormal;

	setPlayOrder(newOrder);
}

void PlaybackController::toggleRandomPlayOrder()
{
	PlayOrder newOrder = playOrder_;
	if (newOrder >= PlayOrderNormal &&	newOrder <= PlayOrderRepeatAll)
		newOrder = static_cast<PlayOrder>(PlayOrderRandom + newOrder - PlayOrderNormal);
	else if (newOrder >= PlayOrderRandom && newOrder <= PlayOrderRepeatRandom)
		newOrder = static_cast<PlayOrder>(PlayOrderNormal + newOrder - PlayOrderRandom);

	setPlayOrder(newOrder);
}

void PlaybackController::updateRandomPlayOrderLUT()
{
	if (!aspectSource_) return;

	if (!randomPlayOrderLUT_ || count_ != aspectSource_->itemList()->count())
		generateRandomPlayOrderLUT();
}

void PlaybackController::generateRandomPlayOrderLUT()
{
	if (!aspectSource_) return;

	randomPlayOrderLUTIndex_ = 0;

	if (randomPlayOrderLUT_)
		delete randomPlayOrderLUT_;

	randomPlayOrderLUT_ = NULL;

	if (count_ < 1)
		return;

	if (playOrder_ == PlayOrderRandom || playOrder_ == PlayOrderRepeatRandom)
	{
		randomPlayOrderLUT_ = new int[count_];

		for (int i = 0; i != count_; ++i)
			randomPlayOrderLUT_[i] = i;

		// Randomize
		std::srand(time(NULL));

		for (int i = 0; i != count_; ++i)
			swap(randomPlayOrderLUT_[i], randomPlayOrderLUT_[std::rand() % count_]);

		if (activeItemIndex_ > -1)
		{
			// make selected index the first in our play order table...
			for (int i = 0; i != count_; ++i)
				if (randomPlayOrderLUT_[i] == activeItemIndex_)
				{
					swap(randomPlayOrderLUT_[i], randomPlayOrderLUT_[0]);
					break;
				}
		}
	}
}

void PlaybackController::setActiveItemIndex(int itemIndex)
{
	int oldActiveItemIndex = activeItemIndex_;

	activeItemIndex_ = itemIndex;

	// copy from the current item in the list if we have an active item set...
	if (activeItemIndex_ > -1)
		activeItem_ = *(aspectSource_->itemList()->at(activeItemIndex_));

	emit activeItemChanged(activeItemIndex_, oldActiveItemIndex);
}

int PlaybackController::nextActiveItemIndex(bool modifyRandomPlayOrderTableIndex)
{
	return relativeActiveItemIndex(1, modifyRandomPlayOrderTableIndex);
}

int PlaybackController::previousActiveItemIndex(bool modifyRandomPlayOrderTableIndex)
{
	return relativeActiveItemIndex(-1, modifyRandomPlayOrderTableIndex);
}

int PlaybackController::relativeActiveItemIndex(int offset, bool modifyRandomPlayOrderTableIndex)
{
	if (!aspectSource_ || count_ < 1)
		return -1;

	if (offset == 0 || playOrder_ == PlayOrderRepeatOne)
		return (activeItemIndex_ == -1 ? aspectSource_->selectedItemIndex() : activeItemIndex_);

	bool isRandom = playOrder_ == PlayOrderRandom || playOrder_ == PlayOrderRepeatRandom;
	bool shallRepeat = playOrder_ == PlayOrderRepeatAll || playOrder_ == PlayOrderRepeatRandom;

	int itemIndex = -1;
	int refItemIndex = isRandom ? randomPlayOrderLUTIndex_ : activeItemIndex_;

	if (refItemIndex == -1)
		itemIndex = qMax(0, aspectSource_->selectedItemIndex());
	else if (shallRepeat)
	{
		itemIndex = (refItemIndex + offset) % count_;
		if (itemIndex < 0)
			itemIndex += count_;
	}
	else
	{
		if ((offset < 0 && refItemIndex <= 0) || (offset > 0 && refItemIndex >= count_ - 1))
			itemIndex = -1;
		else
			itemIndex = qBound(0, refItemIndex + offset, count_ - 1);
	}

	if (isRandom && itemIndex > -1)
	{
		updateRandomPlayOrderLUT();

		if (modifyRandomPlayOrderTableIndex)
			randomPlayOrderLUTIndex_ = itemIndex;

		itemIndex = randomPlayOrderLUT_[itemIndex];
	}

	return itemIndex;
}

int PlaybackController::activeItemIndex()
{
	if (!aspectSource_ || count_ < 1)
		return -1;

	return activeItemIndex_;
}

void PlaybackController::setActiveItem(PlayListItem *item)
{
	if (!aspectSource_)
		return;

	// try to find activeItem in the itemList and set activeItemIndex
	int mediaIDmatchIndex = -1;
	int newActiveItemIndex = -1;

	if (item)
	{
		PlayListItemList *itemList = aspectSource_->itemList();

		for (int i = 0; i < itemList->count(); ++i)
		{
			PlayListItem *listItem = itemList->at(i);

			if (listItem->mediaID() == item->mediaID())
				mediaIDmatchIndex = i;

			if (listItem->id() == item->id() && mediaIDmatchIndex == i)
			{
				newActiveItemIndex = i;
				break;
			}
		}
	}

	if (newActiveItemIndex == -1 && mediaIDmatchIndex > -1)
		newActiveItemIndex = mediaIDmatchIndex;

	setActiveItemIndex(newActiveItemIndex);
}

PlayListItem *PlaybackController::activeItem()
{
	int index = activeItemIndex();

	if (index > -1)
		return aspectSource_->itemList()->at(index);
	else
		return &activeItem_;
}

PlayListItem *PlaybackController::nextItem(bool modifyRandomPlayOrderTableIndex)
{
	int index = nextActiveItemIndex(modifyRandomPlayOrderTableIndex);

	if (index > -1)
		return aspectSource_->itemList()->at(index);
	else
		return NULL;
}

PlayListItem *PlaybackController::previousItem(bool modifyRandomPlayOrderTableIndex)
{
	int index = nextActiveItemIndex(modifyRandomPlayOrderTableIndex);

	if (index > -1)
		return aspectSource_->itemList()->at(index);
	else
		return NULL;
}

bool PlaybackController::isPlaying()
{
	return mplayer_->isPlaying();
}

void PlaybackController::startOrPauseCurrentPlaybackOrStartNewPlayback()
{
	if (!mplayer_->isPlaying())
		stopCurrentPlaybackAndPlayCurrentItem();
	else
		startOrPauseCurrentPlaybackOrStartNewPlayback(false);
}

void PlaybackController::startOrPauseCurrentPlaybackOrStartNewPlayback(bool forcePlayNewFile)
{
	// Toggle play/pause
	if (mplayer_->isPlaying() && !forcePlayNewFile)
	{
		if (mplayer_->isPaused())
		{
			mplayer_->enableTemporaryOSD();
			mplayer_->pause();

			emit playbackUnpaused();

			osdTimer_->start(3000, true);
		}
		else
		{
			mplayer_->enableTemporaryOSD();
			mplayer_->pause();

			emit playbackPaused();
		}
	}
	else
	{
		generateRandomPlayOrderLUT();
		internalStartPlayback();
	}
}

void PlaybackController::stopCurrentPlaybackAndPlayCurrentItem()
{
	DENTERMETHOD;

	setActiveItemIndex(aspectSource_->selectedItemIndex());

	PlayListItem *item = activeItem();

	if (!item->media())
		return;

	if (item->media()->errorDetected())
	{
		emit problemWithMedia(item);
	}
	else
	{
		internalStopPlayback(true, false);
		startOrPauseCurrentPlaybackOrStartNewPlayback(true);
	}

	DEXITMETHOD;
}

void PlaybackController::stopCurrentPlayback()
{
	setActiveItemIndex(-1);
	internalStopPlayback();
}

void PlaybackController::internalStartPlayback()
{
	Media *media = activeItem()->media();

	emit startingPlayback();

	DPRINTF("Current Media: %s", (const char *)(media->location().toString().toUtf8()));

	nextMediaPrefetched_ = false;
	currentMediaLength_ = media->length();

	// Start mplayer
	if (mplayer_->play(media) == MPlayer::InitializationFailed)
	{
		emit playerInitializationFailed(activeItem());
		return;
	}

	emit playbackStarted();
}

void PlaybackController::internalStopPlayback(bool temporarily, bool stopBackend)
{
	emit stoppingPlayback(temporarily);

	if (stopBackend)
		mplayer_->stop();

	emit playbackStopped(temporarily);
}

void PlaybackController::playNextItem()
{
	int index = nextActiveItemIndex(true);
	if (index > -1)
	{
		++keyEventCount_;
		qApp->processEvents();
		--keyEventCount_;

		if (keyEventCount_ <= 0)
		{
			internalStopPlayback(true, false);
			setActiveItemIndex(index);
			internalStartPlayback();
			keyEventCount_ = 0;
		}
	}
	else
	{
		stopCurrentPlayback();
	}
}

void PlaybackController::playPreviousItem()
{
	int index = previousActiveItemIndex(true);
	if (index > -1)
	{
		++keyEventCount_;
		qApp->processEvents();
		--keyEventCount_;

		if (keyEventCount_ <= 0)
		{
			internalStopPlayback(true, false);
			setActiveItemIndex(index);
			internalStartPlayback();
			keyEventCount_ = 0;
		}
	}
}

void PlaybackController::setAbsoluteSeconds(int secs)
{
	emit playbackSeekedToAbsoluteSeconds(secs);
	mplayer_->seekAbsoluteSeconds(secs);
}

void PlaybackController::setRelativeSeconds(int secs)
{
	mplayer_->enableTemporaryOSD();
	mplayer_->seekRelativeSeconds(secs);

	osdTimer_->start(3000, true);

	// TODO: Implement handling of relative seeking for external services...
	//qExternalServices.playbackSeekedToAbsoluteSeconds(mplayer_->currentPlayTime());
}

void PlaybackController::osdTimerTimedUpdate()
{
	if (!mplayer_->isPaused())
		mplayer_->disableTemporaryOSD();
}

void PlaybackController::mplayerErrorMessage(QString msg)
{
	DENTERMETHODF("[%s]", (const char *)msg.toUtf8());

	if (activeItem_.mediaID())
		activeItem_.media()->setLastError(msg);

	DEXITMETHODF("[%s]", (const char *)msg.toUtf8());
}

void PlaybackController::mplayerCurrentPlayTime(int seconds)
{
	emit currentPlayTime(seconds);

	if (!nextMediaPrefetched_ && currentMediaLength_ - seconds < 21)
	{
		// prefetch playNextItem track...
		nextMediaPrefetched_ = true;

		PlayListItem *item = nextItem(false);
		Media *media = NULL;

		if (item)
			media = item->media();

		if (media)
		{
			QFile mediaFile(media->location());

			// make sure all necessary data is read from database...
			if (media->isAudio())
				MediaAudio *audio = media->mediaAudio();
			else if (media->isVideo())
				MediaVideo *video = media->mediaVideo();

#ifdef QT4
			if (qConfig.isPrefetchFileEnabled && mediaFile.open(QIODevice::Unbuffered | QIODevice::ReadOnly))
#else
			if (qConfig.isPrefetchFileEnabled && mediaFile.open(IO_Raw | IO_ReadOnly))
#endif
			{
				char buffer[65536];

				int maxFetchBytes = 1024 * 1024;
				if (qConfig.isCacheEnabled && (qConfig.cacheSize + 128) * 1024 > maxFetchBytes)
					maxFetchBytes = (qConfig.cacheSize + 128) * 1024;

				QTime time;
				time.start();

		        while (!mediaFile.atEnd() && mediaFile.at() < maxFetchBytes)
		        {
		        	// only read a block any x milliseconds
		        	// in order not to overload the I/O and induce skips
		        	while (time.elapsed() < 25)
		        		qApp->processOneEvent();

		        	mediaFile.readBlock(&buffer[0], 65536);

		        	time.restart();
		        }

		        mediaFile.close();
			}

			emit mediaPrefetched(item);
		}
	}
}

