/*
 * playlist.cpp
 *
 *  Created on: 18.11.2009
 *      Author: darkstar
 */

#include <qtextstream.h>
#ifndef QT4
#include <qtl.h>
#endif

#ifdef QT4
#include <QTextCodec>
#endif

#include "playlist.h"
#include "helpers.h"
#include "media.h"
#include "mediadatabase.h"
#include "mediaidentifier.h"
#include "sqlite3.h"

#include "configuration.h"
#include "debug.h"
#include "compathack.h"

void clearPlayListItemList(PlayListItemList *items)
{
#ifdef QT4
	qDeleteAll(*items);
#endif
	items->clear();
}

PlayList::PlayList(MediaDatabase *mediaDatabase, const QString dbTable)
	:	QObject(),
	 	mediaDatabase_(mediaDatabase),
	 	dbTable_(dbTable),
		stmtSelectTotalFileSizeAndCountForMediaLocation(NULL),
		stmtSelectAllMediaIDsForMediaLocation(NULL),
		stmtInsertMediaIDIntoPlaylist(NULL),
		type_(Static),
		playListFileName_(QString::null),
		dynPlayList_(),
	 	updateCount_(0),
		itemListRequiresUpdate_(true),
		items_()
{
	connect(mediaDatabase_, SIGNAL(preparingStatements(sqlite3*)),
			this, SLOT(mediaDatabasePreparingStatements(sqlite3*)));

	connect(mediaDatabase_, SIGNAL(finalizingStatements(sqlite3*)),
			this, SLOT(mediaDatabaseFinalizingStatements(sqlite3*)));
}

PlayList::~PlayList()
{
	clearPlayListItemList(&items_);
}

void PlayList::beginUpdate()
{
	DENTERMETHOD;

	if (updateCount_ == 0)
	{
		blockSignals(true);
		mediaDatabase_->beginUpdate();
		itemListRequiresUpdate_ = true;
	}
	++updateCount_;

	DEXITMETHOD;
}

void PlayList::endUpdate()
{
	DENTERMETHOD;

	--updateCount_;
	DPRINTF("updateCount: %d", updateCount_);

	if (updateCount_ == 0)
	{
		DTIMERINIT(time);
		mediaDatabase_->endUpdate();
		DTIMERPRINT(time, "mediaDatabase_->endUpdate()");

		DTIMERSTART(time);
		blockSignals(false);
		propagateChanges();
		DTIMERPRINT(time, "change propagation");
	}

	DEXITMETHOD;
}

void PlayList::propagateChanges()
{
	DENTERMETHOD;

	if (updateCount_ == 0)
	{
		DPRINTF("Emitting playListChanged()...");
		emit playListChanged();
	}

	DEXITMETHOD;
}

void PlayList::setType(Type type)
{
	if (type_ != type)
	{
		type_ = type;
		reset();
	}
}

void PlayList::requireRepopulation()
{
	DENTERMETHOD;

	if (updateCount_ == 0)
		repopulate();

	DEXITMETHOD;
}

void PlayList::repopulate()
{
	DENTERMETHOD;

	beginUpdate();

	clearTable();
	addFromFileList(dynPlayList_, true);

	endUpdate();

	DEXITMETHOD;
}

void PlayList::reset()
{
	DENTERMETHOD;

	playListFileName_ = QString::null;
	clear();

	DEXITMETHOD;
}

void PlayList::clear()
{
	DENTERMETHOD;

	dynPlayList_.clear();
	clearTable();

	DEXITMETHOD;
}

void PlayList::clearTable()
{
	DENTERMETHOD;

	beginUpdate();

	QString query = "DELETE FROM \"" + dbSafeString(dbTable_) + "\";";
	sqlite3_exec(mediaDatabase_->db(), (const char *)query.toUtf8(), NULL, NULL, NULL);

	endUpdate();

	DEXITMETHOD;
}

void PlayList::setDynPlayList(const QExtStringList &dynPlayList)
{
	DENTERMETHOD;

	if (isDynamic())
	{
		dynPlayList_ = dynPlayList;
		requireRepopulation();
	}

	DEXITMETHOD;
}

void PlayList::addFile(const QString &fileName)
{
	DENTERMETHODF("[%s]", (const char*)fileName.toUtf8());

	QString location = QFileInfo(fileName).absoluteFilePath();

	if (isDynamic())
	{
		dynPlayList_.append(location);
		requireRepopulation();
	}
	else
	{
		QExtStringList filelist;
		filelist.append(location);
		addFromFileList(filelist, false);
	}

	DEXITMETHODF("[%s]", (const char*)fileName.toUtf8());
}

void PlayList::addDir(const QString &path, const bool subdir)
{
	DENTERMETHODF("[%s]", (const char*)path.toUtf8(), subdir ? "true" : "false");

	QString location = path;

	if (location.right(1) != "/")
		location += "/";

	if (subdir)
		location += "**";

	if (isDynamic())
	{
		dynPlayList_.append(location);
		requireRepopulation();
	}
	else
	{
		QExtStringList filelist;
		filelist.append(location);
		addFromFileList(filelist, true);
	}

	DEXITMETHODF("[%s]", (const char*)path.toUtf8(), subdir ? "true" : "false");
}

void PlayList::addURL(const QString &url)
{
	DENTERMETHODF("[%s]", (const char*)url.toUtf8());

	if (isDynamic())
	{
		dynPlayList_.append(url);
		requireRepopulation();
	}
	else
	{
		QExtStringList filelist;
		filelist.append(url);
		addFromFileList(filelist, false);
	}

	DEXITMETHODF("[%s]", (const char*)url.toUtf8());
}

int PlayList::getNextIndex() const
{
	DENTERMETHOD;

	int result = 0;
	sqlite3_stmt *vm;

	// query playlist index for first new item...
	QString query = "SELECT MAX(idx)+1 FROM \"" + dbSafeString(dbTable_) + "\";";
	sqlite3_prepare_v2(mediaDatabase_->db(), (const char*)query.toUtf8(), -1, &vm, 0);

	if (sqlite3_step(vm) == SQLITE_ROW)
		result = sqlite3_column_int(vm, 0);

	sqlite3_finalize(vm);

	DEXITMETHOD;

	return result;
}

void PlayList::appendMediaIDs(const MediaIDList &mediaIDs, PlayListItemList *outItemList)
{
	DENTERMETHOD;

	if (outItemList)
	{
		// This is for exclusive mode and requires a subsequent call to PlayList::storeItemList or the new
		// items will be lost.
		PlayListItem *item;
		for (MediaIDList::ConstIterator it = mediaIDs.begin(); it != mediaIDs.end(); ++it )
		{
			item = new PlayListItem((*it), NULL, mediaDatabase_);
			outItemList->append(item);
			// Also add to our items list to take care of the ownership.
			items_.append(item);
			itemListRequiresUpdate_ = false;
		}
	}
	else
	{
		beginUpdate();

		int newIndex = getNextIndex();

		sqlite3_stmt *vm;

		QString query = "INSERT INTO \"" + dbSafeString(dbTable_) + "\" VALUES(?1, NULL, ?2);";
		sqlite3_prepare_v2(mediaDatabase_->db(), (const char*)query.toUtf8(), -1, &vm, 0);
		for (MediaIDList::ConstIterator it = mediaIDs.begin(); it != mediaIDs.end(); ++it )
		{
			sqlite3_bind_int(vm, 1, newIndex);
			sqlite3_bind_int(vm, 2, (*it));
			sqlite3_step(vm);
			sqlite3_reset(vm);
			++newIndex;
		}

		sqlite3_finalize(vm);

		endUpdate();
	}

	DEXITMETHOD;
}

void PlayList::deletePlayListIDs(const PlayListIDList &playListIDs)
{
	DENTERMETHOD;

	beginUpdate();

	sqlite3_stmt *vm;

	// TODO: Optimize delete query to use WHERE playlist_id IN (...)
	QString query = "DELETE FROM " + dbSafeString(dbTable_) + " WHERE playlist_id = ?1;";
	sqlite3_prepare_v2(mediaDatabase_->db(), query.toUtf8(), -1, &vm, 0);

	for (PlayListIDList::ConstIterator it = playListIDs.begin(); it != playListIDs.end(); ++it )
	{
		sqlite3_bind_int(vm, 1, (*it));
		sqlite3_step(vm);
		sqlite3_reset(vm);
	}

	sqlite3_finalize(vm);

	endUpdate();

	DEXITMETHOD;
}

void PlayList::mediaDatabasePreparingStatements(sqlite3 *db)
{
	DENTERMETHOD;

	QString query =
		"SELECT SUM(filesize), COUNT() "
		"FROM media, media_location "
		"WHERE media.location_id = media_location.location_id AND media_location.location = ?1";

	sqlite3_prepare_v2(db, query.toUtf8(), -1, &stmtSelectTotalFileSizeAndCountForMediaLocation, 0);

	query =
		"SELECT media_id "
		"FROM media, media_location "
		"WHERE media_location.location = ?1 AND media.location_id = media_location.location_id "
		"ORDER BY filename";

	sqlite3_prepare_v2(db, query.toUtf8(), -1, &stmtSelectAllMediaIDsForMediaLocation, 0);

	query =
		"INSERT INTO \"" + dbSafeString(dbTable_) + "\" "
		"VALUES(?1, ?2, ?3);";

	sqlite3_prepare_v2(db, query.toUtf8(), -1, &stmtInsertMediaIDIntoPlaylist, 0);

	DEXITMETHOD;
}

void PlayList::mediaDatabaseFinalizingStatements(sqlite3 *db)
{
	DENTERMETHOD;

	if (stmtSelectTotalFileSizeAndCountForMediaLocation)
	{
		sqlite3_finalize(stmtSelectTotalFileSizeAndCountForMediaLocation);
		stmtSelectTotalFileSizeAndCountForMediaLocation = NULL;
	}

	if (stmtSelectAllMediaIDsForMediaLocation)
	{
		sqlite3_finalize(stmtSelectAllMediaIDsForMediaLocation);
		stmtSelectAllMediaIDsForMediaLocation = NULL;
	}

	if (stmtInsertMediaIDIntoPlaylist)
	{
		sqlite3_finalize(stmtInsertMediaIDIntoPlaylist);
		stmtInsertMediaIDIntoPlaylist = NULL;
	}

	DEXITMETHOD;
}

void PlayList::fillFileList(QExtStringList &filelist, const QString &path, const bool subdir)
{
	DENTERMETHOD;

	QDir dir(path);
	if (!dir.exists())
	{
		DEXITMETHOD;
		return;
	}

	dir.setFilter(QDir::Files | QDir::Hidden | (subdir ? QDir::Dirs : QDir::Files));
	dir.setSorting(QDir::Unsorted);

	QExtStringList localFiles;
	uint localFilesTotalSize = 0;
	QExtString fileName;

#ifdef QT4
	QFileInfoList list = dir.entryInfoList();
	QListIterator<QFileInfo> it(list);

	while (it.hasNext())
	{
		const QFileInfo *info = &it.next();
#else
	QFileInfo *info;

	const QFileInfoList *list = dir.entryInfoList();
	QFileInfoListIterator it(*list);

	for (; (info = it.current()); ++it)
	{
#endif
		if (info->isFile())
		{
			// only add the file, if its extension is known...
			if (qConfig.isValidExtension(info->suffix()))
			{
				localFiles.append(info->absoluteFilePath());
				localFilesTotalSize += info->size();
			}
		}
		else
		{
			fileName = info->fileName();
			if (fileName != "." && fileName != "..")
				fillFileList(filelist, info->absoluteFilePath(), subdir);
		}
	}

	if (localFiles.count() > 0 && localFilesTotalSize > 0)
	{
#ifdef WINDOWS
		// On Windows canonicalPath() has a really weird and slow behavior (possibly due to QT_GETCWD and QT_CHDIR calls)
		// using absPath() instead...
		QString canonicalPath = dir.absPath();
#else
		QString canonicalPath = dir.canonicalPath();
#endif
		if (canonicalPath.right(1) != "/")
			canonicalPath += '/';

		sqlite3_bind_text(stmtSelectTotalFileSizeAndCountForMediaLocation, 1, canonicalPath.toUtf8(), -1, SQLITE_TRANSIENT);

		uint filesize = 0;
		uint count = 0;

		if (sqlite3_step(stmtSelectTotalFileSizeAndCountForMediaLocation) == SQLITE_ROW)
		{
			filesize = sqlite3_column_int(stmtSelectTotalFileSizeAndCountForMediaLocation, 0);
			count = sqlite3_column_int(stmtSelectTotalFileSizeAndCountForMediaLocation, 1);
		}

		DHPRINTF("canonicalPath: %s", (const char*)canonicalPath.toUtf8());
		DHPRINTF("filesize: %d, count: %d", filesize, count);

		sqlite3_reset(stmtSelectTotalFileSizeAndCountForMediaLocation);

		if (filesize == localFilesTotalSize && count == localFiles.count())
		{
			DHPRINTF("using fastpath 1");
			sqlite3_bind_text(stmtSelectAllMediaIDsForMediaLocation, 1, canonicalPath.toUtf8(), -1, SQLITE_TRANSIENT);

			qSort(localFiles);
			QExtStringList::Iterator it = localFiles.begin();

			while (sqlite3_step(stmtSelectAllMediaIDsForMediaLocation) == SQLITE_ROW)
			{
				QExtString &item = *it;
				item.setID(sqlite3_column_int(stmtSelectAllMediaIDsForMediaLocation, 0));
				DHPRINTF("item: %s, id: %d", (const char *)item.toUtf8(), item.id());
				++it;
			}

			sqlite3_reset(stmtSelectAllMediaIDsForMediaLocation);
		}
		// TODO: Implement and profile a second fast path attempt, that reads
		// partially available information from the database and assigns media IDs
		// based on this information...

		filelist += localFiles;
	}

	DEXITMETHOD;
}

void PlayList::addFromFileList(const QExtStringList &filelist, bool interpretPlayList, const QString &playlistDirectory)
{
	DENTERMETHOD;

	DTIMERINIT(timer);

	QExtStringList list(filelist);

	// temporarily unblock signal emission...
	blockSignals(false);
	emit startingActivity(tr("Scanning for files..."));

	// expand various entries in the list if specifically wished for...
	if (interpretPlayList)
	{
		// expand dynamic playlist to real content...
		QExtStringList expandedList;
		for (QExtStringList::ConstIterator it = list.begin(); it != list.end(); ++it)
		{
			QString location(*it);

			if (location.startsWith("#") || location.startsWith(";"))
				continue;

			// WHY OH WHY is there startsWith but no endsWith in QString in Qt 2.3.x ?!?!
			if (location.right(3) == "/**" || location == "**")
			{
				// recursive scanning of location...
				location = resolveLocation(playlistDirectory, location.left(qMax(0, location.length() - 3)));
				QExtStringList fileList;
				fillFileList(fileList, location, true);
				qSort(fileList);
				expandedList += fileList;
			}
			else if (location.right(2) == "/*" || location.right(1) == "/")
			{
				// all files of the location...
				location = resolveLocation(playlistDirectory, location.left(location.lastIndexOf('/')));
				QExtStringList fileList;
				fillFileList(fileList, location, false);
				qSort(fileList);
				expandedList += fileList;
			}
			else if (location.startsWith("http://") || location.startsWith("mms://"))
				expandedList.append(location);
			else
			{
				location = resolveLocation(playlistDirectory, location);
				QFileInfo fi(location);

				if (fi.exists())
				{
					if (fi.isDir())
					{
						QExtStringList fileList;
						fillFileList(fileList, location, false);
						qSort(fileList);
						expandedList += fileList;
					}
					else if (qConfig.isValidExtension(fi.suffix()))
						// just add the (file) location to the expanded list...
						expandedList.append(location);
				}
			}
		}

		list = expandedList;
	}

	// start reading the playlist entries...
	int numFiles = list.count();
	bool cancel = false;

	QTime time;
	time.start();

	emit updateActivityProgress(tr("Adding files..."), 0, numFiles, cancel);

	beginUpdate();
	// temporarily unblock signal emission...
	blockSignals(false);

	int index = getNextIndex();
	int i = 0;

	sqlite3_stmt *vm = stmtInsertMediaIDIntoPlaylist;

	for (QExtStringList::ConstIterator it = list.begin(); it != list.end(); ++it, ++i)
	{
		const QExtString &location = *it;

		unsigned long mediaID = NULL;

		if (location.id())
		{
			DHPRINTF("ID %d set on %s", location.id(), (const char *)location.toUtf8());
			mediaID = location.id();
		}
		else
		{
			// skip comments
			if (location.startsWith("#") || location.startsWith(";"))
				continue;

			DHPRINTF("No ID set on %s", (const char *)location.toUtf8());

			QString resolvedLoc;

			if (!(isPathAbsolute(location) || location.startsWith("http://") || location.startsWith("mms://")))
				resolvedLoc = resolveLocation(playlistDirectory, location);
			else
				resolvedLoc = location;

			mediaID = mediaDatabase_->getMediaIDForLocation(MediaLocation(resolvedLoc));
		}

		sqlite3_bind_int(vm, 1, index++);
		sqlite3_bind_null(vm, 2);
		sqlite3_bind_int(vm, 3, mediaID);

		sqlite3_step(vm);

		sqlite3_reset(vm);

		if (time.elapsed() > 1000)
		{
			if (location.length() > 55)
				emit updateActivityProgress(location.left(20) + "..." + location.right(40), i, numFiles, cancel);
			else
				emit updateActivityProgress(location, i, numFiles, cancel);

			time.restart();
		}

		if (cancel)
			break;
	}

	DTIMERPRINT(timer, "playlist added to DB in");

	emit finishedActivity(tr("Scanning for files..."), cancel);

	// disable signal emission again...
	blockSignals(true);
	endUpdate();

	DMEMSTAT();

	DEXITMETHOD;
}

bool PlayList::loadFromFile(const QString &filename)
{
	DENTERMETHOD;

	DTIMERINIT(timer);

	bool result = false;

	// Load PlayListView
	QFile file(filename);

#ifdef QT4
	if (file.open(QIODevice::ReadOnly))
	{
		QString playlistDirectory = QFileInfo(filename).absolutePath();
#else
	if (file.open(IO_ReadOnly))
	{
		QString playlistDirectory = QFileInfo(filename).dirPath(true);
#endif

		// Load PlayListView
		QTextStream stream(&file);

		QExtStringList list;
		QString line;

		while (!(line = stream.readLine()).isNull())
		{
			if (containsUTF8(line))
			{
				DHPRINTF("UTF8 detected: %s", (const char *)line.toLatin1());
				line = reinterpretUTF8(line);
			}
			else
				DHPRINTF("no UTF8: %s", (const char *)line.toLatin1());

			if (!line.startsWith("#") && !line.startsWith(";") && !line.isEmpty())
				line = resolveLocation(playlistDirectory, line);

			list.append(line);
		}

		// Are we dealing with a dynamic playlist here?
		// If so, we need to expand various entries in that list...
		bool interpretPlayList = filename.toLower().right(12) == ".dynplaylist";
		if (interpretPlayList)
		{
			type_ = Dynamic;
			dynPlayList_ = list;
		}
		else
			type_ = Static;

		addFromFileList(list, interpretPlayList, playlistDirectory);

		file.close();

		playListFileName_ = filename;
		result = true;
	}

	DTIMERPRINT(timer, "playlist read in");
	DEXITMETHOD;

	return result;
}

const QString PlayList::saveToFile()
{
	DENTERMETHOD;

	QString retval;

	if (!playListFileName_.isEmpty())
		retval = exportToFile(playListFileName_);
	else
		retval = QString::null;

	DEXITMETHOD;
	return retval;
}

const QString PlayList::saveToFile(const QString &filename)
{
	DENTERMETHODF("[%s]", (const char *)filename.toUtf8());
	playListFileName_ = exportToFile(filename);
	DEXITMETHODF("[%s]", (const char *)filename.toUtf8());
	return playListFileName_;
}

void PlayList::storeItemList(PlayListItemList &itemList)
{
	DENTERMETHOD;

	beginUpdate();
	clearTable();

	QString query = "INSERT INTO " + dbSafeString(sourceTable()) + " VALUES(?1, ?2, ?3);";

	sqlite3_stmt *vm;
	sqlite3_prepare_v2(mediaDatabase_->db(), query.toUtf8(), -1, &vm, 0);

	DOP(mediaDatabase_->dbDebug(query));

	for (int i = 0; i < itemList.count(); ++i)
	{
		PlayListItem *item = itemList.at(i);

		if (!item->id())
		{
			// set new playlist ID if the playlist ID was previously NULL
			sqlite3_bind_int(vm, 1, i);
			sqlite3_bind_null(vm, 2);
			sqlite3_bind_int(vm, 3, item->mediaID());
			sqlite3_step(vm);
			item->setID(sqlite3_last_insert_rowid(mediaDatabase_->db()));
			DHPRINTF("new item id: %d", item->id());
		}
		else
		{
			DHPRINTF("item id: %d", item->id());
			sqlite3_bind_int(vm, 1, i);
			sqlite3_bind_int(vm, 2, item->id());
			sqlite3_bind_int(vm, 3, item->mediaID());
			sqlite3_step(vm);
			DHOP(mediaDatabase_->dbDebug("sqlite3_step(vm);"));
		}

		sqlite3_reset(vm);
	}

	sqlite3_finalize(vm);

	endUpdate();

	DEXITMETHOD;
}

const PlayListItemList &PlayList::itemList() const
{
	if (itemListRequiresUpdate_)
	{
		PlayList *self = const_cast<PlayList *>(this);
		self->updateItemList();
	}

	return items_;
}

void PlayList::updateItemList()
{
	QString query = "SELECT playlist_id, media_id FROM " + dbSafeString(sourceTable()) + ";";

	DTIMERINIT(time);
	sqlite3_stmt *vm;
	sqlite3_prepare_v2(mediaDatabase_->db(), query.toUtf8(), -1, &vm, 0);

	DOP(mediaDatabase_->dbDebug(query));

	DTIMERPRINT(time, "updateItemList query");

	DTIMERSTART(time);

	clearPlayListItemList(&items_);

	PlayListItem *item = NULL;

	while (sqlite3_step(vm) == SQLITE_ROW)
	{
		item = new PlayListItem(sqlite3_column_int(vm, 1), sqlite3_column_int(vm, 0), mediaDatabase_);
		items_.append(item);
	}

	DOP(mediaDatabase_->dbDebug("data"));

	DTIMERPRINT(time, "updateItemList t update");

	sqlite3_finalize(vm);

	itemListRequiresUpdate_ = false;
}

const QString PlayList::exportToFile(const QString &filename)
{
	DENTERMETHODF("[%s]", (const char *)filename.toUtf8());

	QString fullFilename = filename;
	QString extension = QFileInfo(fullFilename).suffix();

	if (isDynamic())
	{
		if (extension != "dynplaylist")
			fullFilename += ".dynplaylist";
	}
	else if (extension != "playlist")
		fullFilename += ".playlist";

	QString retval;

#ifdef QT4
	QString playListDir = QFileInfo(fullFilename).absolutePath();

	QFile file(fullFilename);
	if (file.open(QIODevice::WriteOnly))
	{
		QTextStream s(&file);
		s.setCodec(QTextCodec::codecForName("UTF-8"));
#else
	QString playListDir = QFileInfo(fullFilename).dirPath(true);

	QFile file(fullFilename);
	if (file.open(IO_WriteOnly))
	{
		QTextStream s(&file);
		s.setEncoding(QTextStream::UnicodeUTF8);
#endif

		if (isDynamic())
		{
			for (QExtStringList::Iterator it = dynPlayList_.begin(); it != dynPlayList_.end(); ++it )
				s << makeLocationRelative(playListDir, *it) << "\n";
		}
		else
		{
			QString query = "SELECT media_id FROM \"" + dbSafeString(dbTable_) + "\" ORDER BY idx, ASC;";

			sqlite3_stmt *vm;
			sqlite3_prepare_v2(mediaDatabase_->db(), query.toUtf8(), -1, &vm, 0);

			while (sqlite3_step(vm) == SQLITE_ROW)
			{
				Media *media = mediaDatabase_->media(sqlite3_column_int(vm, 0));
				if (media)
					s << makeLocationRelative(playListDir, media->location().toString()) << "\n";
			}

			sqlite3_finalize(vm);
		}

		// Close File
		file.close();

		retval = fullFilename;
	}
	else
		retval = QString::null;

	DEXITMETHODF("[%s]", (const char *)filename.toUtf8());

	return retval;
}

void PlayList::loadFromDatabase()
{
	DENTERMETHOD;

	QStringList values;
	mediaDatabase_->loadState(dbTable_, values);

	if (values.count() == 3)
	{
		playListFileName_ = qGlobalConfig.resolveRelativeFilePath(values[0].mid(values[0].indexOf(":") + 1));

		QString playListDir;
		if (playListFileName_.isEmpty())
			playListDir = qGlobalConfig.appPath();
		else
			playListDir = QFileInfo(playListFileName_).dirPath(true);

		type_ = values[1].mid(values[1].indexOf(":") + 1) == "dynamic" ? Dynamic : Static;

		QString dynPlayListSerialized = values[2].mid(values[2].indexOf(":") + 1);
#ifdef QT4
		QStringList lines(dynPlayListSerialized.split('\n', QString::SkipEmptyParts));
#else
		QStringList lines(QStringList::split('\n', dynPlayListSerialized));
#endif
		dynPlayList_.clear();
		for (QStringList::ConstIterator it = lines.begin(); it != lines.end(); ++it)
			dynPlayList_.append(resolveLocation(playListDir, *it));
	}

	DEXITMETHOD;
}

void PlayList::saveToDatabase()
{
	DENTERMETHOD;

	QStringList values;

	QString playListDir;
	if (playListFileName_.isEmpty())
		playListDir = qGlobalConfig.appPath();
	else
		playListDir = QFileInfo(playListFileName_).dirPath(true);

	values.append("filename:" + qGlobalConfig.getRelativeFilePath(playListFileName_));
	values.append("type:" + (type_ == Dynamic ? QString("dynamic") : QString("static")));

	QString dynPlayListSerialized = "";
	for (QExtStringList::Iterator it = dynPlayList_.begin(); it != dynPlayList_.end(); ++it )
		dynPlayListSerialized += makeLocationRelative(playListDir, *it) + "\n";

	values.append("dynplaylist:" + dynPlayListSerialized);

	mediaDatabase_->saveState(dbTable_, values);

	DEXITMETHOD;
}
