/*
 * playlist.h
 *
 *  Created on: 18.11.2009
 *      Author: darkstar
 */

#ifndef PLAYLIST_H_
#define PLAYLIST_H_

#include <qobject.h>

#ifdef QT4
#include <QList>
#else
#include <qvaluelist.h>
#endif

#include "mediadatabase.h"
#include "media.h"

class PlayListItem: public MediaListener
{
public:
	PlayListItem(unsigned long mediaID, unsigned long playlistID, MediaDatabase *mediaDatabase)
		: mediaID_(mediaID), id_(playlistID), media_(NULL), mediaDatabase_(mediaDatabase) {}

	virtual ~PlayListItem() { if (media_) media_->unregisterListener(this); }

	Media *media() const { return media_ ? media_ : getAndSetMedia(); }

	inline unsigned long id() { return id_; }
	inline void setID(unsigned long newID) { id_ = newID; }

	inline unsigned long mediaID() { return mediaID_; }
	inline void setMediaID(unsigned long newID) { mediaID_ = newID; }

	inline MediaDatabase *mediaDatabase() { return mediaDatabase_; }
	inline void setMediaDatabase(MediaDatabase *mediaDatabase) { mediaDatabase_ = mediaDatabase; }

	void reset() { mediaID_ = id_ = 0; media_ = NULL; }

	PlayListItem & operator = (const PlayListItem & src)
	{
		if (&src != this)
		{
			mediaID_ = src.mediaID_;
			id_ = src.id_;
			media_ = NULL; // reset, because this item is not registered as MediaListener...
			mediaDatabase_ = src.mediaDatabase_;
		}
		return *this;
	}

protected:
	void mediaDestroyed(Media *) { media_ = NULL; }
	Media *getAndSetMedia() const;

private:
	unsigned long mediaID_;
	unsigned long id_;
	Media *media_;
	MediaDatabase *mediaDatabase_;
};

#ifdef QT4
typedef QList<unsigned long> MediaIDList;
typedef QList<unsigned long> PlayListIDList;
typedef QList<PlayListItem *> PlayListItemList;
#else
typedef QValueList<unsigned long> MediaIDList;
typedef QValueList<unsigned long> PlayListIDList;
typedef QList<PlayListItem> PlayListItemList;
#endif

void clearPlayListItemList(PlayListItemList *items);

class PlayList : public QObject
{
	Q_OBJECT
public:
	PlayList(MediaDatabase *mediaDatabase, const QString dbTable);
	virtual ~PlayList();

	MediaDatabase *mediaDatabase() { return mediaDatabase_; }
	const QString &sourceTable() const { return dbTable_; }

	void beginUpdate();
	void endUpdate();
	void propagateChanges();

	void reset();      // clears the content and resets the filename
	void clear();      // clears the content (table and dyn playlist)
	void clearTable(); // clears the table content only

	void addFile(const QString &fileName);
	void addDir(const QString &path, const bool subdir);
	void addURL(const QString &url);

	void fillFileList(QExtStringList &filelist, const QString &path, const bool subdir);
	void addFromFileList(const QExtStringList &filelist, bool interpretPlayList, const QString &playlistDirectory = QString::null);

	void appendMediaIDs(const MediaIDList &mediaIDs, PlayListItemList *outItemList = NULL);
	int getNextIndex() const;

	void deletePlayListIDs(const PlayListIDList &playListIDs);

	const PlayListItemList &itemList() const;
	void storeItemList(PlayListItemList &itemList);

	enum Type {
		Static = 0,
		Dynamic = 1
	};

	void setType(Type type);
	Type type() const { return type_; }

	bool isDynamic() { return type_ == Dynamic; }
	const QExtStringList &dynPlayList() { return dynPlayList_; }
	void setDynPlayList(const QExtStringList &dynPlayList);

	void repopulate();

	const QString fileName() const { return playListFileName_; }

	bool loadFromFile(const QString &filename);

	const QString saveToFile();
	const QString saveToFile(const QString &filename);
	const QString exportToFile(const QString &filename);

	void loadFromDatabase();
	void saveToDatabase();

signals:
	void playListChanged();

	void startingActivity(const QString &activityName);
	void updateActivityProgress(const QString &text, int stepNo, int stepsTotal, bool &cancel);
	void finishedActivity(const QString &activityName, bool canceled);

private slots:
	void mediaDatabasePreparingStatements(sqlite3 *db);
	void mediaDatabaseFinalizingStatements(sqlite3 *db);

private:
	MediaDatabase *mediaDatabase_;
	QString dbTable_;

	sqlite3_stmt *stmtSelectTotalFileSizeAndCountForMediaLocation;
	sqlite3_stmt *stmtSelectAllMediaIDsForMediaLocation;
	sqlite3_stmt *stmtInsertMediaIDIntoPlaylist;

	Type type_;
	QString playListFileName_;

	QExtStringList dynPlayList_;

	int updateCount_;

	void requireRepopulation();

	bool itemListRequiresUpdate_;
	PlayListItemList items_;

	void updateItemList();
};

#endif /* PLAYLIST_H_ */
