/*
 * playlistaspect.h
 *
 *  Created on: 15.04.2010
 *      Author: darkstar
 */

#include "playlistaspect.h"
#include "sqlite3.h"
#include "debug.h"
#include "compathack.h"

/* PlayListAspectExtension */

void PlayListAspectExtension::setPlayListAspect(PlayListAspect *playListAspect)
{
	if (playListAspect == playListAspect_) return;

	if (playListAspect_)
		removedFromAspect();

	playListAspect_ = playListAspect;

	if (playListAspect)
		attachedToAspect();
}

void PlayListAspectExtension::update()
{
	bool shallStop = false;
	playListAspect_->runPreCondition(shallStop);
	if (shallStop) return;

	execute();

	continueWithNextAspectExtension();
}

void PlayListAspectExtension::continueWithNextAspectExtension()
{
	DENTERMETHOD;

	PlayListAspectExtension *next = nextAspectExtension();

	if (next)
		next->update();
	else
		playListAspect_->runPostCondition();

	DEXITMETHOD;
}

void PlayListAspectExtension::reset()
{
	// descendants can do work here...
}

bool PlayListAspectExtension::isActive()
{
	return true;
}

bool PlayListAspectExtension::isFiltering()
{
	return false;
}

bool PlayListAspectExtension::isGraphFiltered()
{
	PlayListAspectExtension *prev = previousAspectExtension();
	return isFiltering() && prev && prev->isGraphFiltered();
}

void PlayListAspectExtension::saveExtensionState(QStringList &dst)
{
	// descendants can do work here...
}

void PlayListAspectExtension::loadExtensionState(QStringList &src)
{
	// descendants can do work here...
}

QString PlayListAspectExtension::inputViewSource()
{
	PlayListAspectExtension *prev = previousAspectExtension();

	if (prev)
		return prev->outputViewSource();
	else
		return QString::null;
}

PlayListAspectExtension *PlayListAspectExtension::previousAspectExtension()
{
#ifdef QT4
	int index = extensionList_->indexOf(this);

	if (index > -1 && this != extensionList_->first())
		return extensionList_->at(index - 1);
	else
		return NULL;
#else
	int index = extensionList_->findRef(this);

	if (index > -1 && this != extensionList_->getFirst())
		return extensionList_->at(index - 1);
	else
		return NULL;
#endif
}

PlayListAspectExtension *PlayListAspectExtension::nextAspectExtension()
{
#ifdef QT4
	int index = extensionList_->indexOf(this);

	if (index > -1 && this != extensionList_->last())
		return extensionList_->at(index + 1);
	else
		return NULL;
#else
	int index = extensionList_->findRef(this);

	if (index > -1 && this != extensionList_->getLast())
		return extensionList_->at(index + 1);
	else
		return NULL;
#endif
}

void PlayListAspectExtension::extensionGraphExecutionStarted(bool &stopIt)
{
	// descendants can do work here...
}

void PlayListAspectExtension::extensionGraphExecutionFinished()
{
	// descendants can do work here...
}

class PlayListAspectFilterExtensionStart: public PlayListAspectExtension
{
public:
	virtual const QString outputViewSource()
	{
		return playListAspect()->playList()->sourceTable();
	}

	virtual const QString identifier()
	{
		return playListAspect()->prefix() + "start";
	}

protected:
	virtual void execute() { /* DOH */ }
};

/* PlayListAspect */

PlayListAspect::PlayListAspect(PlayList *playList, const QString aspectName)
	:	QObject(),
	 	playList_(playList),
	 	aspectName_(aspectName),
	 	prefix_(playList->sourceTable() + "_" + aspectName + "_"),
	 	updateCount_(0),
	 	preConditionFired_(false)
{
	filterExtensionStart_ = new PlayListAspectFilterExtensionStart();
	registerExtension(filterExtensionStart_);

	connect(playList, SIGNAL(playListChanged()), this, SLOT(update()));
}

PlayListAspect::~PlayListAspect()
{
	unregisterExtension(filterExtensionStart_);
	delete filterExtensionStart_;
	filterExtensionStart_ = NULL;
}

void PlayListAspect::update()
{
	DENTERMETHOD;
	filterExtensionStart_->update();
	DEXITMETHOD;
}

void PlayListAspect::beginUpdate()
{
	DENTERMETHOD;
	++updateCount_;
	DEXITMETHOD;
}

void PlayListAspect::endUpdate()
{
	DENTERMETHOD;
	--updateCount_;
	DEXITMETHOD;
}

void PlayListAspect::runPreCondition(bool &stopExecution)
{
	if (updateCount_ != 0)
	{
		stopExecution = true;
		return;
	}

	stopExecution = false;

	if (!preConditionFired_)
	{
		preConditionFired_ = true;

		for (int i = 0; i < extensions_.count(); ++i)
			extensions_.at(i)->extensionGraphExecutionStarted(stopExecution);
	}
}

void PlayListAspect::runPostCondition()
{
	if (updateCount_ != 0)
		return;

	preConditionFired_ = false;

	for (int i = 0; i < extensions_.count(); ++i)
		extensions_.at(i)->extensionGraphExecutionFinished();
}

void PlayListAspect::registerExtension(PlayListAspectExtension *extension)
{
	DENTERMETHODF("[%p]", extension);
	extensions_.append(extension);
	extension->setExtensionList(&extensions_);
	extension->setPlayListAspect(this);
	DEXITMETHODF("[%p]", extension);
}

void PlayListAspect::unregisterExtension(PlayListAspectExtension *extension)
{
	DENTERMETHODF("[%p]", extension);
	extension->setPlayListAspect(NULL);
	extension->setExtensionList(NULL);
	extensions_.remove(extension);
	DEXITMETHODF("[%p]", extension);
}

void PlayListAspect::loadFromDatabase()
{
	QStringList state;
	beginUpdate();

	for (int i = 0; i < extensions_.count(); ++i)
	{
		PlayListAspectExtension* ext = extensions_.at(i);
		playList_->mediaDatabase()->loadState(ext->identifier(), state);
		ext->loadExtensionState(state);
		state.clear();
	}

	endUpdate();
	update();
}

void PlayListAspect::saveToDatabase()
{
	DENTERMETHOD;
	DTIMERINIT(timer);

	QStringList state;
	for (int i = 0; i < extensions_.count(); ++i)
	{
		PlayListAspectExtension* ext = extensions_.at(i);
		DPRINTF("Saving extension %p, identifier: %s", ext, (const char*)ext->identifier().toUtf8());
		ext->saveExtensionState(state);
		playList_->mediaDatabase()->saveState(ext->identifier(), state);
		state.clear();
	}

	DTIMERPRINT(timer, "saveToDatabase");
	DEXITMETHOD;
}
