/*
 * Copyright (C) 2006-2007	Andre Beckedorf
 * 					 		<evilJazz _AT_ katastrophos _DOT_ net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <qapplication.h>
#include <qtextstream.h>
#include <qstringlist.h>
#include <qtextcodec.h>
#include <qmessagebox.h>
#include <qregexp.h>

#include <unistd.h>

#ifndef QT4
#include <qtl.h>
#endif

#ifdef QT4
#include <QDesktopWidget>
#include <QProgressDialog>
#include <QTextCodec>
#include <classcompathack.h>
#else
#include <qwidgetstack.h>
#include "qprogressdialog.h"
#endif

#include "simplefiledialog.h"
#include "qinputdialog.h"
#include "configuration.h"
#include "playlistmanager.h"
#include "playbackcontroller.h"
#include "playlistsearchfilterlineedit.h"
#include "playlistview.h"
#include "playlistoverview.h"
#include "coverartflow.h"
#include "helpers.h"
#include "media.h"
#include "mediadatabase.h"
#include "mediaidentifier.h"
#include "sqlite3.h"

#include "playlist.h"
#include "playlistaspect.h"
#include "playlistaspectlistviewrenderer.h"
#include "playlistaspectoverviewfilter.h"
#include "playlistaspectsearchfilter.h"
#include "playlistaspectstatisticsgatherer.h"
#include "playlistaspectalbumgroupbuilder.h"

#include "debug.h"

#include "compathack.h"

PlayListManager::PlayListManager(PlayListView *targetPlayList, PlaybackController *playbackController, QWidgetStack *container, Configuration &config, QWidget *owner)
	:	QObject(),
		owner_(owner),
		playbackController_(playbackController),
		searchFilterLineEdit_(NULL),
		playListContainer_(container),
		config_(config),
		playListView_(targetPlayList),
		playListOverview_(NULL),
		coverArtFlow_(NULL),
		lboxDynPlayList_(NULL),
		previouslyVisibleWidget_(NULL),
		activeMode_(PlayListManager::ModeMax),
		progress_(NULL),
		progressUpdateTime_(),
		currentPlayList_(NULL),
		currentAspectComp_(NULL)
{
	// Create media database
	DPRINTF("Initialize MediaCache class...");
	mediaDatabase_ = new MediaDatabase(qGlobalConfig.databaseFilename());
	mediaDatabase_->load();
	DPRINTF("Done");

	// Create main play list
	playList_ = new PlayList(mediaDatabase_, "playlist");
	connect(playList_, SIGNAL(startingActivity(const QString &)), this, SLOT(playListStartingActivity(const QString &)));
	connect(playList_, SIGNAL(updateActivityProgress(const QString &, int, int, bool &)), this, SLOT(playListUpdateActivityProgress(const QString &, int, int, bool &)));
	connect(playList_, SIGNAL(finishedActivity(const QString &, bool)), this, SLOT(playListFinishedActivity(const QString &, bool)));

	initializeAspectCompound(&playListAspectComp_, playList_);

	// Create OTG play list
	otgPlayList_ = new PlayList(mediaDatabase_, "onthego_playlist");
	connect(otgPlayList_, SIGNAL(startingActivity(const QString &)), this, SLOT(playListStartingActivity(const QString &)));
	connect(otgPlayList_, SIGNAL(updateActivityProgress(const QString &, int, int, bool &)), this, SLOT(playListUpdateActivityProgress(const QString &, int, int, bool &)));
	connect(otgPlayList_, SIGNAL(finishedActivity(const QString &, bool)), this, SLOT(playListFinishedActivity(const QString &, bool)));

	initializeAspectCompound(&otgPlayListAspectComp_, otgPlayList_);

	if (qConfig.restorePreviousSessionOnRestartEnabled)
		loadStates();
	else
	{
		playList_->reset();
		playListAspectComp_.aspect->update();
		otgPlayListAspectComp_.aspect->update();
	}

	// Set current start play list
	currentPlayList_ = playList_;
	currentAspectComp_ = &playListAspectComp_;
}

PlayListManager::~PlayListManager()
{
	deinitializeAspectCompound(&otgPlayListAspectComp_);
	deinitializeAspectCompound(&playListAspectComp_);
}

void PlayListManager::initializeAspectCompound(PlayListAspectCompound *comp, PlayList *playList)
{
	DENTERMETHOD;
	comp->aspect = new PlayListAspect(playList, "aspect");

	comp->searchFilter = new PlayListAspectSearchFilter();
	comp->overviewFilter = new PlayListAspectOverviewFilter();
	comp->statistics = new PlayListAspectStatisticsGatherer();
	comp->listViewRenderer = new PlayListAspectListViewRenderer();
	comp->groupBuilder = new PlayListAspectAlbumGroupBuilder();

	comp->aspect->registerExtension(comp->searchFilter);
	comp->aspect->registerExtension(comp->overviewFilter);
	comp->aspect->registerExtension(comp->statistics);
	comp->aspect->registerExtension(comp->listViewRenderer);
	comp->aspect->registerExtension(comp->groupBuilder);
	DEXITMETHOD;
}

void PlayListManager::deinitializeAspectCompound(PlayListAspectCompound *comp)
{
	DENTERMETHOD;
	comp->aspect->unregisterExtension(comp->searchFilter);
	comp->aspect->unregisterExtension(comp->overviewFilter);
	comp->aspect->unregisterExtension(comp->statistics);
	comp->aspect->unregisterExtension(comp->listViewRenderer);
	comp->aspect->unregisterExtension(comp->groupBuilder);

	delete comp->searchFilter;     comp->searchFilter = NULL;
	delete comp->overviewFilter;   comp->overviewFilter = NULL;
	delete comp->statistics;       comp->statistics = NULL;
	delete comp->listViewRenderer; comp->listViewRenderer = NULL;
	delete comp->groupBuilder;     comp->groupBuilder = NULL;

	delete comp->aspect; comp->aspect = NULL;
	DEXITMETHOD;
}

void PlayListManager::loadStates()
{
	DENTERMETHOD;
	playList_->loadFromDatabase();
	playListAspectComp_.aspect->loadFromDatabase();

	otgPlayList_->loadFromDatabase();
	otgPlayListAspectComp_.aspect->loadFromDatabase();
	DEXITMETHOD;
}

void PlayListManager::saveStates()
{
	DENTERMETHOD;
	playListView_->detach();

	playList_->saveToDatabase();
	playListAspectComp_.aspect->saveToDatabase();

	otgPlayList_->saveToDatabase();
	otgPlayListAspectComp_.aspect->saveToDatabase();
	DEXITMETHOD;
}

QString PlayListManager::getDisplayPlayListFilename()
{
	if (isOnTheGoPlayListActive())
		return "On-The-Go Play List";
	else if (playList_->fileName().isNull() || playList_->fileName().isEmpty())
		return "Untitled." + (isDynamicPlaylist() ? QString("dynplaylist") : QString("playlist"));
	else
		return QFileInfo(playList_->fileName()).fileName();
}

void PlayListManager::execDialogAddFileToPlayList()
{
	addDialog(false);
}

void PlayListManager::execDialogAddDirToPlayList()
{
	addDialog(true);
}

void PlayListManager::execDialogAddDirWithSubDirsToPlayList()
{
	addDialog(true, true);
}

void PlayListManager::addDialog(const bool dir, const bool subdir)
{
	if (isDynamicPlaylist())
		setActiveMode(PlayListManager::ModeDynPlaylistEditor);

	if (!dir)
	{
		QString formatFilter = QString::null;

		for (QStringList::ConstIterator it = config_.validExtensions().begin(); it != config_.validExtensions().end(); ++it)
			formatFilter += "*." + (*it) + ";";

		formatFilter = formatFilter.left(formatFilter.length() - 1);

		QString selectedFilename = SimpleFileDialog::getOpenFileName(
			config_.lastDirectory,
			"All registered formats (" + formatFilter + ");;All files (*.*)",
			owner_,
			"add to play list dialog",
			tr("Add file to Playlist")
		);

		if (!selectedFilename.isNull() && !selectedFilename.isEmpty())
		{
			addFile(selectedFilename);

			// Update last directory
			config_.lastDirectory = QFileInfo(selectedFilename).filePath();
		}
	}
	else
	{
		QString selectedDirectory = SimpleFileDialog::getExistingDirectory(
			config_.lastDirectory,
			owner_,
			"add directory to playlist dialog",
			tr("Add directory to Playlist"),
			true,
			true
		);

		if (!selectedDirectory.isNull() && !selectedDirectory.isEmpty())
		{
			addDir(selectedDirectory, subdir);

			// Update last directory
			config_.lastDirectory = selectedDirectory;
		}
	}
}

bool PlayListManager::openASX(const QString &asx)
{
	QFile file(asx);

	// Parse ASX
#ifdef QT4
	if (file.open(QIODevice::ReadOnly))
	{
		QTextStream stream(&file);
		const QStringList list(stream.read().split('\n', QString::SkipEmptyParts));
#else
	if (file.open(IO_ReadOnly))
	{
		QTextStream stream(&file);
		const QStringList list = QStringList::split('\n', stream.read());
#endif
		for (QStringList::ConstIterator it = list.begin(); it != list.end(); ++it)
		{
			QString line(*it);
			QString uri;

#ifdef QT4
			if (line.contains("ref href", Qt::CaseInsensitive))
#else
			if (line.contains("ref href", false))
#endif
			{
				// String Indexes
				int uriBegin, uriEnd;

				// Find URI Index
#ifdef QT4
				uriBegin = line.indexOf("href", 0, Qt::CaseInsensitive);
#else
				uriBegin = line.indexOf("href", 0, false);
#endif
				uriBegin = line.indexOf('\"', uriBegin);
				++uriBegin;

				uriEnd = line.indexOf('\"', uriBegin);

				// Get URI
				uri = line.mid(uriBegin, uriEnd - uriBegin);

				// Add to PlayListView
				openURL(uri);
			}
		}

		file.close();

		return true;
	}
	else
		return false;
}

void PlayListManager::openURL(const QString &uri)
{
	bool ok;
	QString url;

	if (uri.isNull())
#ifdef QT4
		url = QInputDialog::getText(owner_, tr("URL to open..."), tr("URL"), QLineEdit::Normal, QString::null, &ok);
#else
		url = QInputDialog::getText(tr("URL to open..."), tr("URL"), QLineEdit::Normal, QString::null, &ok, owner_);
#endif
	else
		url = uri;

	if (ok && !url.isEmpty())
	{
		if (isDynamicPlaylistEditorActive())
			lboxDynPlayList_->insertItem(url);
		else
			currentPlayList_->addURL(url);
	}
}

void PlayListManager::addFile(const QString &path)
{
	if (isDynamicPlaylistEditorActive())
		lboxDynPlayList_->insertItem(path);
	else
		currentPlayList_->addFile(path);
}

void PlayListManager::addDir(const QString &path, const bool subdir)
{
	if (isDynamicPlaylistEditorActive())
	{
		QString location = path;

		if (location.right(1) != "/")
			location += "/";

		if (subdir)
			location += "**";

		lboxDynPlayList_->insertItem(location);
	}
	else
		currentPlayList_->addDir(path, subdir);
}

void PlayListManager::newPlayList()
{
	setActiveMode(PlayListManager::ModeOpenedPlaylist);

	clearPlayList();

	currentPlayList_->setType(PlayList::Static);

	emit playlistFileNameChanged();
}

void PlayListManager::newDynamicPlayList()
{
	setActiveMode(PlayListManager::ModeOpenedPlaylist);

	clearPlayList();

	currentPlayList_->setType(PlayList::Dynamic);

	checkInputMode();

	emit playlistFileNameChanged();

	//setActiveMode(PlayListManager::ModeDynPlaylistEditor);
}

void PlayListManager::clearPlayList()
{
	if (activeMode_ == ModeDynPlaylistEditor)
	{
		lboxDynPlayList_->clear();
	}

	currentPlayList_->clear();

	emit playlistCleared();

	DMEMSTAT();
}

void PlayListManager::savePlayList()
{
	if (currentPlayList_->fileName().isNull() ||
		currentPlayList_->fileName().isEmpty() ||
		activeMode_ == ModeOnTheGoPlaylist)
	{
		execDialogSavePlayListAs();
		return;
	}
	else
		writePlayList(currentPlayList_->fileName());
}

void PlayListManager::execDialogSavePlayListAs()
{
	QString newFilename = SimpleFileDialog::getSaveFileName(
		config_.lastPlaylistDirectory,
		(currentPlayList_->isDynamic() && activeMode_ != ModeOnTheGoPlaylist) ?
			"Dynamic playlist (*.dynplaylist)" :
			"Playlist files (*.playlist;*.m3u)",
		owner_,
		"save play list dialog",
		tr("Save Playlist Dialog")
	);

	if (!newFilename.isNull())
	{
		if (activeMode_ == ModeOnTheGoPlaylist)
			currentPlayList_->exportToFile(newFilename);
		else
			currentPlayList_->saveToFile(newFilename);

		emit playlistFileNameChanged();
	}
}

void PlayListManager::execDialogOpenPlayList()
{
	QString selectedFilename = SimpleFileDialog::getOpenFileName(
		config_.lastPlaylistDirectory,
		"All supported playlist formats (*.dynplaylist;*.playlist;*.m3u;*.asx);;"\
			"Quasar dynamic playlists (*.dynplaylist);;"\
			"Quasar playlists (*.playlist);;"\
			"M3U playlists (*.m3u);;"\
			"Windows Media Meta Files (*.asx)",
		owner_,
		"open play list dialog",
		tr("Open Playlist Dialog")
	);

	if (!selectedFilename.isNull())
	{
		if (activeMode_ == ModeOnTheGoPlaylist)
			clearPlayList();
		else
			newPlayList();

		readPlayList(selectedFilename);
		playListView_->setFocus();
	}
}


void PlayListManager::refreshDynamicPlayList()
{
	currentPlayList_->repopulate();
}

void PlayListManager::writePlayList(const QString &filename)
{
	QString outputFileName = currentPlayList_->saveToFile(filename);

	config_.lastPlaylistDirectory = QFileInfo(outputFileName).dirPath(true);

	emit playlistFileNameChanged();
}

void PlayListManager::playListStartingActivity(const QString &activityName)
{
	if (progress_)
		return;

	// initialize dialog and show it...
#ifdef QT4
	progress_ = new QProgressDialog(tr("Please wait..."), tr("Cancel"), 0, 0, owner_);
#else
	progress_ = new QProgressDialog(tr("Please wait..."), tr("Cancel"), 0, owner_, "progress", true);
	progress_->setFixedWidth(QMIN(qApp->desktop()->width() * 0.9, 640));
 	progress_->setAutoClose(false);
#endif

	progress_->setCaption(activityName);
	progress_->setMinimumDuration(1000);
	progress_->setAutoReset(false);
	progress_->show();
	qApp->processEvents();
	progressUpdateTime_.restart();
}

void PlayListManager::playListUpdateActivityProgress(const QString &text, int stepNo, int stepsTotal, bool &cancel)
{
	if (progress_ && progressUpdateTime_.elapsed() > 1000)
	{
		progress_->setLabelText(text);
#ifdef QT4
		progress_->setMaximum(stepsTotal);
		progress_->setValue(stepNo);
#else
		progress_->setTotalSteps(stepsTotal);
		progress_->setProgress(stepNo);
#endif
		qApp->processEvents();
#ifdef QT4
		cancel = progress_->wasCanceled();
#else
		cancel = progress_->wasCancelled();
#endif
		progressUpdateTime_.restart();
	}
}

void PlayListManager::playListFinishedActivity(const QString &activityName, bool canceled)
{
	if (!progress_)
		return;

	progress_->setCaption(activityName);

	if (!canceled)
#ifdef QT4
		progress_->setValue(progress_->maximum());
#else
		progress_->setProgress(progress_->totalSteps());
#endif

	qApp->processEvents();

	progress_->hide();
	qApp->processEvents();

	delete progress_;
	progress_ = NULL;
}

bool PlayListManager::readPlayList(const QString &filename, bool assumeInDB)
{
	DENTERMETHOD;
	DTIMERINIT(timer);

	// Load ASX
#ifdef QT4
	QRegExp asxExp("^[^\\.]+.asx$", Qt::CaseInsensitive);

	if (asxExp.indexIn(filename) != -1)
		return openASX(filename);
#else
	QRegExp asxExp("^[^\\.]+.asx$", false);

	if (asxExp.match(filename) != -1)
		return openASX(filename);
#endif

	bool result = false;

#ifdef QT4
	QString playlistDirectory = QFileInfo(filename).absolutePath();
#else
	QString playlistDirectory = QFileInfo(filename).dirPath(true);
#endif

	config_.lastPlaylistDirectory = playlistDirectory;

	if (assumeInDB)
		result = true;
	else
		result = currentPlayList_->loadFromFile(filename);

	if (result)
		emit playlistFileNameChanged();

	DTIMERPRINT(timer, "playlist read in");
	DEXITMETHOD;

	return result;
}

void PlayListManager::removeFileFromPlayList()
{
	if (isDynamicPlaylist())
	{
		if (isDynamicPlaylistEditorActive())
		{
			for (int i = lboxDynPlayList_->count() - 1; i > -1; --i)
				if (lboxDynPlayList_->isSelected(i))
					lboxDynPlayList_->removeItem(i);
		}
		else
		{
			int result = QMessageBox::information(owner_, tr("Dynamic playlist"),
					 tr("This playlist is dynamic, thus you can't delete\nan item from this playlist directly.\nDo you want to switch to the editor\nfor dynamic playlists and delete\nthe item there?"),
					 QMessageBox::Yes | QMessageBox::Default, QMessageBox::No | QMessageBox::Escape);

			if (result == QMessageBox::Yes)
				setActiveMode(PlayListManager::ModeDynPlaylistEditor);
		}
	}
	else
		playListView_->removeSelectedItems();
}

void PlayListManager::cutSelectionFromPlayList()
{
	mediaIDClipboard_ = playListView_->getSelectedMediaIDs();
	removeFileFromPlayList();
	emit clipboardChanged();
}

void PlayListManager::copySelectionFromPlayList()
{
	mediaIDClipboard_ = playListView_->getSelectedMediaIDs();
	emit clipboardChanged();
}

void PlayListManager::pasteIntoPlayList()
{
	if (!isDynamicPlaylist() && mediaIDClipboard_.count() > 0)
		playListView_->insertMediaIDsAtCurrentIndex(mediaIDClipboard_);
}

void PlayListManager::selectAll()
{
	if (isDynamicPlaylistEditorActive())
	{
		lboxDynPlayList_->selectAll(true);
		if (lboxDynPlayList_->currentItem() == -1 && lboxDynPlayList_->count() > 0)
			lboxDynPlayList_->setCurrentItem(0);
	}
	else
		playListView_->selectAll();
}

void PlayListManager::deselectAll()
{
	if (isDynamicPlaylistEditorActive())
	{
		lboxDynPlayList_->selectAll(false);
		lboxDynPlayList_->setCurrentItem(-1);
	}
	else
		playListView_->clearSelection();
}

void PlayListManager::copyToOnTheGoPlayList()
{
	if (playListView_->selectedItemsCount() > 0 && activeMode_ != PlayListManager::ModeOnTheGoPlaylist)
	{
		otgPlayList_->appendMediaIDs(playListView_->getSelectedMediaIDs());

		if (playListView_->selectedItemsCount() > 1)
			emit showMessage(tr("Items copied to On-The-Go Playlist."), 1500);
		else
			emit showMessage(tr("Item copied to On-The-Go Playlist."), 1500);
	}
}

bool PlayListManager::setActiveMode(Mode mode)
{
	DENTERMETHODF("%s", (mode == ModeDynPlaylistEditor ? "ModeDynPlaylistEditor" : (mode == ModeOpenedPlaylist ? "ModeOpenedPlaylist" : (mode == ModeOnTheGoPlaylist ? "ModeOnTheGoPlaylist" : "ModeLastActivePlaylist"))));

	bool result = false;

	if (mode == activeMode_)
		result = true;

	// if we are already in a playlist view and the last active playlist was requested,
	// do nothing and return positive.
	if (mode == PlayListManager::ModeLastActivePlaylist &&
		(activeMode_ == PlayListManager::ModeOnTheGoPlaylist || activeMode_ == PlayListManager::ModeOpenedPlaylist))
		result = true;

	emit switchingModeTo(mode);

	if (!result)
	{
		if (mode == PlayListManager::ModeOpenedPlaylist ||
			mode == PlayListManager::ModeOnTheGoPlaylist ||
			mode == PlayListManager::ModeLastActivePlaylist)
		{
			if (isDynamicPlaylist() && activeMode_ == PlayListManager::ModeDynPlaylistEditor)
			{
				QExtStringList newDynPlayList;

				for (int i = 0; i < lboxDynPlayList_->count(); i++)
					newDynPlayList.append(lboxDynPlayList_->text(i));

				// did anything change?
				if (!(newDynPlayList == playList_->dynPlayList()))
				{
					// if so, copy over
					playList_->setDynPlayList(newDynPlayList);
					// and refresh the real playlist...
					refreshDynamicPlayList();
				}

				// make previous widget visible...
				playListContainer_->raiseWidget(previouslyVisibleWidget_);
				previouslyVisibleWidget_ = NULL;

				delete lboxDynPlayList_;
				lboxDynPlayList_ = NULL;
			}

			// resolve last active play list and set mode accordingly...
			if (mode == PlayListManager::ModeLastActivePlaylist)
				mode = (currentPlayList_ == playList_ ? PlayListManager::ModeOpenedPlaylist : PlayListManager::ModeOnTheGoPlaylist);

			// Detach from aspect widgets...
			playListView_->detach();
			if (searchFilterLineEdit_) searchFilterLineEdit_->detach();
			if (playListOverview_) playListOverview_->detach();
			if (coverArtFlow_) coverArtFlow_->detach();
			currentAspectComp_->statistics->disconnect(this, SIGNAL(showInfoText(const QString &)));

			// Set references to current playlist and aspect compound...
			if (mode == PlayListManager::ModeOpenedPlaylist)
			{
				currentPlayList_ = playList_;
				currentAspectComp_ = &playListAspectComp_;
			}
			else if (mode == PlayListManager::ModeOnTheGoPlaylist)
			{
				currentPlayList_ = otgPlayList_;
				currentAspectComp_ = &otgPlayListAspectComp_;
			}

			// Attach to aspect widgets...
			connect(currentAspectComp_->statistics, SIGNAL(statisticsUpdated(const QString &)), this, SIGNAL(showInfoText(const QString &)));
			emit showInfoText(currentAspectComp_->statistics->displayText());

			if (searchFilterLineEdit_) searchFilterLineEdit_->attachTo(currentAspectComp_->searchFilter);
			if (playListOverview_) playListOverview_->attachTo(currentAspectComp_->overviewFilter);
			if (coverArtFlow_) coverArtFlow_->attachTo(currentAspectComp_->groupBuilder);

			playbackController_->attachTo(currentAspectComp_->listViewRenderer);
			playListView_->attachTo(currentAspectComp_->listViewRenderer, playbackController_);

			checkInputMode();

			result = true;
			activeMode_ = mode;
		}
		else if (mode == PlayListManager::ModeDynPlaylistEditor)
		{
			if (isDynamicPlaylist())
			{
				if (!lboxDynPlayList_)
				{
					lboxDynPlayList_ = new QListBox(playListContainer_);
					connect(lboxDynPlayList_, SIGNAL(selectionChanged()), this, SIGNAL(selectionChanged()));

				    for (QExtStringList::ConstIterator it = playList_->dynPlayList().begin(); it != playList_->dynPlayList().end(); ++it)
				    	lboxDynPlayList_->insertItem(new QListBoxText(*it), -1);

					// remember previously visible widget, so we can reselect it afterwards...
					previouslyVisibleWidget_ = playListContainer_->visibleWidget();
				}

				// show our editor...
				playListContainer_->raiseWidget(lboxDynPlayList_);

				result = true;
				activeMode_ = PlayListManager::ModeDynPlaylistEditor;
			}
			else
				result = false;
		}
	}

	if (result)
		emit switchedModeTo(mode);

	checkInputMode();

	DEXITMETHOD;
	return result;
}

void PlayListManager::checkInputMode()
{
	if (isDynamicPlaylist() && playListView_->inputMode() == PlayListView::Move)
		playListView_->setInputMode(PlayListView::Select);

	if (isDynamicPlaylistEditorActive())
	{
		switch (inputMode())
		{
		case PlayListView::Select:
			lboxDynPlayList_->selectAll(false);
			lboxDynPlayList_->setSelectionMode(QListBox::Single);
			if (lboxDynPlayList_->currentItem())
				lboxDynPlayList_->setSelected(lboxDynPlayList_->currentItem(), true);
			break;
		case PlayListView::MultiSelect:
		case PlayListView::Move:
			lboxDynPlayList_->setSelectionMode(QListBox::Extended);
			break;
		}
	}
}

PlayListView::InputMode PlayListManager::inputMode()
{
	return playListView_->inputMode();
}

void PlayListManager::setInputMode(PlayListView::InputMode inputMode)
{
	playListView_->setInputMode(inputMode);
	checkInputMode();
}
