/*
 * Copyright (C) 2007-2008 Andre Beckedorf <evilJazz _AT_ katastrophos _DOT_ net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include <qglobal.h>
#include <qpainter.h>
#include <qapplication.h>

#include "customflow.h"

#include "debug.h"

#ifndef QT4
#define qMax QMAX
#define qMin QMIN
#endif

CustomFlow::CustomFlow(LayeredPaintBox *parent)
	:	QObject(parent),
		PositionedLayer(parent)
{
	itemCount_ = 0;
	currentValue_ = 0;
	transitionValue_ = 0;
	slideSize_ = QSize(200, 200);
	distance_ = 10;
	minFlowValue_ = 0.5;
	minDistanceValue_ = 0;
	flowSpan_ = 1;
	distanceSpan_ = 1;
	scale_ = 1;

	orientation_ = Qt::Horizontal;

	timer_.setTimeBetweenSteps(0);
	timer_.setTransitionTime(500);
	connect(&timer_, SIGNAL(transitionStep(float, float, float, float)),
			this, SLOT(transitionStep(float, float, float, float)));

	connect(&timer_, SIGNAL(transitionStopped()),
			this, SLOT(transitionStopped()));

	connect(&timer_, SIGNAL(transitionFinished()),
			this, SLOT(transitionFinished()));
}

CustomFlow::~CustomFlow()
{

}

void CustomFlow::setSlideSize(QSize size)
{
	slideSize_ = size;
	invalidate();
}

void CustomFlow::setScale(float scale)
{
	scale_ = scale;
	invalidate();
}

void CustomFlow::setItemCount(int count)
{
	itemCount_ = count;
	if (currentValue_ > itemCount_ - 1)
		jumpTo(itemCount_ - 1);
	else
		invalidate();
}

void CustomFlow::setDistance(float distance)
{
	distance_ = distance;
	invalidate();
}

void CustomFlow::setMinFlowValue(float minVal)
{
	minFlowValue_ = minVal;
	invalidate();
}

void CustomFlow::setMinDistanceValue(float minVal)
{
	minDistanceValue_ = minVal;
	invalidate();
}

void CustomFlow::setFlowSpan(float span)
{
	flowSpan_ = span;
	invalidate();
}

void CustomFlow::setDistanceSpan(float span)
{
	distanceSpan_ = span;
	invalidate();
}

void CustomFlow::setCurrentValue(float value)
{
	currentValue_ = qMin((float)(itemCount_ - 1), qMax((float)0, value));
	emit currentValueChanged(currentValue_);

	transitionValue_ = currentValue_;
	emit transitionValueChanged(transitionValue_);

	timer_.setCurrentValue(currentValue_);
}

void CustomFlow::jumpTo(float value)
{
	stopTransition();
	setCurrentValue(value);
	invalidate();
}

void CustomFlow::transitionTo(float value)
{
	if (value == currentValue_)
		return;

	transitionValue_ = currentValue_;
	emit transitionValueChanged(transitionValue_);

	currentValue_ = qMin((float)(itemCount_ - 1), qMax((float)0, value));
	emit currentValueChanged(currentValue_);

	timer_.transitionTo(currentValue_);
}

void CustomFlow::stopTransition()
{
	timer_.stopTransition();
}

void CustomFlow::showPreviousItem()
{
	transitionTo(qMin((float)ceil(transitionValue_), currentValue_ - 1));
}

void CustomFlow::showNextItem()
{
	transitionTo(qMax((float)floor(transitionValue_), currentValue_ + 1));
}

void CustomFlow::showPreviousPage()
{
	transitionTo(qMin((float)ceil(transitionValue_), currentValue_ - elementCount()));
}

void CustomFlow::showNextPage()
{
	transitionTo(qMax((float)floor(transitionValue_), currentValue_ + elementCount()));
}

void CustomFlow::showIndex(int index)
{
	//qDebug("CustomFlow::showIndex(%d)", index);
	transitionTo(index);
}

void CustomFlow::setOrientation(Qt::Orientation orientation)
{
	orientation_ = orientation;
	invalidate();
}

float CustomFlow::flowValue(float value, float minValue, float width)
{
	if (value < -width)
		return minValue;
	else if (value > width)
		return minValue;
	else
		return ((cosf(value / width * M_PI) + 1) / 2) * (1 - minValue) + minValue;
}

float CustomFlow::distanceValue(float value, float minValue, float width)
{
	// for a symmetric distribution of the distance,
	// we need to duplicate the midpoint value to its predecessor (Midpoint - 1)
	if (value < 0 && value >= -1)
		return 1;

	// after the predecessor continue with the normal function...
	if (value < -1)
		value = value + 1; // shift value by one so as to make relative to zero here...

	if (value < -width)
		return minValue;
	else if (value > width)
		return minValue;
	else
		return (1 - fabs(value / width)) * (1 - minValue) + minValue;

	return minValue;
}

void CustomFlow::calcScaleValues(FloatArray &dst, int count, int midIndex, float shift, float minValue, float flowSpan, CustomFlow::ValueFuncPtr func)
{
	dst.resize(count);
	for (int i = 0; i < count; ++i)
	{
		dst[i] = (this->*func)(i - midIndex - shift, minValue, flowSpan);
		//qDebug("flowValues: dst[%d] : %f", i, dst[i]);
	}
}

void CustomFlow::calcWidthValues(FloatArray &dst, const FloatArray &scales, float maxWidth)
{
	dst.resize(scales.size());
	for (int i = 0; i < scales.size(); ++i)
	{
		dst[i] = scales[i] * maxWidth;
		//qDebug("widthValues: dst[%d] : %f", i, dst[i]);
	}
}

void CustomFlow::calcOffsetValues(FloatArray &dst, const FloatArray &widths, const FloatArray &distances)
{
	dst.resize(widths.size() + 1);

	float sum = 0;

	for (int i = 0; i < widths.size(); ++i)
	{
		dst[i] = sum;
		sum += widths[i] + distances[i];
		//qDebug("offsetValues: dst[%d] : %f", i, dst[i]);
	}

	dst[dst.size() - 1] = sum;
}

void CustomFlow::transitionStep(float start, float stop, float value, float step)
{
	transitionValue_ = value;
	emit transitionValueChanged(transitionValue_);

	invalidate();
}

void CustomFlow::transitionFinished()
{
	// nothing.
}

void CustomFlow::transitionStopped()
{
	// nothing.
}

/*
void CustomFlow::paintToBuffer()
{
	imageFillRectS(buffer(), 0, 0, width(), height(), QColor(0, 0, 0));

	for (int i = startIndex; i < stopIndex; ++i)
	{
		int midIndex = i - mid;  // Index relative to the center box. This can be negative
		int absIndex = midIndex + (int)transitionValue_; // absolute item index

		QRect boxRect(
			QPoint(
				midPoint.x() + (int)offsetValues_[i],
				midPoint.y() - 50 - (int)(widthScaleValues_[i] * (slideSize_.height() - 50))
			),
			QPoint(
				midPoint.x() + (int)offsetValues_[i + 1] - distance_,
				midPoint.y() + 50 + (int)(widthScaleValues_[i] * (slideSize_.height() - 50))
			)
		);

		imageFillRectS(buffer(), boxRect, QColor(255, 0, 0));
	}
}
*/

void CustomFlow::updateElementCountAndMidIndex()
{
	// Determine mid and length of arrays so that items are visible on screen
	// This is an educated guess, probably too much...
	// TODO: refine length guesstimate...
	int size, slideSize;

	if (orientation_ == Qt::Horizontal)
	{
		size = width();
		slideSize = slideSize_.width();
	}
	else
	{
		size = height();
		slideSize = slideSize_.height();
	}

	elementCount_ = (int)ceil(size / (slideSize * scale_ * minFlowValue_ + distance_ * scale_ * minDistanceValue_));

	// Make sure the length of arrays is at least 2 times bigger than
	// the biggest flow/distance span...
	int maxFlowWidth = ceil(qMax(flowSpan_ * 2, distanceSpan_ * 2));
	elementCount_ = qMax(maxFlowWidth, elementCount_);

	if (elementCount_ % 2 == 0)
		++elementCount_;

	midIndex_ = elementCount_ / 2;
}

void CustomFlow::updateValues()
{
	updateElementCountAndMidIndex();

	int slideSize = orientation_ == Qt::Horizontal ? slideSize_.width() : slideSize_.height();
	float shift = fabs(transitionValue_ - floor(transitionValue_));

	//qDebug("transitionValue: %f,  shift: %f", transitionValue_, shift);

	// TODO: combine functions into one so to get rid of the extra loops
	calcScaleValues(widthScaleValues_, elementCount_, midIndex_, shift, minFlowValue_, flowSpan_, &CustomFlow::flowValue);
	calcWidthValues(widthValues_, widthScaleValues_, slideSize * scale_);

	//qDebug("distance: %f, minDistanceValue: %f", distance_, minDistanceValue_);

	calcScaleValues(distanceScaleValues_, elementCount_, midIndex_, shift, minDistanceValue_, distanceSpan_, &CustomFlow::distanceValue);
	calcWidthValues(distanceValues_, distanceScaleValues_, distance_ * scale_);

	calcOffsetValues(offsetValues_, widthValues_, distanceValues_);

	// Make offsets relative to midpoint + shift
	float midPointShiftDiff =
		offsetValues_[offsetValues_.size() - 1] / 2 +
		(slideSize * scale_ * minFlowValue_ + distance_ * scale_ * minDistanceValue_) * shift -
		(distance_ * scale_ * minDistanceValue_) / 2;

	//qDebug("midIndex: %d", midIndex_);
	//qDebug("midPointShiftDiff: %f", midPointShiftDiff);

	for (int i = 0; i < offsetValues_.size(); ++i)
	{
		offsetValues_[i] -= midPointShiftDiff;
		//qDebug("offsetValues: dst[%d] : %f", i, offsetValues_[i]);
	}

	leftBound_ = qMax((float)(midIndex_ - floor(transitionValue_)), (float)0);
	rightBound_ = qMin((float)((itemCount_ - 1) + midIndex_ - floor(transitionValue_)), (float)(offsetValues_.size() - 2));

	//qDebug("leftBound_: %d", leftBound_);
	//qDebug("rightBound_: %d", rightBound_);
}

void CustomFlow::invalidate()
{
	updateValues();
	PositionedLayer::invalidate();
}
