#ifndef SCROLLBAR_H_
#define SCROLLBAR_H_

#include <qobject.h>
#include <qtimer.h>
#include <qrect.h>

#ifdef QT4
#include <QPixmap>
#include <QMouseEvent>
#endif

#include "paintbox.h"
#include "customflow.h"

class AbstractScrollBar : public QObject, public PositionedLayer
{
	Q_OBJECT
public:
	AbstractScrollBar(LayeredPaintBox *parent);

signals:
	void sliderReleased(float newValue);
	void sliderStepped(float newValue);
	void sliderMoved(float newValue);
	
protected:
	virtual int itemCount() = 0;
	virtual int itemsPerPage() = 0;
	virtual float position() = 0;
	virtual float displayPosition() { return position(); }
	virtual bool changePosition(float position) = 0;
	
	virtual int buttonSize() { return 16; };
	virtual int minSliderWidth() { return 30; }
	virtual int maxSliderWidth() { return INT_MAX; }
	
	virtual void resized();
	virtual void paintToBuffer(QImage &buffer, const QRect &clipRect);
	virtual void paintOverlay(QPainter &p, const QPixmap &pixmap);
	virtual void mousePressEvent(QMouseEvent *event);
	virtual void mouseMoveEvent(QMouseEvent *event);
	virtual void mouseReleaseEvent(QMouseEvent *event);

	const QRect &rectButtonLeft() { return rectButtonLeft_; }
	const QRect &rectButtonRight() { return rectButtonRight_; }
	const QRect &rectSlider() { return rectSlider_; }
	const QRect &rectSliderAreaLeft() { return rectSliderAreaLeft_; }
	const QRect &rectSliderAreaRight() { return rectSliderAreaRight_; }

protected slots:
	void scrollToNextItem();
	void scrollToPreviousItem();
	void scrollToNextPage();
	void scrollToPreviousPage();
	
private:
	QRect rectButtonLeft_;
	QRect rectButtonRight_;
	QRect rectSlider_;
	QRect rectSliderAreaLeft_;
	QRect rectSliderAreaRight_;
	QTimer *scrollTimer_;
	
	int sliderPressedOffset_;
	bool mouseDown_;
	
	void setCurrentValueFromScrollBarPosition(int x);
};

class ScrollBar : public AbstractScrollBar
{
	Q_OBJECT
public:
	ScrollBar(LayeredPaintBox *parent) : AbstractScrollBar(parent), itemCount_(0), itemsPerPage_(1), position_(0) {}

	float position() { return position_; }
	
	int itemCount() { return itemCount_; }
	void setItemCount(int count) { itemCount_ = count; invalidate(); }
	
	int itemsPerPage() { return itemsPerPage_; }
	void setItemsPerPage(int count) { itemsPerPage_ = count; invalidate(); }

public slots:	
	void setPosition(float position) { position_ = position; invalidate(); }

signals:
	void userChangedPosition(float newValue, float oldValue);

protected:
	bool changePosition(float position);
	
private:
	int itemCount_;
	int itemsPerPage_;
	float position_;
};

class FlowScrollBar : public AbstractScrollBar
{
	Q_OBJECT
public:
	FlowScrollBar(CustomFlow *flow, LayeredPaintBox *parent);
	virtual ~FlowScrollBar();

protected:
	virtual int itemCount() { return flow_->itemCount(); }
	virtual int itemsPerPage() { return flow_->elementCount() / 2; }
	virtual float displayPosition() { return flow_->transitionValue(); }
	virtual float position() { return flow_->currentValue(); }
	virtual bool changePosition(float position) { flow_->transitionTo(position); invalidate(); return true; }

private slots:
	void flowTransitionValueChanged(float value);
	void flowCurrentValueChanged(float value);
	
private:
	CustomFlow *flow_;
};

#endif /*SCROLLBAR_H_*/
