/*
 * Copyright (C) 2007 Andre Beckedorf <evilJazz _AT_ katastrophos _DOT_ net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <unistd.h>
#include <qthread.h>

#ifdef QT4
#include <QEvent>
#include <QCoreApplication>
#endif

#ifdef WINDOWS
#include "wincompat.h"
#endif

#include "backgroundtasks.h"
#include "debug.h"


class TaskProcessingThread : public QThread
{
public:
	TaskProcessingThread(TaskProcessingController *owner);
	virtual ~TaskProcessingThread();
	
	virtual void run();
	
	bool sleeping() { return sleeping_; }
	
	bool loopFinished() { return loopFinished_; }

	void terminate() { terminated_ = true; }
	Task *currentTask() { return currentTask_; }
	
protected:
	TaskProcessingController *owner_;
	Task *currentTask_;
	bool terminated_;
	bool sleeping_;
	bool loopFinished_;
};


class TaskFinishedEvent : public QEvent
{
public:
	TaskFinishedEvent(Task *task) : QEvent(QEvent::User), task_(task) { }
	Task* task() { return task_; }
private:
	Task *task_;
};


/* Task */

Task::Task()
	:	aborted_(false),
		finished_(false)
{
	
}

Task::~Task()
{
}


/* TaskProcessingThread */

TaskProcessingThread::TaskProcessingThread(TaskProcessingController *owner)
	:	QThread(),
		owner_(owner),
		currentTask_(NULL),
		terminated_(false),
		sleeping_(false),
		loopFinished_(true)
{
	
}

TaskProcessingThread::~TaskProcessingThread()
{

}

void TaskProcessingThread::run()
{
	DENTERMETHOD;
	// We are using our own crappy finished indicator
	// because Qt 3's QThread implementation is quirky on some OSs.
	loopFinished_ = false;

	while (!terminated_)
	{
		currentTask_ = owner_->getNextTask();
		
		if (currentTask_)
			DPRINTF("thread: %p -> task: %p", currentThread(), currentTask_);
		
		if (currentTask_)
		{
			if (!currentTask_->aborted())
			{
				currentTask_->run();
				
				if (!currentTask_->aborted() && !terminated_)
				{
					currentTask_->finished_ = true;
#ifdef QT4
					QCoreApplication::postEvent(owner_, new TaskFinishedEvent(currentTask_));
#else
					postEvent(owner_, new TaskFinishedEvent(currentTask_));
#endif
					currentTask_ = NULL;
				}
				else
				{
					currentTask_->finished_ = true;
					currentTask_->notifyTaskFinished();
					delete currentTask_;
					currentTask_ = NULL;
				}
			}
			else
			{
				currentTask_->finished_ = true;
				currentTask_->notifyTaskFinished();
				delete currentTask_;
				currentTask_ = NULL;
			}
			
			usleep(1);
		}
		else
		{
			DPRINTF("thread: %p -> nothing to be done, Zzz...", currentThread());
			sleeping_ = true;
#ifdef WINDOWS			
			usleep(100000);
#endif			
#ifdef QT4
			owner_->conditionWorkAvailable_.wait(&owner_->waitMutex_);
#else
			owner_->conditionWorkAvailable_.wait();
#endif
			sleeping_ = false;
			DPRINTF("thread: %p -> nugded, woke up...", currentThread());
		}
	}
	DPRINTF("thread %p died.", currentThread());

	loopFinished_ = true;
	DEXITMETHOD;
}


/* TaskProcessingController */

TaskProcessingController::TaskProcessingController(QObject *parent)
	:	QObject(parent),
#ifdef QT4
	 	mutexTaskQueue_(QMutex::Recursive),
		mutexLowPrioTaskQueue_(QMutex::Recursive),
		waitMutex_(QMutex::NonRecursive),
#else
	 	mutexTaskQueue_(true),
		mutexLowPrioTaskQueue_(true),
		waitMutex_(false),
#endif
		conditionWorkAvailable_(),
		threads_(),
		taskQueue_(),
		lowPrioTaskQueue_()
{
#ifndef QT4
	threads_.setAutoDelete(true);
	taskQueue_.setAutoDelete(true);
	lowPrioTaskQueue_.setAutoDelete(true);
#endif
	
	// only two threads for now...
	TaskProcessingThread *thread = new TaskProcessingThread(this); 
	threads_.append(thread);
	thread->start();

#ifndef QTOPIA
	thread = new TaskProcessingThread(this);
	threads_.append(thread);
	thread->start();
#endif
}

TaskProcessingController::~TaskProcessingController()
{
	DENTERMETHOD;
	
	TaskProcessingThread *thread;
	
	// Signal termination...
	for (int i = 0; i < threads_.count(); ++i)
		threads_.at(i)->terminate();
	
	// Clear the queues...
	clear();
	
	// Signal sleeping threads to wake up and discover they are marked for termination...
	for (int i = 0; i < threads_.count(); ++i)
#ifdef QT4
		while (threads_.at(i)->isRunning())
#else
		while (!threads_.at(i)->loopFinished())
#endif
		{
			conditionWorkAvailable_.wakeAll();
			usleep(1);
		}
	
	// One last sanity waitfor...
	for (int i = 0; i < threads_.count(); ++i)
		threads_.at(i)->wait();
	
	// Now go die...
#ifdef QT4
	qDeleteAll(threads_);
#endif
	threads_.clear();
	
	DEXITMETHOD;
}

void TaskProcessingController::clear()
{
	lockQueues();	
	
	for (int i = taskQueue_.count() - 1; i == 0; --i)
	{
		taskQueue_.at(i)->notifyTaskFinished();
#ifdef QT4
		delete taskQueue_.at(i);
		taskQueue_.removeAt(i);
#else
		taskQueue_.remove(i); // autoDelete is enabled, so also free the task...
#endif
	}

	for (int i = lowPrioTaskQueue_.count() - 1; i == 0; --i)
	{
		lowPrioTaskQueue_.at(i)->notifyTaskFinished();
#ifdef QT4
		delete lowPrioTaskQueue_.at(i);
		lowPrioTaskQueue_.removeAt(i);
#else
		lowPrioTaskQueue_.remove(i); // autoDelete is enabled, so also free the task...
#endif
	}
	
	unlockQueues();
}

void TaskProcessingController::addTask(Task *task, TaskPriority prio, bool prepend)
{
	DPRINTF("Adding task %p", task);
	
	if (prio == TaskPriorityHigh)
	{
		mutexTaskQueue_.lock();
		if (prepend)
			taskQueue_.insert(0, task);
		else
			taskQueue_.append(task);
		mutexTaskQueue_.unlock();
	}
	else
	{
		mutexLowPrioTaskQueue_.lock();
		if (prepend)
			lowPrioTaskQueue_.insert(0, task);
		else
			lowPrioTaskQueue_.append(task);
		mutexLowPrioTaskQueue_.unlock();
	}
	
	conditionWorkAvailable_.wakeOne();
}

bool TaskProcessingController::removeTask(Task *task)
{
	lockQueues();
	
	DPRINTF("Removing task %p", task);

#ifdef QT4
	bool result = taskQueue_.removeOne(task);
#else
	bool result = taskQueue_.removeRef(task);
#endif
	
	if (!result)
#ifdef QT4
		result = lowPrioTaskQueue_.removeOne(task);
#else
		result = lowPrioTaskQueue_.removeRef(task);
#endif

	unlockQueues();	
	
	return result;
}

bool TaskProcessingController::taskEnqueued(Task *task)
{
	lockQueues();
	
#ifdef QT4
	bool result = taskQueue_.contains(task);
#else
	bool result = taskQueue_.containsRef(task);
#endif

	if (!result)
#ifdef QT4
		result = lowPrioTaskQueue_.contains(task);
#else
		result = lowPrioTaskQueue_.containsRef(task);
#endif

	unlockQueues();	
	
	return result;
}

bool TaskProcessingController::taskRunning(Task *task)
{
	lockQueues();
	
	bool result = false;
	
	for (int i = 0; i < threads_.count(); ++i)
		if (threads_.at(i)->currentTask() == task)
		{
			result = true;
			break;
		}
	
	unlockQueues();
	
	return result;
}

bool TaskProcessingController::rescheduleTask(Task *task, TaskPriority prio, bool prepend)
{
	lockQueues();

	bool result = false;

	if (taskEnqueued(task) && !taskRunning(task))
	{
#ifdef QT4
		lowPrioTaskQueue_.removeOne(task);
		taskQueue_.removeOne(task);
#else
		if (lowPrioTaskQueue_.findRef(task) > -1)
			lowPrioTaskQueue_.take();
		
		if (taskQueue_.findRef(task) > -1)
			taskQueue_.take();
#endif

		if (prio == TaskPriorityHigh)
		{
			if (prepend)
				taskQueue_.insert(0, task);
			else
				taskQueue_.append(task);
		}
		else if (prio == TaskPriorityLow)
		{
			if (prepend)
				lowPrioTaskQueue_.insert(0, task);
			else
				lowPrioTaskQueue_.append(task);
		}
		
		result = true;
	}
	
	unlockQueues();
	
	return result;
}

bool TaskProcessingController::idle()
{
	lockQueues();
	bool queuesEmpty = taskQueue_.count() == 0 && lowPrioTaskQueue_.count() == 0;
	bool threadsSleeping = false;
	
	if (queuesEmpty)
	{
		threadsSleeping = true;

		for (int i = 0; i < threads_.count(); ++i)
			if (threads_.at(i)->currentTask())
			{
				threadsSleeping = false;
				break;
			}
	}

	unlockQueues();
	
	DPRINTF("queuesEmpty: %d, threadsSleeping: %d", queuesEmpty == true ? 1 : 0, threadsSleeping == true ? 1 : 0);

	return queuesEmpty && threadsSleeping;
}

void TaskProcessingController::waitUntilFinished()
{
	while (!idle())
		usleep(10000); // sleep 10 ms
}

void TaskProcessingController::lockQueues()
{
	mutexTaskQueue_.lock();
	mutexLowPrioTaskQueue_.lock();	
}

void TaskProcessingController::unlockQueues()
{
	mutexLowPrioTaskQueue_.unlock();
	mutexTaskQueue_.unlock();
}

Task *TaskProcessingController::getNextTask()
{
	DPRINTF("Getting next task...");
	
	Task *result = NULL;
	
	mutexTaskQueue_.lock();
	
	if (!taskQueue_.isEmpty())
	{
		Task *currentTask = taskQueue_.first();
		
		while (!taskQueue_.isEmpty() && currentTask && currentTask->aborted())
		{
			currentTask->notifyTaskFinished();
#ifdef QT4
			delete taskQueue_.at(0);
			taskQueue_.removeAt(0);
#else
			taskQueue_.remove(); // autoDelete is enabled, so also free the task...
#endif
		}
		
		if (!taskQueue_.isEmpty())
#ifdef QT4
			result = taskQueue_.takeAt(0);
#else
			result = taskQueue_.take();
#endif
	}
	
	mutexTaskQueue_.unlock();
	
	if (!result)
	{
		mutexLowPrioTaskQueue_.lock();
		
		if (!lowPrioTaskQueue_.isEmpty())
		{
			Task *currentTask = lowPrioTaskQueue_.first();
			
			while (!lowPrioTaskQueue_.isEmpty() && currentTask && currentTask->aborted())
			{
				currentTask->notifyTaskFinished();
#ifdef QT4
				delete lowPrioTaskQueue_.at(0);
				lowPrioTaskQueue_.removeAt(0);
#else
				lowPrioTaskQueue_.remove(); // autoDelete is enabled, so also free the task...
#endif
			}
			
			if (!lowPrioTaskQueue_.isEmpty())
#ifdef QT4
				result = lowPrioTaskQueue_.takeAt(0);
#else
				result = lowPrioTaskQueue_.take();
#endif
		}
		
		mutexLowPrioTaskQueue_.unlock();
	}
	
	DPRINTF("Next task: %p", result);
	
	return result;
}

bool TaskProcessingController::event(QEvent *e)
{
	DPRINTF("TaskProcessingController::event(QEvent *e)");
	
	if (e->type() == QEvent::User)
	{
		TaskFinishedEvent *event = static_cast<TaskFinishedEvent *>(e);
		DPRINTF("Task %p finished.", event->task());
		event->task()->notifyTaskFinished();
		delete event->task();
		return true;
	}
	
	return false;
}
