/*
 * Copyright (C) 2007 Andre Beckedorf <evilJazz _AT_ katastrophos _DOT_ net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef BACKGROUNDTASKS_H_INCLUDED
#define BACKGROUNDTASKS_H_INCLUDED

#include <qobject.h>
#include <qimage.h>
#include <qevent.h>
#include <qthread.h>

#ifdef QT4
#include <QMutex>
#include <QWaitCondition>
#endif

class QImage;
class TaskProcessingController;
class TaskProcessingThread;
class CustomCachedImageProvider;

class Task
{
	friend class TaskProcessingThread;
public:
	Task();
	virtual ~Task();
	
	virtual void run() = 0;
	void abort() { aborted_ = true; }
	virtual void notifyTaskFinished() { finished_ = true; }
	
	bool finished() const { return finished_; }
	bool aborted() const { return aborted_; }
private:
	bool aborted_;
	bool finished_;
};

class TaskProcessingController : public QObject
{
	friend class TaskProcessingThread;
public:
	
	enum TaskPriority
	{
		TaskPriorityHigh,
		TaskPriorityLow
	};
	
	TaskProcessingController(QObject *parent = 0);
	virtual ~TaskProcessingController();
	
	void lockQueues();
	void unlockQueues();
	
	void addTask(Task *task, TaskPriority prio = TaskPriorityHigh, bool prepend = false);
	bool removeTask(Task *task);
	bool taskRunning(Task *task);
	bool taskEnqueued(Task *task);
	bool rescheduleTask(Task *task, TaskPriority prio = TaskPriorityHigh, bool prepend = false);

	bool idle();
	void waitUntilFinished();
	
	void clear();
	
protected:
	QMutex mutexTaskQueue_;
	QMutex mutexLowPrioTaskQueue_;
	
	QMutex waitMutex_;
	QWaitCondition conditionWorkAvailable_;
	
#ifdef QT4
	QList<TaskProcessingThread *> threads_;
	QList<Task *> taskQueue_;
	QList<Task *> lowPrioTaskQueue_;
#else
	QList<TaskProcessingThread> threads_;
	QList<Task> taskQueue_;
	QList<Task> lowPrioTaskQueue_;
#endif
		
	Task *getNextTask();
	
	virtual bool event(QEvent *e);
};

#endif /*BACKGROUNDTASKS_H_INCLUDED*/
