/*
 * Copyright (C) 2007-2008 Andre Beckedorf <evilJazz _AT_ katastrophos _DOT_ net>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "customscaledcachedimageprovider.h"
#include "debug.h"

#include <qfile.h>
#include <qbuffer.h>
#include <qdatetime.h>

#ifdef QT4
#include <QImageWriter>
#include <QImageReader>
#else
#endif

#include "compathack.h"

const int sizeScalesCount = 7;
const float sizeScales[sizeScalesCount] = { 1, 2.0/3, 1.0/2, 1.0/3, 1.0/4, 1.0/5, 1.0/6};


CustomScaledCachedImageProvider::CustomScaledCachedImageProvider(QObject *owner)
	:	CustomCachedImageProvider(owner),
		scale_(1),
		scaleSizeBlock_(0)
{
	
}

void CustomScaledCachedImageProvider::setScale(float scale)
{
	scale_ = scale;
	
	//qDebug("scale_: %f", scale_);
	
	int newScaleSizeBlock = scaleSizeBlock_;
	
	for (int i = sizeScalesCount - 1; i >= 0; --i)
	{
		//qDebug("sizeScales[%d]: %f", i, sizeScales[i]);
		if (sizeScales[i] > scale)
		{
			newScaleSizeBlock = i;
			break;
		}
	}

	//qDebug("scaleSizeBlock_: %d, newScaleSizeBlock: %d", scaleSizeBlock_, newScaleSizeBlock);	
	
	if (newScaleSizeBlock != scaleSizeBlock_)
	{
		scaleSizeBlock_ = newScaleSizeBlock;
		
		flushMemoryCache();
		emit scaleChangedAndFlushed();
	}
}

void CustomScaledCachedImageProvider::threadedSaveImageToCacheDirectory(CachedImage *cachedImage)
{
	DPRINTF("threadedSaveImageToCacheDirectory");

	//DTIMERINIT(savetimer);
	
	if (cachedImage)
	{
		if (cachedImage->image())
		{
			DPRINTF("Saving image '%s' to '%s'...", (const char *)(cachedImage->name().toUtf8()), (const char*)(cachedImage->cacheFileName().toUtf8()));
			//cachedImage->image()->save(cachedImage->cacheFileName(), "JPEG", 70);
			
			QFile file(cachedImage->cacheFileName());
#ifdef QT4
			file.open(QIODevice::WriteOnly);
			QImageWriter iio(&file, "JPEG");
#else
			file.open(IO_WriteOnly);
			QImageIO iio(&file, "JPEG");
#endif
		
			int cx = cachedImage->image()->width();
			int cy = cachedImage->image()->height();
			
			int offsets[sizeScalesCount];
		
			for (int i = 0; i < sizeScalesCount; ++i)
			{
				// increase JPEG quality for smaller images...
				int quality = 80 + 3 * i;
				
#ifdef QTOPIA
				iio.setParameters(QString::number(quality).latin1());
#else
				iio.setQuality(quality);
#endif

				QImage image =
					i == 0 ?
					*cachedImage->image() :
#ifdef QT4
					cachedImage->image()->scaled((int)((float)cx * sizeScales[i]), (int)((float)cy * sizeScales[i]), Qt::IgnoreAspectRatio, Qt::SmoothTransformation);
#else
					cachedImage->image()->smoothScale((int)((float)cx * sizeScales[i]), (int)((float)cy * sizeScales[i]));
#endif
				
#ifdef QT4
				iio.write(image);
				offsets[i] = file.pos();
#else
				iio.setImage(image);
				iio.write();
				offsets[i] = file.at();
#endif
			}

			// Write offsets as suffix...
#ifdef QT4
			file.write((const char *)offsets, sizeScalesCount * sizeof(int));
			file.write((const char *)&cx, sizeof(int));
			file.write((const char *)&cy, sizeof(int));
			file.write("CBCI", 4);
#else
			file.writeBlock((const char *)offsets, sizeScalesCount * sizeof(int));
			file.writeBlock((const char *)&cx, sizeof(int));
			file.writeBlock((const char *)&cy, sizeof(int));
			file.writeBlock("CBCI", 4);
#endif
			file.close();
		}
		else
			DPRINTF("Can't save null image '%s'...", (const char *)(cachedImage->name().toUtf8()));
	}
	else
		DPRINTF("UH OH, cachedImage isn't assigned!!!");
	
	//DPRINTF("Timer threadedSaveImageToCacheDirectory...");
	//DTIMERPRINT(savetimer, "threadedSaveImageToCacheDirectory");
}


QImage* CustomScaledCachedImageProvider::loadImageFromCacheDirectory(const QString &cacheFileName)
{
    DTIMERINIT(timer);
	
    QFile file(cacheFileName);
#ifdef QT4
    file.open(QIODevice::ReadOnly);
#else
    file.open(IO_ReadOnly);
#endif

    char signature[4];
#ifdef QT4
    file.seek(file.size() - 4);
    file.read(signature, 4);
#else
    file.at(file.size() - 4);
    file.readBlock(signature, 4);
#endif    
    
    if (memcmp(signature, "CBCI", 4) != 0)
    {
    	qWarning("No valid signature found in cache file %s. Deleting file.", (const char*)cacheFileName.toLatin1());
	    QFile::remove(cacheFileName);
    	return new QImage();
    }
    
    // Read offsets from the end of the file...
    int offsets[sizeScalesCount];
#ifdef QT4
    file.seek(file.size() - sizeScalesCount * sizeof(int) - 2 * sizeof(int) - 4);
    file.read((char *)offsets, sizeScalesCount * sizeof(int));

    // Read size of the biggest image...
    int cx, cy;
    file.read((char *)&cx, sizeof(int));
    file.read((char *)&cy, sizeof(int));
#else
    file.at(file.size() - sizeScalesCount * sizeof(int) - 2 * sizeof(int) - 4);
    file.readBlock((char *)offsets, sizeScalesCount * sizeof(int));

    // Read size of the biggest image...
    int cx, cy;
    file.readBlock((char *)&cx, sizeof(int));
    file.readBlock((char *)&cy, sizeof(int));
#endif
    // Determine the block to use...
    int block = scaleSizeBlock_;
    
    // Determine the start and end offsets of the source block...
    int startOffset = 0;
    int endOffset;
    
    if (block > 0)
    {
    	startOffset = offsets[block - 1];
    	endOffset = offsets[block];
    }
    else
    	endOffset = offsets[0];
    
    // Read content of block into our buffer...
    int blockLength = endOffset - startOffset;
#ifdef QT4
    QByteArray content(blockLength, 0);
    file.seek(startOffset);
    file.read(content.data(), blockLength);
    file.close();

    // Read image from buffer...    
    QBuffer buffer;
    buffer.open(QIODevice::ReadWrite);
    buffer.write(content, content.size());
    buffer.seek(0);

    QImageReader iio(&buffer, "JPEG");
#else
    QByteArray content(blockLength);
    file.at(startOffset);
    file.readBlock(content.data(), blockLength);
    file.close();

    // Read image from buffer...    
    QBuffer buffer;
    buffer.open(IO_ReadWrite);
    buffer.writeBlock(content, content.size());
    buffer.at(0);

	QImageIO iio(&buffer, "JPEG");
#endif	
    
	DPRINTF("Fastloading Image '%s'...", (const char *)(cacheFileName.toUtf8()));
    
#ifdef QTOPIA
   	//iio.setParameters(QString("Fast Shrink( %1 )").arg(scaleDenominator_).latin1());
	iio.setParameters("Fast");
#else
    //iio.setParameters(QString("Scale( %1, %2, ScaleMax )").arg(dstWidth).arg(dstHeight).latin1());
#endif
    
   	QImage *dstImage = new QImage();
   	
#ifdef QT4
   	bool result = iio.read(dstImage);
#else
    bool result = iio.read();
    if (result)
    	*dstImage = iio.image();
#endif
    
    DTIMERPRINT(timer, "fastLoadJPEGImage");
    
    return dstImage;
}
