/* -*- mode: c++; tab-width: 4; c-basic-offset: 4 -*- */
/*
 * Copyright (C) 2006-2008	Andre Beckedorf
 * 					 		<evilJazz _AT_ katastrophos _DOT_ net>
 *
 * Copyright (C) 2005 Atmark <atmarkat _AT_ msn _DOT_ com>
 *                    AGAWA Koji <i _AT_ atty _DOT_ jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>

#ifdef OSX_Carbon
#include <Carbon/Carbon.h>
// Get rid of DPRINTF defined in Carbon.h...
#undef DPRINTF
#endif

#include <qapplication.h>

#ifdef QTOPIA
#include <qpe/qpemenubar.h>
#include <qpe/qpetoolbar.h>
#include <qpe/qcopenvelope_qws.h>
#include <qpe/config.h>
#include <qpe/global.h>
#else
#include <qmenubar.h>
#include <config.h>
// for Qtopia we are using the QPE variants for toolbar and menubar
// to enable special features like the menu key etc.
// Define aliases to avoid ifdefs...
#define QPEMenuBar QMenuBar
#define QPEToolBar QToolBar
#endif

#ifdef QT4
#include <QToolBar>
#include <QResizeEvent>
#include <QCloseEvent>
#include <QDragEnterEvent>
#include <QKeyEvent>
#include <QEvent>
#include <QFrame>
#include <QDropEvent>
#include <QDragMoveEvent>
#include <QDesktopWidget>
#else
#include <qaccel.h>
#include <qpopupmenu.h>
#include <qheader.h>
#include <qwidgetstack.h>
#include <qvbox.h>
#include <qdragobject.h>
#include <qlistview.h>
#endif

#include <qmime.h>
#include <qmessagebox.h>
#include <qlayout.h>
#include <qdir.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qtextstream.h>
#include <qlineedit.h>
#include <qcursor.h>
#include <qaction.h>
#include <qsignalmapper.h>
#include <qtextcodec.h>
#include <qstringlist.h>
#include <qregexp.h>

#ifdef WINDOWS
#include "wincompat.h"
#endif

#include "addonmanager.h"
#include "platform/platformservices.h"
#include "textviewdialog.h"
#include "quasar.h"
#include "playbackcontroller.h"
#include "mplayer.h"
#include "operationpanel.h"
#include "toolpanel.h"
#include "videoplaybackwindow.h"
#include "playinfo.h"
#include "configuration.h"
#include "configurationdialog.h"
#include "media.h"
#include "qinputdialog.h"
#include "qprogressdialog.h"
#include "qsplitter.h"
#include "aboutdialog.h"
#include "action.h"
#include "system_volume.h"
#include "externalservices.h"
#ifdef QTOPIA
#include "batteryplus.h"
#endif
#include "playlistsearchfilterlineedit.h"
#include "playlistview.h"
#include "playlistoverview.h"
#include "paintbox.h"
#include "coverartflow.h"
#include "scrollbar.h"
#include "coverartprovider.h"
#include "mediainfodialog.h"
#include "playlisttoolpanel.h"
#include "skin.h"
#include "keynames.h"
#include "helpers.h"
#include "debug.h"

#include "compathack.h"

#define MinSliderLength 22

#ifdef QTOPIA
#ifndef Q_MOC_RUN
class QuasarStyle: public QPEStyle
{
public:
	QuasarStyle(): QPEStyle() {}
	~QuasarStyle() {}
	int sliderLength () const { return MinSliderLength; };
	void scrollBarMetrics(const QScrollBar *scrollbar, int &sliderMin, int &sliderMax, int &sliderLength, int &buttonDim)
	{
		QPEStyle::scrollBarMetrics(scrollbar, sliderMin, sliderMax, sliderLength, buttonDim);
		if (sliderLength < MinSliderLength)
		{
			sliderMax = sliderMax - (MinSliderLength - sliderLength);
			sliderLength = MinSliderLength;
		}
	}
	/*
    int pixelMetric(PixelMetric metric, const QWidget *widget) const
    {
        if (metric == PM_SplitterWidth)
            return 6;
        else
            return QWindowsStyle::pixelMetric(metric, widget);
    }
    */
};
#endif

#else

#ifndef QT4

#include <qwindowsstyle.h>

class QuasarStyle: public QWindowsStyle
{
public:
	QuasarStyle(): QWindowsStyle() {}
	~QuasarStyle() {}

    int pixelMetric(PixelMetric metric, const QWidget *widget) const
    {
        if (metric == PM_SplitterWidth)
            return 14;
        else if (metric == PM_ScrollBarSliderMin)
        	return 25;
        else
            return QWindowsStyle::pixelMetric(metric, widget);
    }
};
#endif

#endif

#ifdef OSX_Carbon
OSErr appleEventHandler(const AppleEvent *ae, AppleEvent *, SRefCon handlerRefcon)
{
	AEDescList docs;
	if (AEGetParamDesc(ae, keyDirectObject, typeAEList, &docs) == noErr)
	{
		long cnt = 0;
		AECountItems(&docs, &cnt);
		UInt8 strBuffer[256];
		QStringList filenameList;

		for (int i = 0; i < cnt; i++)
		{
			FSRef ref;
			if (AEGetNthPtr(&docs, i + 1, typeFSRef, 0, 0, &ref, sizeof(ref), 0) != noErr)
				continue;

			if (FSRefMakePath(&ref, strBuffer, 256) == noErr)
				filenameList.append(QString::fromUtf8(reinterpret_cast<char *>(strBuffer)));
		}

		if (filenameList.count() > 0)
		{
			Quasar *quasar = (Quasar *)handlerRefcon;
			quasar->handleExternalDocuments(filenameList);
		}
	}

	return noErr;
}

#endif

Quasar::Quasar()
{
	DMEMSTAT();
	setCaption(APPNAME);

	// configuration
	DPRINTF("Initialize Configuration class...");
	bool isValidConfig = qConfig.isValidConfig();
	if (isValidConfig)
		qConfig.read();
	DPRINTF("Done");

#ifdef QTOPIA
	qPlatformServices.setRemoteCon();
#endif

	receivedExternalDocuments_ = false;

#ifdef OSX_Carbon
    AEInstallEventHandler(kCoreEventClass, kAEOpenDocuments, appleEventHandler, (long)this, false);
#endif

	DMEMSTAT();

	playbackController_ = new PlaybackController();

	connect(playbackController_, SIGNAL(playOrderChanged(PlaybackController::PlayOrder)), this, SLOT(playbackControllerChangedPlayOrder(PlaybackController::PlayOrder)));

	connect(playbackController_, SIGNAL(stoppingPlayback(bool)), this, SLOT(playbackControllerStoppingPlayback(bool)));
	connect(playbackController_, SIGNAL(playbackStopped(bool)), this, SLOT(playbackControllerPlaybackStopped(bool)));

	connect(playbackController_, SIGNAL(playbackStarted()), this, SLOT(playbackControllerPlaybackStarted()));

	connect(playbackController_, SIGNAL(currentPlayTime(int)), this, SLOT(playbackControllerCurrentPlayTime(int)));

	connect(playbackController_, SIGNAL(playbackPaused()), this, SLOT(playbackControllerPlaybackPaused()));
	connect(playbackController_, SIGNAL(playbackUnpaused()), this, SLOT(playbackControllerPlaybackUnpaused()));

	connect(playbackController_, SIGNAL(mediaPrefetched(PlayListItem *)), this, SLOT(playbackControllerMediaPrefetched(PlayListItem *)));
	connect(playbackController_, SIGNAL(problemWithMedia(PlayListItem *)), this, SLOT(playbackControllerProblemWithMedia(PlayListItem *)));
	connect(playbackController_, SIGNAL(playerInitializationFailed(PlayListItem *)), this, SLOT(playbackControllerPlayerInitializationFailed(PlayListItem *)));

	DMEMSTAT();

	// skin
	DPRINTF("Initialize SkinManager class...");
	skinManager_ = new SkinManager(qGlobalConfig.skinPath(), this);
	if (!skinManager_->load(qConfig.skin1)) {
		QMessageBox::warning(this, tr("Warning"), tr("Could not load skin."));
		if (!skinManager_->load("default")) {
			QMessageBox::warning(this, tr("Fatal Error"), tr("Could not load default skin."));
			exit(1);
		}
	}

	// TODO: get rid of hardcoding...
#ifdef QTOPIA
	if (qApp->desktop()->width() == 640 || qApp->desktop()->width() == 320) {
		skinManager_->setStyle(SkinManager::Landscape);
		DPRINTF("  Initial style is Landscape");
	} else {
		skinManager_->setStyle(SkinManager::Portrait);
		DPRINTF("  Initial style is Portrait");
	}
#else
	skinManager_->setStyle(SkinManager::Landscape);
#endif
	DPRINTF("Done");

	DMEMSTAT();

	centralWidget_ = new QVBox(this);

	// Toggle Widget(Playlist/PlayInfo)
	DPRINTF("Initialize widget stack class...");

	widgetStack_ = new QWidgetStack(centralWidget_);
	widgetStack_->setFrameStyle(QFrame::NoFrame);
	widgetStack_->setSizePolicy(QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding));

	DPRINTF("Done");
	DMEMSTAT();

	DPRINTF("Initialize playlist container...");

	playListContainer_ = new QVBox(widgetStack_);
	playListContainer_->setFrameStyle(QFrame::NoFrame);
	widgetStack_->addWidget(playListContainer_, PlayListWidget);

	DPRINTF("Done");
	DMEMSTAT();

	DPRINTF("Initialize Playlist tool panel class...");
	playListToolPanel_ = new PlayListToolPanel(skinManager_, playListContainer_);
	// signals are connected further down...

	DPRINTF("Done");
	DMEMSTAT();

	playlistStack_ = new QWidgetStack(playListContainer_);
	playlistStack_->setFrameStyle(QFrame::NoFrame);
	playlistStack_->setSizePolicy(QSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding));

	DPRINTF("Initialize splitter container...");

	horizontalSplitterContainer_ = new QSplitter(playlistStack_);
	horizontalSplitterContainer_->setFrameStyle(QFrame::NoFrame);
	horizontalSplitterContainer_->setOpaqueResize(true);
	horizontalSplitterContainer_->setOrientation(Qt::Horizontal);

	verticalSplitterContainer_ = new QSplitter(horizontalSplitterContainer_);
	verticalSplitterContainer_->setFrameStyle(QFrame::NoFrame);
	verticalSplitterContainer_->setOpaqueResize(true);
	verticalSplitterContainer_->setOrientation(Qt::Vertical);

	playlistStack_->addWidget(horizontalSplitterContainer_, 0);
	playlistStack_->raiseWidget(horizontalSplitterContainer_);

	DPRINTF("Done");
	DMEMSTAT();

	DPRINTF("Checking application version...");

	// TODO: The following lines are temporary until a better solution is found.
	// Check for pre-TagLib condition and remove database to
	// force rescanning of all files...
	if (qConfig.readAppVersion() == -1)
	{
		qDebug("Removing old database...");
		QFile::remove(qGlobalConfig.databaseFilename());
	}

	DPRINTF("Initialize Playlist class...");

	playListView_ = new PlayListView();
	connect(playListView_, SIGNAL(selectionChanged()), this, SLOT(scheduleUpdateButtonAndMenuState()));

	DPRINTF("Done");
	DMEMSTAT();

	DPRINTF("Initialize PlaylistManager class...");

	playListManager_ = new PlayListManager(playListView_, playbackController_, playlistStack_, qConfig, this);
	connect(playListManager_, SIGNAL(switchingModeTo(PlayListManager::Mode)), this, SLOT(playlistManagerSwitchingModeTo(PlayListManager::Mode)));
	connect(playListManager_, SIGNAL(switchedModeTo(PlayListManager::Mode)), this, SLOT(playlistManagerSwitchedModeTo(PlayListManager::Mode)));
	connect(playListManager_, SIGNAL(playlistFileNameChanged()), this, SLOT(playlistManagerPlayListFileNameChanged()));
	connect(playListManager_, SIGNAL(playlistLoaded()), this, SLOT(playlistManagerPlayListFileNameChanged()));
	connect(playListManager_, SIGNAL(clipboardChanged()), this, SLOT(updateButtonAndMenuState()));
	connect(playListManager_, SIGNAL(selectionChanged()), this, SLOT(scheduleUpdateButtonAndMenuState()));

	connect(playListToolPanel_, SIGNAL(modeChanged(PlayListManager::Mode)), this, SLOT(playlistToolPanelModeChanged(PlayListManager::Mode)));
	connect(playListToolPanel_, SIGNAL(inputModeChanged(PlayListView::InputMode)), this, SLOT(setInputMode(PlayListView::InputMode)));
	connect(playListToolPanel_, SIGNAL(copyClicked()), playListManager_, SLOT(copySelectionFromPlayList()));
	connect(playListToolPanel_, SIGNAL(cutClicked()), playListManager_, SLOT(cutSelectionFromPlayList()));
	connect(playListToolPanel_, SIGNAL(pasteClicked()), playListManager_, SLOT(pasteIntoPlayList()));
	connect(playListToolPanel_, SIGNAL(deleteClicked()), playListManager_, SLOT(removeFileFromPlayList()));
	connect(playListToolPanel_, SIGNAL(resetFilterAndSortingClicked()), this, SLOT(resetFilterAndSorting()));
	connect(playListToolPanel_, SIGNAL(copyToOTGClicked()), playListManager_, SLOT(copyToOnTheGoPlayList()));
	connect(playListToolPanel_, SIGNAL(openPlaylistClicked()), playListManager_, SLOT(execDialogOpenPlayList()));
	connect(playListToolPanel_, SIGNAL(savePlaylistClicked()), playListManager_, SLOT(savePlayList()));

	DPRINTF("Done");
	DMEMSTAT();

	DPRINTF("Initialize Playlist overview class...");
	playListOverview_ = new PlayListOverview();
	connect(playListOverview_, SIGNAL(focusOutLeft()), this, SLOT(playlistOverviewFocusOutLeft()));
	connect(playListOverview_, SIGNAL(focusOutRight()), this, SLOT(playlistOverviewFocusOutRight()));
	playListManager_->setPlayListOverview(playListOverview_);

	DPRINTF("Done");
	DMEMSTAT();

	DPRINTF("Initialize CoverArtFlow class...");
	coverArtProviderCoverArtFlow_ = new CoverArtImageProvider(this);

	coverArtProviderCoverArtFlow_->setImageCacheDirectory(qConfig.coverArtFlowCacheDirectory);
	coverArtProviderCoverArtFlow_->setPreloadImageFilename(skinManager_->activeSkinDirectory() + "/nocover.png");

	coverArtFlowArea_ = new LayeredPaintBox();
	connect(coverArtFlowArea_, SIGNAL(rearrangeLayers()), this, SLOT(coverArtFlowAreaRearrangeLayers()));

	coverArtFlow_ = new CoverArtFlow(coverArtProviderCoverArtFlow_, coverArtFlowArea_);
	flowScrollBar_ = new SkinFlowScrollBar(Skin::FlowScrollBar, Skin::FlowScrollBarSlider, skinManager_, coverArtFlow_, coverArtFlowArea_);
	playListManager_->setCoverArtFlow(coverArtFlow_);
	coverArtFlow_->setPlayListView(playListView_);

	DPRINTF("Done");
	DMEMSTAT();

	DPRINTF("Setting up playlist layout...");
	playListOverview_->reparent(verticalSplitterContainer_, QPoint(0, 0), true);
	coverArtFlowArea_->reparent(verticalSplitterContainer_, QPoint(0, 0), true);
	playListView_->reparent(verticalSplitterContainer_, QPoint(0, 0), true);

	// hide playlist overview initially, so setOverviewVisibility works correctly...
	coverArtFlowArea_->hide();
	playListOverview_->hide();
	playListView_->setFocus();

	if (skinManager_->style() == SkinManager::Landscape)
		playListView_->loadColumnOrder(qGlobalConfig.configFilename(), "PlayListLandscape");
	else
		playListView_->loadColumnOrder(qGlobalConfig.configFilename(), "PlayListPortrait");

	changeOrientation(skinManager_->style() == SkinManager::Landscape, false, false);

	DPRINTF("Done");
	DMEMSTAT();

	DPRINTF("Initialize PlayInfo class...");
	// PlayInfo
	coverArtProviderPlayInfo_ = new CoverArtImageProvider(this);

	coverArtProviderPlayInfo_->setImageCacheDirectory(qConfig.coverArtFlowCacheDirectory);
	coverArtProviderPlayInfo_->setPreloadImageFilename(skinManager_->activeSkinDirectory() + "/nocover.png");

	playInfo_ = new PlayInfo(coverArtProviderPlayInfo_, skinManager_, widgetStack_);
	widgetStack_->addWidget(playInfo_, InfoWidget);

	DPRINTF("Done");
	DMEMSTAT();

	// operation panel
	DPRINTF("Initialize OperationPanel class...");
	operationPanel_ = new OperationPanel(skinManager_, centralWidget_);
	connect(operationPanel_, SIGNAL(playClicked()), playbackController_, SLOT(startOrPauseCurrentPlaybackOrStartNewPlayback()));
	connect(operationPanel_, SIGNAL(stopClicked()), playbackController_, SLOT(stopCurrentPlayback()));
	connect(operationPanel_, SIGNAL(nextClicked()), playbackController_, SLOT(playNextItem()));
	connect(operationPanel_, SIGNAL(previousClicked()), playbackController_, SLOT(playPreviousItem()));
	connect(playbackController_, SIGNAL(playbackStarted()), operationPanel_, SLOT(showPlaybackWidgets()));
	connect(playbackController_, SIGNAL(playbackStopped(bool)), operationPanel_, SLOT(hidePlaybackWidgets()));
	connect(operationPanel_, SIGNAL(seek(int)), playbackController_, SLOT(setAbsoluteSeconds(int)));

	DPRINTF("Done");
	DMEMSTAT();

	DPRINTF("Initialize ToolPanel class...");
	toolPanel_ = new ToolPanel(skinManager_, centralWidget_);
	connect(toolPanel_, SIGNAL(playOrderChanged(PlaybackController::PlayOrder)), playbackController_, SLOT(setPlayOrder(PlaybackController::PlayOrder)));
	connect(toolPanel_, SIGNAL(overviewVisibilityChanged(bool)), this, SLOT(setOverviewVisibility(bool)));
	connect(toolPanel_, SIGNAL(coverArtFlowVisibilityChanged(bool)), this, SLOT(setCoverArtFlowVisibility(bool)));
	connect(toolPanel_, SIGNAL(viewPlayListActivated()), this, SLOT(switchToLastActivePlayList()));
	connect(toolPanel_, SIGNAL(viewPlayInfoActivated()), this, SLOT(switchToInfoWidget()));

	connect(playListManager_, SIGNAL(showInfoText(const QString &)), toolPanel_, SLOT(setInfoText(const QString &)));
	connect(playListManager_, SIGNAL(showMessage(const QString &, int)), toolPanel_, SLOT(setStatusText(const QString &, int)));

	DPRINTF("Done");
	DMEMSTAT();

	// menu bar
	DPRINTF("Initialize Menus...");
#ifdef QT4
	menuBar_ = menuBar();

	toolBar_ = new QToolBar(this);
	subBar_ = new QToolBar(this);
#else
	toolBar_ = new QPEToolBar(this);
	toolBar_->setHorizontalStretchable(true);
	setToolBarsMovable(false);

	subBar_ = new QPEToolBar(this);

	menuBar_ = new QPEMenuBar(toolBar_);
	menuBar_->setMargin(0);
	menuBar_->setFrameStyle(QFrame::NoFrame);
#endif

	setupMenu();

	DPRINTF("Done");
	DMEMSTAT();

	// player window
	DPRINTF("Initialize PlayerWindow class...");
	videoPlaybackWindow_ = new VideoPlaybackWindow(this);
	videoPlaybackWindow_->hide();
	connect(videoPlaybackWindow_, SIGNAL(seekAbsolutePercentage(int)), playbackController_, SLOT(seekAbsolutePercentage(int)));

	// TODO: implement slots in PlaybackController for VideoPlaybackWindow's signals...
	// TODO: connect(videoPlaybackWindow_, SIGNAL(seekSliderMoved(int)), mplayer_, SLOT(osdShowProgressBar(int)));
	// TODO: connect(videoPlaybackWindow_, SIGNAL(seekStart()), mplayer_, SLOT(enableTemporaryOSD()));
	// TODO: connect(videoPlaybackWindow_, SIGNAL(seekEnd()), mplayer_, SLOT(disableTemporaryOSD()));

	DPRINTF("Done");

	DMEMSTAT();

	setupActions();

	DMEMSTAT();

#ifndef QT4
	setStyleRecursively(this, new QuasarStyle());
#endif

	// set current display info mode...
	DPRINTF("Setting Playlist font...");
	setPlayListFont(qConfig.infoMode);

	DPRINTF("Setting Playlist input mode...");
	setInputMode(qConfig.inputMode);

	isFullScreen_ = false;
	wasMaximized_ = false;

	// Set Repeat Mode
	DPRINTF("Setting play order...");
	playbackController_->setPlayOrder(qConfig.playOrder);

	DPRINTF("Setting central widget...");
	setCentralWidget(centralWidget_);

	// raise play list widget...
	DPRINTF("Raising playlist widget...");
	widgetStack_->raiseWidget(PlayListWidget);

	DPRINTF("Setting view on tool panel...");
	toolPanel_->setView(ToolPanel::PlayList);

	// BatteryPlus
	DPRINTF("Initializing BatteryPlus singleton...");
#ifdef QTOPIA
	BatteryPlus::singleton();
#endif

	DPRINTF("Checking for valid config file...");
	if (!isValidConfig)
	{
		QMessageBox::warning(centralWidget_, APPNAME,
							 tr("Invalid or non existing settings file.\nUsing the default settings."),
							 QMessageBox::Ok, Qt::NoButton);
	}

	DPRINTF("Installing event filter...");
	qApp->installEventFilter(this);

	playListView_->installEventFilter(this);
	playListView_->setAcceptDrops(true);

	// allows us to catch cancel/escape to avoid quitting the player by accident
	DPRINTF("Reparenting and setting WFlags...");
#ifdef QTOPIA
	reparent(0, Qt::WStyle_Customize | Qt::WStyle_NoBorder, QPoint(0, 0));
#endif

	DPRINTF("Initialize %s finished.", APPNAME);
	DMEMSTAT();

	qExternalServices.setPlaybackController(playbackController_);
	connect(playbackController_, SIGNAL(playbackStarted()), &qExternalServices, SLOT(playbackStarted()));
	connect(playbackController_, SIGNAL(playbackStopped(bool)), &qExternalServices, SLOT(playbackStopped()));
	connect(playbackController_, SIGNAL(playbackSeekedToAbsoluteSeconds(int)), &qExternalServices, SLOT(playbackSeekedToAbsoluteSeconds(int)));
	connect(playbackController_, SIGNAL(playbackPaused()), &qExternalServices, SLOT(playbackPaused()));
	connect(playbackController_, SIGNAL(playbackUnpaused()), &qExternalServices, SLOT(playbackUnpaused()));

	loadWidgetPosition(this, qGlobalConfig.configFilename(), "QuasarMainWindow");

	qAddOnManager.hookMainWindow(this);

	// we can't use polish, because we need to show the window and
	// some additional dialogs...
	QTimer::singleShot(1, this, SLOT(init()));
}

Quasar::~Quasar()
{
#ifdef OSX_Carbon
	AERemoveEventHandler(kCoreEventClass, kAEOpenDocuments, appleEventHandler, false);
#endif
}

void Quasar::init()
{

	// set overview visibility here since we need to be sure
	// the splitter container is correctly sized for the
	// setting of the splitter position to work correctly.
	setOverviewVisibility(qConfig.isOverviewVisible);
	setCoverArtFlowVisibility(qConfig.isCoverArtFlowVisible);

	readFromConfig();

	DPRINTF("Done");
	DMEMSTAT();

	this->setAcceptDrops(true);

	setActiveView(ViewOpenedPlaylist);

	DPRINTF("Done");
	DMEMSTAT();
}

void Quasar::coverArtFlowAreaRearrangeLayers()
{
	coverArtFlow_->setGeometry(QRect(0, 0, coverArtFlowArea_->width(), coverArtFlowArea_->height()));
	flowScrollBar_->setGeometry(QRect(30, coverArtFlowArea_->height() - 30, coverArtFlowArea_->width() - 60, 20));
}

QAction *Quasar::createAction(const QString& actionName, const QObject *receiver, const char *member)
{
	const Action &act = gActionManager.actionByName(actionName);
	connect(&act, SIGNAL(executed()), receiver, member);

	QAction *a = new QAction(this);
	a->setAccel(act.accelID());
	connect(a, SIGNAL(activated()), &act, SLOT(execute()));
	return a;
}

/*! メニューなどに無いアクションの設定をします。 */
void Quasar::setupActions()
{
	struct SeekAction {
		const char *action;
		int seconds;
	};

	static SeekAction actions[] = {
		{ "Playback/Seek Ahead By 1 Second", 1 },
		{ "Playback/Seek Ahead By 5 Seconds", 5 },
		{ "Playback/Seek Ahead By 10 Seconds", 10 },
		{ "Playback/Seek Ahead By 20 Seconds", 20 },
		{ "Playback/Seek Ahead By 30 Seconds", 30 },
		{ "Playback/Seek Ahead By 1 Minute", 60 * 1 },
		{ "Playback/Seek Ahead By 2 Minutes", 60 * 2 },
		{ "Playback/Seek Ahead By 5 Minutes", 60 * 5 },
		{ "Playback/Seek Ahead By 10 Minutes", 60 * 10 },
		{ "Playback/Seek Back By 1 Second", -1 },
		{ "Playback/Seek Back By 5 Seconds", -5 },
		{ "Playback/Seek Back By 10 Seconds", -10 },
		{ "Playback/Seek Back By 20 Seconds", -20 },
		{ "Playback/Seek Back By 30 Seconds", -30 },
		{ "Playback/Seek Back By 1 Minute", -60 * 1 },
		{ "Playback/Seek Back By 2 Minutes", -60 * 2 },
		{ "Playback/Seek Back By 5 Minutes", -60 * 5 },
		{ "Playback/Seek Back By 10 Minutes", -60 * 10 },
		{ 0, 0 }
	};

	QSignalMapper *sm = new QSignalMapper(this);
	connect(sm, SIGNAL(mapped(int)), playbackController_, SLOT(setRelativeSeconds(int)));
	for (SeekAction *p = actions; p->action != NULL; ++p)
	{
		createAction(p->action, sm, SLOT(map()));
		sm->setMapping(const_cast<Action *>(&gActionManager.actionByName(p->action)), p->seconds);
	}

	// volume
	sm = new QSignalMapper(this);
	connect(sm, SIGNAL(mapped(int)), &qPlatformServices, SLOT(setVolumeRelative(int)));
	createAction("Playback/Volume Up", sm, SLOT(map()));
	sm->setMapping(const_cast<Action *>(&gActionManager.actionByName("Playback/Volume Up")), 5);
	createAction("Playback/Volume Down", sm, SLOT(map()));
	sm->setMapping(const_cast<Action *>(&gActionManager.actionByName("Playback/Volume Down")), -5);

	// misc
#ifdef QTOPIA
	createAction("Playback/Turn On Display", &qPlatformServices, SLOT(turnOnDisplay()));
#endif
	// TODO: createAction("Playback/Toggle OSD", mplayer_, SLOT(toggleOSD()));
	createAction("Playlist/Focus Filterbox", filterEditBox_, SLOT(setFocus()));
	createAction("Playlist/Reset Filter And Focus Filterbox", this, SLOT(resetFilterAndFocusFilterEditBox()));
	createAction("Playlist/Reset Filter", this, SLOT(resetFilter()));
	createAction("Playlist/Play Selected Song", playbackController_, SLOT(stopCurrentPlaybackAndPlayCurrentItem()));
	createAction("Playback/Toggle Play Order", playbackController_, SLOT(togglePlayOrder()));
	createAction("Playback/Toggle Random Play Order", playbackController_, SLOT(toggleRandomPlayOrder()));
	createAction("Playback/Toggle PlayInfo", this, SLOT(toggleInfoWidget()));
	createAction("Playlist/Focus Playlist", playListView_, SLOT(setFocus()));
	createAction("Overview/Focus Overview", this, SLOT(setFocusPlayListOverview()));

#ifdef QTOPIA
	// brightness
	sm = new QSignalMapper(this);
	// TODO: connect(sm, SIGNAL(mapped(int)), mplayer_, SLOT(setBrightness(int)));
	createAction("Playback/Decrease Brightness", sm, SLOT(map()));
	sm->setMapping(&gActionManager.actionByName("Playback/Decrease Brightness"), -10);
	createAction("Playback/Increase Brightness", sm, SLOT(map()));
	sm->setMapping(&gActionManager.actionByName("Playback/Increase Brightness"), 10);
#endif
}

//! アプリケーション全体のイベントを監視するイベントフィルタです。
/*!
  ホットキーを処理するために使われています。
*/
bool Quasar::eventFilter(QObject *object, QEvent *event)
{
#ifndef QTOPIA
	if (event->type() == QEvent::DragEnter)
	{
		this->dragEnterEvent((QDragEnterEvent *)event);
		return true;
	}

	if (event->type() == QEvent::DragMove)
	{
		this->dragMoveEvent((QDragMoveEvent *)event);
		return true;
	}

	if (event->type() == QEvent::Drop)
	{
		this->dropEvent((QDropEvent *)event);
		return true;
	}
#endif

	if (object == filterEditBox_ &&
		((event->type() == QEvent::FocusIn) || (event->type() == QEvent::FocusOut)))
	{
		const SkinModeInformation *modeInfo = skinManager_->getInfo(qConfig.infoMode);

		QPalette palette = QApplication::palette();

		if (event->type() == QEvent::FocusIn)
		{
			setActiveView(Quasar::ViewLastActivePlaylist);

			if (modeInfo->filterBoxBackgroundColorFocused().isUsed())
			{
				palette.setColor(QColorGroup::Base, modeInfo->filterBoxBackgroundColorFocused());
			}

			if (modeInfo->filterBoxFontColorFocused().isUsed())
			{
				palette.setColor(QColorGroup::Text, modeInfo->filterBoxFontColorFocused());
				palette.setColor(QColorGroup::Foreground, modeInfo->filterBoxFontColorFocused());
			}
		}
		else if (event->type() == QEvent::FocusOut)
		{
			if (modeInfo->filterBoxBackgroundColor().isUsed())
			{
				palette.setColor(QColorGroup::Base, modeInfo->filterBoxBackgroundColor());
			}

			if (modeInfo->filterBoxFont().fontColor.isUsed())
			{
				palette.setColor(QColorGroup::Text, modeInfo->filterBoxFont().fontColor);
				palette.setColor(QColorGroup::Foreground, modeInfo->filterBoxFont().fontColor);
			}
		}

		filterEditBox_->setPalette(palette);
	}

	if (event->type() == QEvent::KeyPress)
	{
		if (!qApp->activeWindow()->isA("Quasar") &&
			!qApp->activeWindow()->isA("VideoPlaybackWindow"))
			return QMainWindow::eventFilter(object, event);

		QKeyEvent *e = static_cast<QKeyEvent *>(event);
		int key = e->key();

		DPRINTF("QEvent::KeyPress key=%d", key);

		// override special keys when the playlist overview is focused.
		if (playListOverview_->hasFocus() && (key == Qt::Key_Right || key == Qt::Key_Left || key == Qt::Key_Space))
			return QMainWindow::eventFilter(object, event);

		if (qApp->focusWidget() == filterEditBox_)
		{
			if (key == Qt::Key_Down || key == Qt::Key_Up || key == Qt::Key_PageDown || key == Qt::Key_PageUp)
			{
				if (playListManager_->isDynamicPlaylistEditorActive())
				{
					playlistStack_->visibleWidget()->setFocus();
					QMainWindow::eventFilter(playlistStack_->visibleWidget(), event);
				}
				else
				{
					if (playListOverview_->isVisible())
					{
						playListOverview_->setFocus();
						QMainWindow::eventFilter(playListOverview_, event);
					}
					else
					{
						playListView_->setFocus();
						QMainWindow::eventFilter(playListView_, event);
					}
				}
			}
			else
				return QMainWindow::eventFilter(object, event);
		}

		// Hotkeys are handled in a very non-standard fashion because we support different contexts.
		// Instead of relying on the accelerator key feature provided by Qt, we lookup the current key in our
		// context key -> accelID mapping and then send out a new QEvent::Accel carrying our translated accelID
		// to the event loop.
		// QActions and menuitems that have the accelID set in their accel property are triggered.
		// Thus, the accelID is not really a valid accel key per Qt definition but rather our internal ID that
		// we use to mark and call the associated action...

#ifdef QT4
		if (e->state() & Qt::ShiftModifier)
			key |= Qt::SHIFT;
		if (e->state() & Qt::ControlModifier)
			key |= Qt::CTRL;
		if (e->state() & Qt::AltModifier)
			key |= Qt::ALT;
#else
		if (e->state() & ShiftButton)
			key |= SHIFT;
		if (e->state() & ControlButton)
			key |= CTRL;
		if (e->state() & AltButton)
			key |= ALT;
#endif

		Configuration::HotKeyContext context = Configuration::HOTKEY_CONTEXT_GLOBAL;

		Configuration::HotKeyMap::Iterator it(qConfig.hotKeyMap[context].find(key));

		// if no key was found in the global context, look
		// in the other context key -> accelID mappings...
		if (it == qConfig.hotKeyMap[context].end())
		{
			if (videoPlaybackWindow_->isVisible())
				context = Configuration::HOTKEY_CONTEXT_VIDEO;
			else if (activeView() == ViewPlayInfo)
				context = Configuration::HOTKEY_CONTEXT_AUDIO;
			else if ((activeView() >= ViewDynPlaylistEditor && activeView() < ViewPlayInfo) || playListView_->hasFocus() || playListOverview_->hasFocus())
				context = Configuration::HOTKEY_CONTEXT_PLAYLIST;

			it = qConfig.hotKeyMap[context].find(key);
		}

		// did we find the key in the current context?
		if (it != qConfig.hotKeyMap[context].end())
		{
			int accelID = it.data();
			DPRINTF("%d (%s) -> %d", key, (const char *)(KeyNames::getName(key).toUtf8()), accelID);

			// create a new Accel QEvent...
#ifdef QT4
			QKeyEvent a(QEvent::Shortcut, accelID, 0, 0);
#else
			QKeyEvent a(QEvent::Accel, accelID, 0, 0);
#endif
			// we ignore it...
			a.ignore();
			// and delegate it...
			QApplication::sendEvent(topLevelWidget(), &a);

			// Finally accept the key event, because we've processed it.
			// That is, we created a new QEvent::Accel...
			e->accept();
			return true;
		}
	}

#ifdef QT4
	if (event->type() == QEvent::Shortcut)
#else
	if (event->type() == QEvent::Accel)
#endif
	{
		QKeyEvent *e = static_cast<QKeyEvent *>(event);

		int key = e->key();

		DPRINTF("QEvent::Accel key=%d", key);

		if (key == 4096) // ESC / Cancel
		{
			e->ignore();
			return true;
		}
	}

	return QMainWindow::eventFilter(object, event);
}

/*!
	@brief	関連付からの起動時に呼ばれるスロット
*/
void Quasar::setDocument(const QString &path)
{
	playbackController_->stopCurrentPlayback();
	playListManager_->newPlayList();

	// Load ASX
	QRegExp asxExp("^[^\\.]+.asx$", false);
#if QT4
	if (asxExp.indexIn(path) != -1)
#else
	if (asxExp.match(path) != -1)
#endif
	{
		playListManager_->openASX(path);
	}
	else
	{
		qConfig.lastDirectory = QFileInfo(path).dirPath(true);
		// TODO: Re-enable
		/*
		playListView_->insertItem(path);
		*/
	}

	//playListView_->updateFilter();
	playbackController_->startOrPauseCurrentPlaybackOrStartNewPlayback();
}

void Quasar::handleExternalDocuments(const QStringList &filenames)
{
	if (filenames.count() == 1)
	{
		QString filename = filenames[0];
		if (filename.right(11).lower() == "dynplaylist" ||
			filename.right(8).lower() == "playlist" ||
			filename.right(3).lower() == "m3u" ||
			filename.right(3).lower() == "asx")
		{
			playListManager_->clearPlayList();
			if (!playListManager_->readPlayList(filename, false))
			{
				//playListView_->reloadView();
			}

			receivedExternalDocuments_ = true;
			return;
		}
	}

	QExtStringList playList;

	for (QStringList::ConstIterator it = filenames.begin(); it != filenames.end(); ++it)
    {
        QExtString filename(*it);

#ifdef WINDOWS
        filename = filename.replace('\\', '/');
#endif

		if (QFileInfo(filename).isDir())
		{
			if (filename.right(1) != "/")
				filename += "/**";
			else
				filename += "**";
		}

		playList.append(filename);
    }

	if (playList.count() > 0)
		playListManager_->currentPlayList().addFromFileList(playList, true);

	receivedExternalDocuments_ = true;
}

#ifndef QTOPIA
void Quasar::dragEnterEvent(QDragEnterEvent *e)
{
	if (!QUriDrag::canDecode(e))
	{
		e->ignore();
		return;
	}

	e->accept();
}


void Quasar::dragMoveEvent(QDragMoveEvent *e)
{
	if (!QUriDrag::canDecode(e))
	{
		e->ignore();
		return;
	}

	e->accept();
}

void Quasar::dropEvent(QDropEvent *e)
{
	if (!QUriDrag::canDecode(e))
	{
		e->ignore();
		return;
	}

	QStrList lst;
	QUriDrag::decode(e, lst);

	e->acceptAction();

	QStringList dropList;

	for (uint i = 0; i < lst.count(); ++i)
		dropList.append(QUriDrag::uriToLocalFile(lst.at(i)));

	handleExternalDocuments(dropList);

	e->accept();
}
#endif

void Quasar::closeEvent(QCloseEvent *e)
{
	if (playbackController_->isPlaying())
	{
		playbackController_->stopCurrentPlayback();
		e->ignore();
		return;
	}
	else
	{
		// 終了確認
		if (qConfig.isAskQuitingEnabled)
		{
			int ret;

			ret = QMessageBox::warning(this, tr("Confirmation"),
										tr("Do you really want to quit?"),
										tr("Yes"), tr("No"), QString::null,
										1, 1);

			/*
				MessageBoxを出した後はe->ignore()が無効になっている？
				e->ignore();
				return;
				としても終了してしまう
			*/
			if (ret == 1)
			{
				DPRINTF("close cancelled.");
				e->ignore();
				DPRINTF("event ignored.");
				return;
			}
		}

#ifdef QTOPIA
		qPlatformServices.setRemoteCon(true);
#endif

		if (skinManager_->style() == SkinManager::Landscape)
			playListView_->saveColumnWidth(qGlobalConfig.configFilename(), "PlayListLandscape");
		else
			playListView_->saveColumnWidth(qGlobalConfig.configFilename(), "PlayListPortrait");

		playListView_->saveColumnOrder(qGlobalConfig.configFilename(), "PlayListLandscape");
		playListView_->saveColumnOrder(qGlobalConfig.configFilename(), "PlayListPortrait");

		saveWidgetPosition(this, qGlobalConfig.configFilename(), "QuasarMainWindow");

		playListManager_->saveStates();

		// save current splitter position if playlist overview is visible...
		if (playListOverview_->isVisible())
			qConfig.overviewSplitterPosition = verticalSplitterContainer_->sizes()[0];

		if (coverArtFlowArea_->isVisible())
			qConfig.coverArtFlowSplitterPosition = verticalSplitterContainer_->sizes()[1];

		qConfig.write();

		//delete playListView_;

		DPRINTF("close ok.");

		e->accept();
		return;
	}
}

void Quasar::resizeEvent(QResizeEvent *e)
{
	DPRINTF("Quasar::resizeEvent : oldSize(%dx%d), newSize(%dx%d)",
 		   e->oldSize().width(), e->oldSize().height(),
 		   e->size().width(), e->size().height());
 	DPRINTF("  desktop(%dx%d)", qApp->desktop()->width(), qApp->desktop()->height());

	// are we switching back from full screen mode?
	if (e->oldSize().width() == qApp->desktop()->width() &&
		e->oldSize().height() == qApp->desktop()->height())
	{
		// necessary to switch back to normal mode in a clean fashion
		// otherwise the fullscreen mode won't work again.
		showNormal();
#ifdef QTOPIA
		 // on Qtopia force maximized view immediately after switching back to normal mode...
		showMaximized();
#endif
		isFullScreen_ = false;
	}

	checkOrientation();
}

void Quasar::setFullScreen(int id)
{
	if (id != fullScreenMenuItemID_)
		return;

	qConfig.isFullScreenEnabled = qConfig.isFullScreenEnabled ? false : true;
	controlsMenu_->setItemChecked(fullScreenMenuItemID_, qConfig.isFullScreenEnabled);
}

void Quasar::about()
{
	AboutDialog *dialog = new AboutDialog();
	dialog->show();
}

void Quasar::rescanForCoverArt()
{
	gCoverArtProvider.rescan();

	coverArtFlow_->reset();
	coverArtFlow_->updateView();

	coverArtProviderPlayInfo_->resetNullImageNameList();
	coverArtProviderPlayInfo_->flushMemoryCache();
	if (playInfo_->isVisible())
		playInfo_->repaint();
}

void Quasar::showMediaInfo()
{
	MediaInfoDialog dialog(playListView_, skinManager_->activeSkinDirectory() + "/nocover.png", this);
	loadWidgetPosition(&dialog, qGlobalConfig.configFilename(), "MediaInfoDialog");
	dialog.exec();
	saveWidgetPosition(&dialog, qGlobalConfig.configFilename(), "MediaInfoDialog");
}

#ifdef QT4
QMenu * Quasar::createMenu(QMenuBar *menuBar, const char *menuText, QMenu *parent)
{
	if (parent)
		return parent->addMenu(tr(menuText));
	else
		return menuBar->addMenu(tr(menuText));
}

QuasarMenuID Quasar::createMenuItem(QMenu *menu, const char *menuText, QAction *action)
{
	return menu->addAction(tr(menuText), action, SIGNAL(activated()));
}

QuasarMenuID Quasar::createMenuItem(QMenu *menu, const char *menuText, const QObject * receiver, const char * member)
{
	return menu->addAction(tr(menuText), receiver, member);
}

QuasarMenuID Quasar::createMenuSeparator(QMenu *menu)
{
	return menu->addSeparator();
}

void Quasar::menuItemSetText(QMenu *menu, QuasarMenuID id, const char *text)
{
	id->setText(tr(text));
}

void Quasar::menuItemSetEnabled(QMenu *menu, QuasarMenuID id, bool value)
{
	id->setEnabled(value);
}

void Quasar::menuItemSetChecked(QMenu *menu, QuasarMenuID id, bool value)
{
	id->setChecked(value);
}

#else
QMenu * Quasar::createMenu(QMenuBar *menuBar, const char *menuText, QMenu *parent)
{
	QMenu *menu = new QMenu(menuBar);
	if (parent)
		parent->insertItem(tr(menuText), menu);
	else
		menuBar->insertItem(tr(menuText), menu);
	return menu;
}

QuasarMenuID Quasar::createMenuItem(QMenu *menu, const char *menuText, QAction *action)
{
	return menu->insertItem(tr(menuText), action, SIGNAL(activated()));
}

QuasarMenuID Quasar::createMenuItem(QMenu *menu, const char *menuText, const QObject * receiver, const char * member)
{
	return menu->insertItem(tr(menuText), receiver, member);
}

QuasarMenuID Quasar::createMenuSeparator(QMenu *menu)
{
	return menu->insertSeparator();
}

void Quasar::menuItemSetText(QMenu *menu, QuasarMenuID id, const char *text)
{
	menu->changeItem(id, tr(text));
}

void Quasar::menuItemSetEnabled(QMenu *menu, QuasarMenuID id, bool value)
{
	menu->setItemEnabled(id, value);
}

void Quasar::menuItemSetChecked(QMenu *menu, QuasarMenuID id, bool value)
{
	menu->setItemChecked(id, value);
}

#endif

void Quasar::setupMenu()
{
	updateButtonAndMenuTimer_ = new QTimer(this);
	connect(updateButtonAndMenuTimer_, SIGNAL(timeout()), this, SLOT(updateButtonAndMenuState()));

	// File
	fileMenu_ = createMenu(menuBar_, "File");
	connect(fileMenu_, SIGNAL(aboutToShow()), this, SLOT(fileMenuAboutToShow()));

	idFileMenuNewPlaylist_ = createMenuItem(fileMenu_, "New Playlist", createAction("Playlist/New", playListManager_, SLOT(newPlayList())));
	idFileMenuNewDynPlaylist_ = createMenuItem(fileMenu_, "New Dynamic Playlist...", playListManager_, SLOT(newDynamicPlayList()));
	idFileMenuOpenPlaylist_ = createMenuItem(fileMenu_, "Open Playlist...", createAction("Playlist/Open...", playListManager_, SLOT(execDialogOpenPlayList())));
	createMenuSeparator(fileMenu_);
	idFileMenuSavePlaylist_ = createMenuItem(fileMenu_, "Save Playlist", createAction("Playlist/Save", playListManager_, SLOT(savePlayList())));
	idFileMenuSaveAsPlaylist_ = createMenuItem(fileMenu_, "Save Playlist As...", createAction("Playlist/Save As...", playListManager_, SLOT(execDialogSavePlayListAs())));
	createMenuSeparator(fileMenu_);
	createMenuItem(fileMenu_, "Quit", this, SLOT(close()));

	editMenu_ = createMenu(menuBar_, "Edit");
	// connect handler for updating the above entries everytime the popup menu is shown...
	connect(editMenu_, SIGNAL(aboutToShow()), this, SLOT(editMenuAboutToShow()));

	createMenuItem(editMenu_, "Add From URL...", createAction("Playlist/Add Location...", playListManager_, SLOT(execDialogAddURLToPlayList())));
	createMenuItem(editMenu_, "Add From File...", createAction("Playlist/Add Files...", playListManager_, SLOT(execDialogAddFileToPlayList())));
	//createMenuItem(editMenu_, "Add From Directory...", createAction("Playlist/Add Directory...", playListManager_, SLOT(execDialogAddDirToPlayList())));
	createMenuItem(editMenu_, "Add From Directory Including Subdirectories...", playListManager_, SLOT(execDialogAddDirWithSubDirsToPlayList()));
	createMenuSeparator(editMenu_);
	idEditMenuCutItem_ = createMenuItem(editMenu_, "Cut", createAction("Playlist/Cut", playListManager_, SLOT(cutSelectionFromPlayList())));
	idEditMenuCopyItem_ = createMenuItem(editMenu_, "Copy", createAction("Playlist/Copy", playListManager_, SLOT(copySelectionFromPlayList())));
	idEditMenuPasteItem_ = createMenuItem(editMenu_, "Paste", createAction("Playlist/Paste", playListManager_, SLOT(pasteIntoPlayList())));
	idEditMenuDeleteItem_ = createMenuItem(editMenu_, "Delete", createAction("Playlist/Delete", playListManager_, SLOT(removeFileFromPlayList())));
	createMenuSeparator(editMenu_);
	createMenuItem(editMenu_, "Clear", createAction("Playlist/Clear", playListManager_, SLOT(clearPlayList())));
	createMenuSeparator(editMenu_);
	createMenuItem(editMenu_, "Select All", createAction("Playlist/Select All", playListManager_, SLOT(selectAll())));
	createMenuItem(editMenu_, "Deselect All", createAction("Playlist/Deselect All", playListManager_, SLOT(deselectAll())));
	createMenuSeparator(editMenu_);
	idEditMenuSelectModeItem_ = createMenuItem(editMenu_, "Select Mode", createAction("Playlist/Select Mode", this, SLOT(toggleSelectMode())));
	idEditMenuMultiSelectModeItem_ = createMenuItem(editMenu_, "Multiselect Mode", createAction("Playlist/Multiselect Mode", this, SLOT(toggleMultiSelectMode())));
	idEditMenuMoveItemModeItem_ = createMenuItem(editMenu_, "Move Item Mode", createAction("Playlist/Move Item Mode", this, SLOT(toggleMoveItemMode())));
	createMenuSeparator(editMenu_);
	idEditMenuCopyToOTGItem_ = createMenuItem(editMenu_, "Copy to On-The-Go play list", createAction("Playlist/Copy To On-The-Go Playlist", playListManager_, SLOT(copyToOnTheGoPlayList())));
	idEditMenuShowMediaInfoItem_ = createMenuItem(editMenu_, "Show Media Info...", createAction("Playlist/Show Media Info...", this, SLOT(showMediaInfo())));

#ifdef QT4
	playListView_->setPopupMenu(editMenu_);
#else
	playListView_->setPopupMenu(editMenu_);
#endif

	// View
	viewMenu_ = createMenu(menuBar_, "View");
	connect(viewMenu_, SIGNAL(aboutToShow()), this, SLOT(viewMenuAboutToShow()));

#ifdef QTOPIA
	createMenuItem(viewMenu_, "Turn Off Display", createAction("Playback/Turn Off Display", &qPlatformServices, SLOT(turnOffDisplay())));
#endif

	idViewMenuFullscreenItem_ = createMenuItem(viewMenu_, "Fullscreen", createAction("Display/Toggle Fullscreen", this, SLOT(toggleFullScreen())));
	createMenuSeparator(viewMenu_);
	idViewMenuShowOverviewItem_ = createMenuItem(viewMenu_, "Show Overview", createAction("Display/Toggle Overview Visibility", this, SLOT(toggleOverviewVisibility())));
	idViewMenuShowCoverArtFlowItem_ = createMenuItem(viewMenu_, "Show Cover Art Flow", createAction("Display/Toggle Cover Art Flow Visibility", this, SLOT(toggleCoverArtFlowVisibility())));
	createMenuSeparator(viewMenu_);

	// Sort By Submenu
	sortByMenu_ = createMenu(menuBar_, "Sort by", viewMenu_);
	connect(sortByMenu_, SIGNAL(activated(int)), playListView_, SLOT(setSortedColumn(int)));

	sortByMenu_->insertItem(tr("List order"), PlayListView::COLUMN_NONE);
	createMenuSeparator(sortByMenu_);
	sortByMenu_->insertItem(tr("Track number"), PlayListView::COLUMN_TRACKNO);
	sortByMenu_->insertItem(tr("Title"), PlayListView::COLUMN_TITLE);
	sortByMenu_->insertItem(tr("Artist"), PlayListView::COLUMN_ARTIST);
	sortByMenu_->insertItem(tr("Album"), PlayListView::COLUMN_ALBUM);
	sortByMenu_->insertItem(tr("Length"), PlayListView::COLUMN_LENGTH);
	//sortByMenu_->setItemChecked(playListView_->sortedColumn(), true);

	createMenuSeparator(viewMenu_);
	idViewMenuDynamicPlayListEditorItem_ = createMenuItem(viewMenu_, "Dynamic Playlist Editor", this, SLOT(switchPlaylistEditorView()));
	idViewMenuCurrentPlayListItem_ = createMenuItem(viewMenu_, "Current Playlist", createAction("View/Current Playlist", this, SLOT(switchToCurrentPlayList())));
	idViewMenuOnTheGoPlayListItem_ = createMenuItem(viewMenu_, "On-The-Go Playlist", createAction("View/On-The-Go Playlist", this, SLOT(switchToOnTheGoPlayList())));
	idViewMenuPlayInfoItem_ = createMenuItem(viewMenu_, "PlayInfo", createAction("View/PlayInfo", this, SLOT(switchToInfoWidget())));
	createMenuSeparator(viewMenu_);
	idViewMenuIncDisplaySize_ = createMenuItem(viewMenu_, "Increase Size", createAction("Display/Increase Size", this, SLOT(incDisplaySize())));
	idViewMenuDecDisplaySize_ = createMenuItem(viewMenu_, "Decrease Size", createAction("Display/Decrease Size", this, SLOT(decDisplaySize())));
	createMenuSeparator(viewMenu_);
	idViewMenuQuickswitchSkin1_ = createMenuItem(viewMenu_, "Quickswitch Skin 1", createAction("Display/Quickswitch Skin 1", this, SLOT(loadPrimarySkin())));
	idViewMenuQuickswitchSkin2_ = createMenuItem(viewMenu_, "Quickswitch Skin 2", createAction("Display/Quickswitch Skin 2", this, SLOT(loadSecondarySkin())));

	// Controls
	controlsMenu_ = createMenu(menuBar_, "Controls");
	controlsMenu_->setCheckable(true);
	fullScreenMenuItemID_ = controlsMenu_->insertItem(tr("Play Videos In Fullscreen Mode"));
	controlsMenu_->setItemChecked(fullScreenMenuItemID_, qConfig.isFullScreenEnabled);
	connect(controlsMenu_, SIGNAL(activated(int)), this, SLOT(setFullScreen(int)));
	createMenuSeparator(controlsMenu_);
	createMenuItem(controlsMenu_, "Play Or Pause", createAction("Playback/Play Or Pause", playbackController_, SLOT(startOrPauseCurrentPlaybackOrStartNewPlayback())));
	createMenuItem(controlsMenu_, "Stop", createAction("Playback/Stop", playbackController_, SLOT(stopCurrentPlayback())));
	createMenuItem(controlsMenu_, "Previous", createAction("Playback/Previous", playbackController_, SLOT(playPreviousItem())));
	createMenuItem(controlsMenu_, "Next", createAction("Playback/Next", playbackController_, SLOT(playNextItem())));
	createMenuSeparator(controlsMenu_);

	controlsMenu_->insertItem(tr("Normal"), PlaybackController::PlayOrderNormal);
	controlsMenu_->insertItem(tr("Repeat All"), PlaybackController::PlayOrderRepeatAll);
	controlsMenu_->insertItem(tr("Repeat One"), PlaybackController::PlayOrderRepeatOne);
	controlsMenu_->insertItem(tr("Random"), PlaybackController::PlayOrderRandom);
	controlsMenu_->insertItem(tr("Repeat Random"), PlaybackController::PlayOrderRepeatRandom);

	controlsMenu_->setItemChecked(qConfig.playOrder, true);
	connect(controlsMenu_, SIGNAL(activated(int)), this, SLOT(controlsMenuPlayOrderChanged(int)));

	// Tools
	toolsMenu_ = createMenu(menuBar_, "Tools");
	connect(toolsMenu_, SIGNAL(aboutToShow()), this, SLOT(toolsMenuAboutToShow()));
	idToolsMenuRescanDynamicPlaylist_ = createMenuItem(toolsMenu_, "Refresh Dynamic Playlist...", playListManager_, SLOT(refreshDynamicPlayList()));
	createMenuSeparator(toolsMenu_);
	idToolsMenuRescanForCoverArt_ = createMenuItem(toolsMenu_, "Rescan For Cover Art Images...", this, SLOT(rescanForCoverArt()));

#ifdef OSX
	// Move menu items to hidden submenu to workaround a bug in Qt's special handling of menu items on OS X
#ifdef QT4
	QMenu *dummyMenuOSX = createMenu(menuBar_, "OSX");
#else
	QPopupMenu *dummyMenuOSX = createMenu(menuBar_, "OSX");
#endif
	createMenuItem(dummyMenuOSX, "Preferences...", this, SLOT(configuration()));
	createMenuItem(dummyMenuOSX, "About...", this, SLOT(about()));

	idToolsMenuConfiguration_ = NULL;
	idToolsMenuAbout_ = NULL;
#else
	createMenuSeparator(toolsMenu_);
	idToolsMenuConfiguration_ = createMenuItem(toolsMenu_, "Preferences...", this, SLOT(configuration()));
	createMenuSeparator(toolsMenu_);
	idToolsMenuAbout_ = createMenuItem(toolsMenu_, "About...", this, SLOT(about()));
#endif

	// Filter Edit Box
	filterEditBox_ = new PlayListSearchFilterLineEdit(subBar_);
	connect(filterEditBox_, SIGNAL(returnPressed()), this, SLOT(filterEditBoxReturnPressed()));
	connect(filterEditBox_, SIGNAL(textChanged(const QString&)), this, SLOT(filterEditBoxTextChanged(const QString&)));
	playListManager_->setSearchFilterLineEdit(filterEditBox_);

#ifndef QT4
	subBar_->setStretchableWidget(filterEditBox_);
#endif
}

void Quasar::fileMenuAboutToShow()
{
	Quasar::View view = activeView();
	bool viewIsOpenedPlaylist = (view == Quasar::ViewOpenedPlaylist || view == Quasar::ViewDynPlaylistEditor);

	// New Playlist Menu Item
	menuItemSetText(fileMenu_, idFileMenuNewPlaylist_, viewIsOpenedPlaylist ? tr("New Playlist") : tr("New Playlist (N/A)"));
	menuItemSetEnabled(fileMenu_, idFileMenuNewPlaylist_, viewIsOpenedPlaylist);

	menuItemSetText(fileMenu_, idFileMenuNewPlaylist_, viewIsOpenedPlaylist ? tr("New Playlist") : tr("New Playlist (N/A)"));
	menuItemSetEnabled(fileMenu_, idFileMenuNewPlaylist_, viewIsOpenedPlaylist);

	// New Dynamic Playlist Menu Item
	menuItemSetText(fileMenu_, idFileMenuNewDynPlaylist_, viewIsOpenedPlaylist ? tr("New Dynamic Playlist") : tr("New Dynamic Playlist (N/A)"));
	menuItemSetEnabled(fileMenu_, idFileMenuNewDynPlaylist_, viewIsOpenedPlaylist);

	// Open Menu Item
	menuItemSetText(fileMenu_, idFileMenuOpenPlaylist_, viewIsOpenedPlaylist ? tr("Open Playlist...") : tr("Load From Playlist File..."));

	// Save Menu Item
	menuItemSetText(fileMenu_, idFileMenuSavePlaylist_, viewIsOpenedPlaylist ? tr("Save Playlist") : tr("Save Playlist (N/A)"));
	menuItemSetEnabled(fileMenu_, idFileMenuSavePlaylist_, viewIsOpenedPlaylist);

	// Save As Menu Item
	menuItemSetText(fileMenu_, idFileMenuSaveAsPlaylist_, viewIsOpenedPlaylist ? tr("Save As Playlist...") : tr("Save To Playlist File..."));
}

void Quasar::updateButtonAndMenuState()
{
	if (updateButtonAndMenuTimer_->isActive())
		updateButtonAndMenuTimer_->stop();

	editMenuAboutToShow();
}

void Quasar::scheduleUpdateButtonAndMenuState()
{
	if (updateButtonAndMenuTimer_->isActive())
		updateButtonAndMenuTimer_->stop();

	updateButtonAndMenuTimer_->start(500, TRUE);
}

void Quasar::editMenuAboutToShow()
{
	bool isDynamicPlaylist = playListManager_->isDynamicPlaylist();
	bool isDynamicPlaylistEditorActive = playListManager_->isDynamicPlaylistEditorActive();
	bool anythingSelected = playListManager_->isAnythingSelected();

	menuItemSetText(editMenu_, idEditMenuCutItem_, isDynamicPlaylist ? tr("Cut (N/A for Dynamic Playlists)") : tr("Cut"));
	bool cutEnabled = !isDynamicPlaylist && !isDynamicPlaylistEditorActive && anythingSelected;
	menuItemSetEnabled(editMenu_, idEditMenuCutItem_, cutEnabled);
	playListToolPanel_->setCutEnabled(cutEnabled);

	bool copyEnabled = !isDynamicPlaylistEditorActive && anythingSelected;
	menuItemSetEnabled(editMenu_, idEditMenuCopyItem_, copyEnabled);
	playListToolPanel_->setCopyEnabled(copyEnabled);

	menuItemSetText(editMenu_, idEditMenuPasteItem_, isDynamicPlaylist ? tr("Paste (N/A for Dynamic Playlists)") : tr("Paste"));
	bool pasteEnabled =
		!isDynamicPlaylist &&
		!isDynamicPlaylistEditorActive &&
		playListManager_->mediaClipboardCount() > 0;

	menuItemSetEnabled(editMenu_, idEditMenuPasteItem_, pasteEnabled);
	playListToolPanel_->setPasteEnabled(pasteEnabled);

	menuItemSetText(editMenu_, idEditMenuDeleteItem_, isDynamicPlaylist && !isDynamicPlaylistEditorActive ? tr("Delete (N/A for Dynamic Playlists)") : tr("Delete"));
	bool deleteEnabled = (!isDynamicPlaylist || isDynamicPlaylistEditorActive) && anythingSelected;
	menuItemSetEnabled(editMenu_, idEditMenuDeleteItem_, deleteEnabled);
	playListToolPanel_->setDeleteEnabled(deleteEnabled);

	bool otgEnabled = playListManager_->activeMode() == PlayListManager::ModeOpenedPlaylist && anythingSelected;
	playListToolPanel_->setCopyToOTGEnabled(otgEnabled);
	menuItemSetEnabled(editMenu_, idEditMenuCopyToOTGItem_, otgEnabled);

	menuItemSetEnabled(editMenu_, idEditMenuShowMediaInfoItem_, anythingSelected);

	menuItemSetChecked(editMenu_, idEditMenuSelectModeItem_, playListView_->inputMode() == PlayListView::Select);
	menuItemSetChecked(editMenu_, idEditMenuMultiSelectModeItem_, playListView_->inputMode() == PlayListView::MultiSelect);
	menuItemSetChecked(editMenu_, idEditMenuMoveItemModeItem_, playListView_->inputMode() == PlayListView::Move);
	menuItemSetChecked(editMenu_, idEditMenuMoveItemModeItem_, !isDynamicPlaylist && anythingSelected);
}

void Quasar::viewMenuAboutToShow()
{
	Quasar::View view = activeView();

	menuItemSetChecked(viewMenu_, idViewMenuFullscreenItem_, isFullScreen_);

	menuItemSetChecked(viewMenu_, idViewMenuShowOverviewItem_, playListOverview_->isVisible());
	menuItemSetChecked(viewMenu_, idViewMenuShowCoverArtFlowItem_, coverArtFlowArea_->isVisible());

	// current playlist menu item
	menuItemSetChecked(viewMenu_, idViewMenuCurrentPlayListItem_, view == Quasar::ViewOpenedPlaylist);

	// on the go playlist menu item
	menuItemSetChecked(viewMenu_, idViewMenuOnTheGoPlayListItem_, view == Quasar::ViewOnTheGoPlaylist);

	// Dynamic playlist editor menu item
	menuItemSetText(viewMenu_, idViewMenuDynamicPlayListEditorItem_, playListManager_->isDynamicPlaylist() ? tr("Dynamic Playlist Editor") : tr("Dynamic Playlist Editor (N/A)"));
	menuItemSetEnabled(viewMenu_, idViewMenuDynamicPlayListEditorItem_, playListManager_->isDynamicPlaylist());
	menuItemSetChecked(viewMenu_, idViewMenuDynamicPlayListEditorItem_, view == Quasar::ViewDynPlaylistEditor);

	// Play info menu item
	menuItemSetChecked(viewMenu_, idViewMenuPlayInfoItem_, view == Quasar::ViewPlayInfo);

	// Update Increase / Decrease size menu items
	menuItemSetEnabled(viewMenu_, idViewMenuIncDisplaySize_, qConfig.infoMode != SkinManager::BigMode);
	menuItemSetEnabled(viewMenu_, idViewMenuDecDisplaySize_, qConfig.infoMode != SkinManager::SmallMode);
}

void Quasar::toolsMenuAboutToShow()
{
	menuItemSetEnabled(toolsMenu_, idToolsMenuRescanDynamicPlaylist_, playListManager_->isDynamicPlaylist());
}

void Quasar::configuration()
{
	configuration(false);
}

void Quasar::configuration(bool showMPlayerConfig)
{
	DPRINTF("[ConfigurationDialog] Initializing...");
	ConfigurationDialog configDialog(qConfig, this, "Configuration Dialog");

	connect(&configDialog, SIGNAL(nukeMediaCacheButtonClicked()), playListManager_->mediaDatabase(), SLOT(compactDB()));
	DPRINTF("[ConfigurationDialog] Done");

	if (showMPlayerConfig)
		configDialog.showMPlayerConfig();

	DPRINTF("[ConfigurationDialog] Executing...");
	if (configDialog.exec() == QDialog::Accepted)
	{
		DPRINTF("[ConfigurationDialog] Done");

		qConfig = configDialog.result();
		qConfig.write();

		readFromConfig();
	}
	DPRINTF("[ConfigurationDialog] Destroy");
}

void Quasar::readFromConfig()
{
#ifdef QTOPIA
	qPlatformServices.setRemoteCon();
#endif

	loadSkin(qConfig.skin1);

	DPRINTF("Scanning cover art directory...");
	gCoverArtProvider.setCoverArtDirectory(qConfig.coverArtDirectory);

	if (qConfig.isTouchModeEnabled)
		playListOverview_->setSelectionMode(QListBox::Multi);
	else
		playListOverview_->setSelectionMode(QListBox::Extended);

#ifndef QT4
	playListView_->setKineticScrollingEnabled(qConfig.isKineticScrollingEnabled);
#endif
}

void Quasar::filterEditBoxReturnPressed()
{
	playListView_->setFocus();
}

void Quasar::filterEditBoxTextChanged(const QString &text)
{
	setActiveView(Quasar::ViewLastActivePlaylist);
}

void Quasar::resetFilter()
{
	filterEditBox_->clear(); // will trigger setFilter
}

void Quasar::resetFilterAndFocusFilterEditBox()
{
	resetFilter();
	filterEditBox_->setFocus();
}

void Quasar::setPlayListFont(SkinManager::InfoMode mode)
{
	const SkinModeInformation *modeInfo = skinManager_->getInfo(mode);

	playListView_->setSkinMode(*modeInfo);
	playListOverview_->setSkinMode(*modeInfo);

	if (modeInfo->playListOverviewSeparatorColor().isUsed())
	{
		QPalette palette(modeInfo->playListOverviewSeparatorColor());
		verticalSplitterContainer_->setPalette(palette);
		horizontalSplitterContainer_->setPalette(palette);
	}
	else
	{
		verticalSplitterContainer_->unsetPalette();
		horizontalSplitterContainer_->unsetPalette();
	}

	if (modeInfo->toolBarColor().isUsed())
	{
		QPalette palette(modeInfo->toolBarColor());

		if (modeInfo->toolBarFont().fontColor.isUsed())
		{
			palette.setColor(QColorGroup::Text, modeInfo->toolBarFont().fontColor);
			palette.setColor(QColorGroup::ButtonText, modeInfo->toolBarFont().fontColor);
			palette.setColor(QColorGroup::Foreground, modeInfo->toolBarFont().fontColor);
		}

		toolBar_->setPalette(palette);
		subBar_->setPalette(palette);
	}
	else
	{
		toolBar_->unsetPalette();
		subBar_->unsetPalette();
	}

	if (modeInfo->toolBarFont().font.isUsed())
	{
		toolBar_->setFont(modeInfo->toolBarFont().font);
		subBar_->setFont(modeInfo->toolBarFont().font);
	}
	else
	{
		toolBar_->unsetFont();
		subBar_->unsetFont();
	}

	if (modeInfo->filterBoxBackgroundColor().isUsed())
	{
		QPalette palette(modeInfo->filterBoxBackgroundColor());

		if (modeInfo->filterBoxFont().fontColor.isUsed())
		{
			palette.setColor(QColorGroup::Text, modeInfo->filterBoxFont().fontColor);
			palette.setColor(QColorGroup::Foreground, modeInfo->filterBoxFont().fontColor);
		}

		filterEditBox_->setPalette(palette);
	}
	else
	{
		filterEditBox_->unsetPalette();
	}

	if (modeInfo->filterBoxFont().font.isUsed())
		filterEditBox_->setFont(modeInfo->filterBoxFont().font);
	else
		filterEditBox_->unsetFont();

#ifndef QTOPIA
	// Workaround for realignment issues in Qt3
	toolBar_->hide();
	toolBar_->show();
#endif
}

void Quasar::incDisplaySize()
{
	if (qConfig.infoMode == SkinManager::BigMode) return;
	if (qConfig.infoMode == SkinManager::SmallMode)
		qConfig.infoMode = SkinManager::NormalMode;
	else
		qConfig.infoMode = SkinManager::BigMode;
	setPlayListFont(qConfig.infoMode);
}

void Quasar::decDisplaySize()
{
	if (qConfig.infoMode == SkinManager::SmallMode) return;
	if (qConfig.infoMode == SkinManager::BigMode)
		qConfig.infoMode = SkinManager::NormalMode;
	else
		qConfig.infoMode = SkinManager::SmallMode;
	setPlayListFont(qConfig.infoMode);
}

void Quasar::changeOrientation(bool landscape, bool setSkinStyle, bool saveCurrentColumnWidths)
{
#ifndef QT4
	if (landscape)
	{
		if (setSkinStyle)
			skinManager_->setStyle(SkinManager::Landscape);

		if (saveCurrentColumnWidths)
			playListView_->saveColumnWidth(qGlobalConfig.configFilename(), "PlayListPortrait");

		// TODO: Remove hardcoded default column widths to some other place...
		const int defaultColumnWidth[6] = {20, 42, 182, 154, 162, 54};
		for (int i = 0; i < playListView_->columns(); ++i)
			playListView_->setColumnWidth(playListView_->header()->mapToSection(i), defaultColumnWidth[i]);

		playListView_->loadColumnWidth(qGlobalConfig.configFilename(), "PlayListLandscape");
	}
	else
	{
		if (setSkinStyle)
			skinManager_->setStyle(SkinManager::Portrait);

		if (saveCurrentColumnWidths)
			playListView_->saveColumnWidth(qGlobalConfig.configFilename(), "PlayListLandscape");

		// TODO: Remove hardcoded default column widths to some other place...
		const int defaultColumnWidth[6] = {24, 40, 107, 105, 119, 48};
		for (int i = 0; i < playListView_->columns(); ++i)
			playListView_->setColumnWidth(playListView_->header()->mapToSection(i), defaultColumnWidth[i]);

		playListView_->loadColumnWidth(qGlobalConfig.configFilename(), "PlayListPortrait");
	}

	if (centralWidget())
	{
		centralWidget()->updateGeometry();
		if (centralWidget()->layout())
			centralWidget()->layout()->activate();
	}

	currentWidth_ = qApp->desktop()->width();
#endif
}

void Quasar::checkOrientation()
{
	if (qApp->desktop()->width() != currentWidth_)
	{
		changeOrientation(qApp->desktop()->width() > qApp->desktop()->height());
	}
}

void Quasar::toggleFullScreen()
{
	if (!isFullScreen_)
	{
		showFullScreen();
		isFullScreen_ = true;
		wasMaximized_ = isMaximized();
	}
	else
	{
		// necessary to switch back to normal mode in a clean fashion
		// otherwise the fullscreen mode won't work again.
		showNormal();
#ifdef QTOPIA
		 // on Qtopia force maximized view immediately after switching back to normal mode...
		showMaximized();
#endif

		if (wasMaximized_)
			showMaximized();

		isFullScreen_ = false;
	}
}

void Quasar::toggleOverviewVisibility()
{
	setOverviewVisibility(!playListOverview_->isVisible());
}

void Quasar::updateSplitterSectionHeights()
{
	// Dirty hack to get actual total size...
	int totalSize = 0;

	for (int i = 0; i < verticalSplitterContainer_->sizes().count(); ++i)
		totalSize += verticalSplitterContainer_->sizes()[i];

	// create list of heights for each of the two widgets in the splitter container
	// that is: PlayListOverview and PlayListView
	QValueList<int> sizes;
	sizes.append(playListOverview_->isVisible() ? qConfig.overviewSplitterPosition : 0);
	sizes.append(coverArtFlowArea_->isVisible() ? qConfig.coverArtFlowSplitterPosition : 0);
	sizes.append(totalSize - sizes[0] - sizes[1]);

	verticalSplitterContainer_->setSizes(sizes);
}

void Quasar::setOverviewVisibility(bool visible)
{
	setActiveView(Quasar::ViewLastActivePlaylist);

	if (visible)
	{
		playListOverview_->show();
		updateSplitterSectionHeights();
		playListOverview_->setFocus();
	}
	else
	{
		// only set current splitter position if playlist overview is visible...
		if (playListOverview_->isVisible())
			qConfig.overviewSplitterPosition = verticalSplitterContainer_->sizes()[0];

		if (coverArtFlowArea_->isVisible())
			qConfig.coverArtFlowSplitterPosition = verticalSplitterContainer_->sizes()[1];

		playListOverview_->hide();
		updateSplitterSectionHeights();
		playListView_->setFocus();
	}

	qConfig.isOverviewVisible = visible;
	toolPanel_->setOverviewVisibility(visible);
}

void Quasar::toggleCoverArtFlowVisibility()
{
	setCoverArtFlowVisibility(!coverArtFlowArea_->isVisible());
}

void Quasar::setCoverArtFlowVisibility(bool visible)
{
	setActiveView(Quasar::ViewLastActivePlaylist);

	if (visible)
	{
		coverArtFlowArea_->show();
		coverArtFlow_->setActive(true);
		updateSplitterSectionHeights();
		coverArtFlowArea_->setFocus();
	}
	else
	{
		// only set current splitter position if playlist overview is visible...
		if (playListOverview_->isVisible())
			qConfig.overviewSplitterPosition = verticalSplitterContainer_->sizes()[0];

		if (coverArtFlowArea_->isVisible())
			qConfig.coverArtFlowSplitterPosition = verticalSplitterContainer_->sizes()[1];

		coverArtFlowArea_->hide();
		coverArtFlow_->setActive(false);
		updateSplitterSectionHeights();
		playListView_->setFocus();
	}

	qConfig.isCoverArtFlowVisible = visible;
	toolPanel_->setCoverArtFlowVisibility(visible);
}

void Quasar::loadSkin(const QString &name, bool force)
{
	if (skinManager_->activeSkin() != name || force)
	{
		operationPanel_->hide();
		skinManager_->load(name);

		if (qApp->desktop()->width() > qApp->desktop()->height())
			skinManager_->setStyle(SkinManager::Landscape);
		else
			skinManager_->setStyle(SkinManager::Portrait);

		setPlayListFont(qConfig.infoMode);
		coverArtProviderCoverArtFlow_->setPreloadImageFilename(skinManager_->activeSkinDirectory() + "/nocover.png");
		coverArtProviderPlayInfo_->setPreloadImageFilename(skinManager_->activeSkinDirectory() + "/nocover.png");

		operationPanel_->show();
		operationPanel_->updateGeometry();
		repaint();
	}
}

void Quasar::loadPrimarySkin()
{
	loadSkin(qConfig.skin1, true);
}

void Quasar::loadSecondarySkin()
{
	loadSkin(qConfig.skin2, true);
}

void Quasar::playlistManagerPlayListFileNameChanged()
{
	setCaption(QString(APPNAME) + " - " + playListManager_->getDisplayPlayListFilename());
}

void Quasar::playlistManagerPlayListCleared()
{
	resetFilter();
	updateButtonAndMenuState();
}

void Quasar::playlistOverviewFocusOutLeft()
{
	filterEditBox_->setFocus();
}

void Quasar::playlistOverviewFocusOutRight()
{
	playListView_->setFocus();
}

void Quasar::setFocusPlayListOverview()
{
	if (!playListOverview_->isVisible())
		setOverviewVisibility(true);

	playListOverview_->setFocus();
}

Quasar::View Quasar::activeView()
{
	if (widgetStack_->visibleWidget() == playInfo_)
		return Quasar::ViewPlayInfo;
	else
	{
		switch (playListManager_->activeMode())
		{
			case PlayListManager::ModeDynPlaylistEditor:
				return Quasar::ViewDynPlaylistEditor;
			case PlayListManager::ModeOpenedPlaylist:
				return Quasar::ViewOpenedPlaylist;
			case PlayListManager::ModeOnTheGoPlaylist:
				return Quasar::ViewOnTheGoPlaylist;
			default:
				return Quasar::ViewLastActivePlaylist;
		}
	}
}

bool Quasar::setActiveView(View view)
{
	bool result = false;

	if (view == activeView())
		return true;

	if (view == Quasar::ViewPlayInfo)
	{
		toolBar_->hide();
		toolPanel_->hide();
		operationPanel_->hide();
		subBar_->hide();

		playListManager_->setActiveMode(PlayListManager::ModeLastActivePlaylist);
		widgetStack_->raiseWidget(InfoWidget);
		toolPanel_->setView(ToolPanel::PlayInfo);
	}
	else
	{
		toolBar_->show();
		toolPanel_->show();
		operationPanel_->show();
		subBar_->show();

		PlayListManager::Mode mode = PlayListManager::ModeLastActivePlaylist;

		switch (view)
		{
			case Quasar::ViewDynPlaylistEditor:
				mode = PlayListManager::ModeDynPlaylistEditor;
				break;
			case Quasar::ViewOpenedPlaylist:
				mode = PlayListManager::ModeOpenedPlaylist;
				break;
			case Quasar::ViewOnTheGoPlaylist:
				mode = PlayListManager::ModeOnTheGoPlaylist;
				break;
			default:
				mode = PlayListManager::ModeLastActivePlaylist;
		}

		// this will fire playlistManagerSwitchingModeTo below...
		// play info widget is taken care of there...
		playListManager_->setActiveMode(mode);
	}

	updateButtonAndMenuState();

	return result;
}

void Quasar::playlistManagerSwitchingModeTo(PlayListManager::Mode mode)
{
	// make sure play info widget is lower than playlist...
	if (widgetStack_->visibleWidget() == playInfo_)
	{
		widgetStack_->raiseWidget(PlayListWidget);
		toolPanel_->setView(ToolPanel::PlayList);

		// if we are switching because the filter edit box was activated,
		// keep the focus in there, otherwise shift focus to playlist widget...
		if (qApp->focusWidget() != filterEditBox_)
			playListView_->setFocus();
	}
}

void Quasar::playlistManagerSwitchedModeTo(PlayListManager::Mode mode)
{
	playlistManagerPlayListFileNameChanged();
	playListToolPanel_->setMode(mode);
	updateButtonAndMenuState();
}

void Quasar::playlistToolPanelModeChanged(PlayListManager::Mode mode)
{
	playListManager_->setActiveMode(mode);
}

void Quasar::toggleInfoWidget()
{
	if (activeView() != Quasar::ViewPlayInfo)
		setActiveView(Quasar::ViewPlayInfo);
	else
		setActiveView(Quasar::ViewLastActivePlaylist);
}

void Quasar::switchPlaylistEditorView()
{
	setActiveView(Quasar::ViewDynPlaylistEditor);
}

void Quasar::switchToLastActivePlayList()
{
	setActiveView(Quasar::ViewLastActivePlaylist);
}

void Quasar::switchToCurrentPlayList()
{
	setActiveView(Quasar::ViewOpenedPlaylist);
}

void Quasar::switchToOnTheGoPlayList()
{
	setActiveView(Quasar::ViewOnTheGoPlaylist);
}

void Quasar::switchToInfoWidget()
{
	setActiveView(Quasar::ViewPlayInfo);
}

void Quasar::setInputMode(PlayListView::InputMode inputMode)
{
	qConfig.inputMode = inputMode;
	playListManager_->setInputMode(inputMode);
	playListToolPanel_->setInputMode(inputMode);
}

void Quasar::resetFilterAndSorting()
{
	/*
	playListView_->resetSorting();
	playListOverview_->reset();
	*/
	resetFilter();
	updateButtonAndMenuState();
}

void Quasar::controlsMenuPlayOrderChanged(int playOrderID)
{
	playbackController_->setPlayOrder(static_cast<PlaybackController::PlayOrder>(playOrderID));
}

void Quasar::playbackControllerChangedPlayOrder(PlaybackController::PlayOrder playOrder)
{
	qConfig.playOrder = playOrder;

	for (int i = 0; i < PlaybackController::PlayOrderMax; ++i)
		controlsMenu_->setItemChecked(i, i == playOrder ? true : false);

	QString statusMsg;
	switch (playOrder)
	{
		case PlaybackController::PlayOrderRepeatAll:
			statusMsg = tr("Switched to play order Repeat All.");
			break;
		case PlaybackController::PlayOrderRepeatOne:
			statusMsg = tr("Switched to play order Repeat One.");
			break;
		case PlaybackController::PlayOrderRandom:
			statusMsg = tr("Switched to play order Random.");
			break;
		case PlaybackController::PlayOrderRepeatRandom:
			statusMsg = tr("Switched to play order Repeat Random.");
			break;
		default:
			statusMsg = tr("Switched to play order Normal.");
			break;
	}

	toolPanel_->setStatusText(statusMsg);
	toolPanel_->setPlayOrder(playOrder);
}

void Quasar::playbackControllerStoppingPlayback(bool temporarily)
{
	// Set Panels
	if (!temporarily)
	{
		setActiveView(Quasar::ViewLastActivePlaylist);

		operationPanel_->setCurrentPlayTime(0);
		operationPanel_->setPlayPause(true);

		playInfo_->clear();
	}
}

void Quasar::playbackControllerPlaybackStopped(bool temporarily)
{
	if (!temporarily)
	{
		//playListView_->setActiveItem(NULL);
		//playListView_->setActiveItemPaused(false);
	}

	if (videoPlaybackWindow_->isVisible())
	{
		videoPlaybackWindow_->hide();
		videoPlaybackWindow_->reparent(this, QPoint(0, 0));
		videoPlaybackWindow_->resize(QSize(0, 0));
		videoPlaybackWindow_->unsetCursor();
	}

#ifdef QTOPIA
	if (!temporarily)
	{
		qPlatformServices.enableSuspend();

		if (qConfig.isBatteryPlusEnabled)
			g_BatteryPlus.setMode(g_BatteryPlus.initialMode());
	}
#endif
}

void Quasar::playbackControllerPlaybackStarted()
{
	Media *media = playbackController_->activeItem()->media();

	// Set Operation Panel
	operationPanel_->setPlayPause(false);
	operationPanel_->setTotalPlayTime(media->length());

	//playListView_->setActiveItemPaused(false);

	if (media->isVideo())
	{
#ifndef QTOPIA
		if (qConfig.isOverlayEnabled)
		{
#endif
			videoPlaybackWindow_->hide();
			videoPlaybackWindow_->reparent(0, QPoint(0, 0));
			videoPlaybackWindow_->resize(qApp->desktop()->size());
			videoPlaybackWindow_->setCursor(QCursor(Qt::blankCursor));
			videoPlaybackWindow_->showFullScreen();
#ifndef QTOPIA
		}
#endif
	}
	else if (media->isAudio())
	{
		playInfo_->setMediaInfo(*playbackController_->activeItem()->media());

		// Get current number
		int n = playbackController_->activeItemIndex() + 1;

		if (playListView_->aspectSource())
			playInfo_->setTrackNumbers(n, playListView_->aspectSource()->itemList()->count());
		else
			playInfo_->setTrackNumbers(n, 0);

		if (qConfig.automaticallySwitchToPlayInfoEnabled)
			setActiveView(Quasar::ViewPlayInfo);
	}

#ifdef QTOPIA
	if (qConfig.isBatteryPlusEnabled) {
		if (media->isVideo()) {
			g_BatteryPlus.setMode(qConfig.batteryPlusMode4Video);
		} else {
			g_BatteryPlus.setMode(qConfig.batteryPlusMode4Audio);
		}
	}

	qPlatformServices.disableSuspend();
#endif
}

void Quasar::playbackControllerCurrentPlayTime(int seconds)
{
	operationPanel_->setCurrentPlayTime(seconds);
	playInfo_->setCurrentPlayTime(seconds);
}

void Quasar::playbackControllerPlaybackPaused()
{
	operationPanel_->setPlayPause(true);
}

void Quasar::playbackControllerPlaybackUnpaused()
{
	operationPanel_->setPlayPause(false);
}

void Quasar::playbackControllerMediaPrefetched(PlayListItem *item)
{
	playInfo_->precacheCoverArt(*item->media());
}

void Quasar::playbackControllerProblemWithMedia(PlayListItem *item)
{
	TextViewDialog errDialog(tr("MPlayer Error Message"),
		 tr("MPlayer could not be started for the following reason:"),
		 item->media()->lastError(),
		 tr("Perhaps your format configuration is wrong.") + "\n" +
		 tr("Do you want to open the configuration dialog?"),
		 tr("Yes"),
		 tr("No"),
		 this
	);

	if (errDialog.exec())
		configuration(true);
}

void Quasar::playbackControllerPlayerInitializationFailed(PlayListItem *item)
{
	int result = QMessageBox::critical(this, tr("mplayer"),
			 tr("mplayer could not be executed or\nwas not found on your system.\nWould you like to specify the location\nof the mplayer binary on your system?"),
			 QMessageBox::Yes | QMessageBox::Default, QMessageBox::No | QMessageBox::Escape);

	if (result == QMessageBox::Yes)
		configuration(true);
}
