#include <math.h>
#include <qpainter.h>

#ifdef QT4
#include <classcompathack.h>
#endif

#include "coverartflow.h"
#include "debug.h"
#include "mediadatabase.h"
#include "sqlite3.h"

#include "compathack.h"

/* CoverArtFlow */

CoverArtFlow::CoverArtFlow(ReflectionImageProvider *imageProvider, LayeredPaintBox *parent)
	: 	ReflectionImageFlow(imageProvider, parent),
		mediaDatabase_(NULL),
		aspectSource_(NULL),
		playListView_(NULL),
		active_(false)
{
	setTransitionTime(500);
	setVerticalOffset(ReflectionHeight / 4);
	//setConstantVerticalOffset(-80);
	setReflectionHeight(ReflectionHeight);
	setSlideSize(QSize(SlideWidthHeight, SlideWidthHeight + ReflectionHeight));

	connect(&preloadImagesTimer_, SIGNAL(timeout()), this, SLOT(preloadImages()));
	connect(&focusAlbumTimer_, SIGNAL(timeout()), this, SLOT(focusCurrentAlbum()));
}

CoverArtFlow::~CoverArtFlow()
{
}

void CoverArtFlow::attachTo(PlayListAspectAlbumGroupBuilder *source)
{
	detach();

	if (source)
	{
		aspectSource_ = source;
		mediaDatabase_ = aspectSource_->playListAspect()->playList()->mediaDatabase();

		updateView();

		if (aspectSource_->selectedIndex() > 0)
			jumpTo(aspectSource_->selectedIndex());

		connect(aspectSource_, SIGNAL(updateView()), this, SLOT(updateView()));

		if (active_)
			aspectSource_->registerUser(this);
	}
}

void CoverArtFlow::detach()
{
	if (aspectSource_)
	{
		disconnect(aspectSource_, SIGNAL(updateView()), this, SLOT(updateView()));

		aspectSource_->unregisterUser(this);
		reset();

		mediaDatabase_ = NULL;
		aspectSource_ = NULL;
	}
}

void CoverArtFlow::setPlayListView(PlayListView *playList)
{
	if (playListView())
		disconnect(playListView(), SIGNAL(currentIndexChanged()), this, SLOT(playListCurrentChanged()));

	playListView_ = playList;

	if (playListView())
		connect(playListView(), SIGNAL(currentIndexChanged()), this, SLOT(playListCurrentChanged()));
}

void CoverArtFlow::playListCurrentChanged()
{
	if (aspectSource_ && playListView_ && itemCount() > 0)
	{
		PlayListItem *item = playListView_->currentPlayListItem();
		if (item)
			transitionTo(aspectSource_->groupIndexForPlayListID(item->id()));
		else
			transitionTo(0);
	}
}

void CoverArtFlow::transitionFinished()
{
	DENTERMETHOD;

	CustomPerspectiveImageFlow::transitionFinished();
	focusAlbumTimer_.start(500, true);

	DEXITMETHOD;
}

void CoverArtFlow::focusCurrentAlbum()
{
	if (!aspectSource_ || !playListView_) return;

	int itemIndex = floor(currentValue());
	int itemIndexViaPlaylist = -1;

	// Try to get the item index via the currently selected playlist item...
	PlayListItem *item = playListView_->currentPlayListItem();
	if (item)
		itemIndexViaPlaylist = aspectSource_->groupIndexForPlayListID(item->id());

	// Only select the first song if the particular group isn't selected yet
	if (itemIndex != itemIndexViaPlaylist)
	{
		// Make sure the album is visible in the list
		// This is a workaround to avoid having the first song
		// of this particular group as last item in the visible area...
		if (itemIndex < aspectSource_->groups().count() - 1)
			playListView()->makeItemVisibleByID(aspectSource_->groups().at(itemIndex + 1)->playlistID);
		else
			playListView()->makeItemVisibleByID(playListView()->aspectSource()->itemList()->last()->id());

		playListView()->makeItemVisibleByID(aspectSource_->groups().at(itemIndex)->playlistID, true);
	}
}

void CoverArtFlow::mouseDoubleClickEvent(QMouseEvent *e)
{
	if (!playListView_) return;

	focusCurrentAlbum();
	playListView()->triggerPlaybackIntent();
}

void CoverArtFlow::paintOverlay(QPainter &p, const QPixmap &pixmap)
{
	ReflectionImageFlow::paintOverlay(p, pixmap);

	if (!aspectSource_) return;

	p.setPen(QColor(255, 255, 255));
	p.setBrush(Qt::NoBrush);

	int itemIndex = floor(currentValue());

	if (itemIndex >= 0 && itemIndex < aspectSource_->groups().count())
	{
		GroupItem *item = aspectSource_->groups().at(itemIndex);

		//int boxHeight = floor((reflectionHeight() - verticalOffset()) * scale());
		int boxHeight = 40;
#ifdef QTOPIA
		int fontHeight = 15; // TODO: FIXME!
#else
		int fontHeight = p.fontInfo().pixelSize();
#endif

		QRect rect;

		if (boxHeight < fontHeight * 2)
		{
			rect = QRect(0, height() - fontHeight - 2 - 40, width(), fontHeight + 2);
			p.drawText(rect, Qt::AlignCenter, item->artist + " - " + item->album);
		}
		else
		{
			rect = QRect(0, height() - boxHeight - 40, width(), boxHeight);
			p.drawText(rect, Qt::AlignCenter, item->artist + "\n" + item->album);
		}
	}

	if (imageProvider() && imageProvider()->isWorking())
	{
		p.drawText(5, 25, "Loading...");
	}
}

void CoverArtFlow::setActive(bool value)
{
	active_= value;

	if (active_)
	{
		if (aspectSource_)
			aspectSource_->registerUser(this);

		updateView();
	}
	else
	{
		if (aspectSource_)
			aspectSource_->unregisterUser(this);

		imageProvider()->flushMemoryCache();
	}
}

void CoverArtFlow::updateView()
{
	if (!aspectSource_) return;

	DENTERMETHOD;

	setItemCount(aspectSource_->groups().count());

	if (playListView_)
	{
		PlayListItem *item = playListView_->currentPlayListItem();
		if (item)
			jumpTo(aspectSource_->groupIndexForPlayListID(item->id()));
		else
			jumpTo(0);
	}

	preloadImages();

	DEXITMETHOD;
}

QImage CoverArtFlow::getImage(int index)
{
	if (aspectSource_ && index >= 0 && index < aspectSource_->groups().count())
	{
		GroupItem *item = aspectSource_->groups().at(index);
		if (item)
		{
			QImage *image = imageProvider()->getImage(item->key, item);

			if (image != NULL)
				return *image;
			else
				return QImage();
		}
		else
			return QImage();
	}
	else
		return QImage();
}

void CoverArtFlow::preloadImages()
{
	if (!aspectSource_) return;

	imageProvider()->lock();

	QStringList keys;

	GroupItem *item;

	for (int i = 0; i < aspectSource_->groups().count(); ++i)
	{
		item = aspectSource_->groups().at(i);

		if (!imageProvider()->imageExists(item->key))
			imageProvider()->queue(item->key, item);

		keys.append(item->key);
	}

	imageProvider()->prefetchHint(keys);

	imageProvider()->unlock();
}

void CoverArtFlow::imageWasCached(CachedImage *image)
{
	ReflectionImageFlow::imageWasCached(image);

	if (!preloadImagesTimer_.isActive())
		preloadImagesTimer_.start(1000, true);
}

void CoverArtFlow::reset()
{
	imageProvider()->resetNullImageNameList();
	imageProvider()->flushMemoryCache();
}
