/*
 * Copyright (C) 2006-2007	Andre Beckedorf
 * 					 		<evilJazz _AT_ katastrophos _DOT_ net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <qstringlist.h>
#include <qlabel.h>
#include <qtimer.h>
#include <qapplication.h>

#ifdef QT4
#include <q3listbox.h>
#include <q3hbox.h>
#include <q3vbox.h>
#include <QShowEvent>
#include <Q3Frame>
#include <QKeyEvent>
#include <QEvent>
#else
#include <qlistbox.h>
#include <qhbox.h>
#include <qvbox.h>
#endif

#include "playlistoverview.h"
#include "playlistmanager.h"
#include "playlistview.h"
#include "configuration.h"
#include "mediadatabase.h"
#include "skin.h"

#include "debug.h"

PlayListOverview::PlayListOverview(QWidget *parent)
	: QHBox(parent),
	mediaDatabase_(NULL),
	aspectSource_(NULL),
	updatingGenres_(false),
	updatingArtists_(false),
	updatingAlbums_(false)
{

	createColumnBox("Genres", genreBox_, genreHeading_, genreListBox_);
	connect(genreListBox_, SIGNAL(selectionChanged()),
			this, SLOT(genreListBoxSelectionChanged()));
	genreListBox_->installEventFilter(this);

	// black line separator between list boxes...
	QWidget *dummywidget = new QWidget(this);
	dummywidget->setFixedWidth(1);
	dummywidget->setBackgroundColor(QColor("#000000"));

	createColumnBox("Artists", artistBox_, artistHeading_, artistListBox_);
	connect(artistListBox_, SIGNAL(selectionChanged()),
			this, SLOT(artistListBoxSelectionChanged()));
	artistListBox_->installEventFilter(this);

	// black line separator between list boxes...
	dummywidget = new QWidget(this);
	dummywidget->setFixedWidth(1);
	dummywidget->setBackgroundColor(QColor("#000000"));

	createColumnBox("Albums", albumBox_, albumHeading_, albumListBox_);
	connect(albumListBox_, SIGNAL(selectionChanged()),
			this, SLOT(albumListBoxSelectionChanged()));
	albumListBox_->installEventFilter(this);

#ifdef QT4
	this->setFocusPolicy(Qt::StrongFocus);
#else
	this->setFocusPolicy(StrongFocus);
#endif
	this->setFocusProxy(genreListBox_);
}

PlayListOverview::~PlayListOverview()
{
}

void PlayListOverview::attachTo(PlayListAspectOverviewFilter *source)
{
	detach();

	if (source)
	{
		aspectSource_ = source;
		mediaDatabase_ = aspectSource_->playListAspect()->playList()->mediaDatabase();

		genreListUpdated();
		artistListUpdated();
		albumListUpdated();

		connect(aspectSource_, SIGNAL(updatedGenreList()), this, SLOT(genreListUpdated()));
		connect(aspectSource_, SIGNAL(updatedArtistList()), this, SLOT(artistListUpdated()));
		connect(aspectSource_, SIGNAL(updatedAlbumList()), this, SLOT(albumListUpdated()));

		if (isVisible())
			aspectSource_->registerUser(this);
	}
}

void PlayListOverview::detach()
{
	if (aspectSource_)
	{
		disconnect(aspectSource_, SIGNAL(updatedAlbumList()), this, SLOT(albumListUpdated()));
		disconnect(aspectSource_, SIGNAL(updatedArtistList()), this, SLOT(artistListUpdated()));
		disconnect(aspectSource_, SIGNAL(updatedGenreList()), this, SLOT(genreListUpdated()));

		albumListBox_->clear();
		artistListBox_->clear();
		genreListBox_->clear();

		aspectSource_->unregisterUser(this);

		mediaDatabase_ = NULL;
		aspectSource_ = NULL;
	}
}

void PlayListOverview::genreListUpdated()
{
	if (!aspectSource_) return;
	doUpdateOnListBox(genreListBox_, aspectSource_->genreList());
}

void PlayListOverview::artistListUpdated()
{
	if (!aspectSource_) return;
	doUpdateOnListBox(artistListBox_, aspectSource_->artistList());
}

void PlayListOverview::albumListUpdated()
{
	if (!aspectSource_) return;
	doUpdateOnListBox(albumListBox_, aspectSource_->albumList());
}

void PlayListOverview::createColumnBox(QString title, QVBox *&box, QLabel *&heading, QListBox *&listbox)
{
	box = new QVBox(this);
	heading = new QLabel(box);
#ifdef QT4
	heading->setAlignment(Qt::AlignCenter | Qt::AlignVCenter | Qt::TextExpandTabs);
#else
	heading->setAlignment(Qt::AlignCenter | Qt::AlignVCenter | Qt::ExpandTabs);
#endif
	heading->setText(title);

	listbox = new QListBox(box);
	listbox->setFrameStyle(QFrame::NoFrame);
	listbox->setSelectionMode(QListBox::Multi);
	listbox->setHScrollBarMode(QScrollView::AlwaysOff);
}

void PlayListOverview::setSkinMode(const SkinModeInformation &modeInfo)
{
	if (modeInfo.playListOverviewFont().font.isUsed())
	{
		genreListBox_->setFont(modeInfo.playListOverviewFont().font);
		artistListBox_->setFont(modeInfo.playListOverviewFont().font);
		albumListBox_->setFont(modeInfo.playListOverviewFont().font);
	}
	else
	{
		genreListBox_->unsetFont();
		artistListBox_->unsetFont();
		albumListBox_->unsetFont();
	}

	// set general palette:
	QPalette palette = QApplication::palette();

	if (modeInfo.playListOverviewFont().fontColor.isUsed())
	{
		palette.setColor(QColorGroup::Text, modeInfo.playListOverviewFont().fontColor);
		palette.setColor(QColorGroup::Foreground, modeInfo.playListOverviewFont().fontColor);
	}

	if (modeInfo.playListOverviewBackgroundColorSelected().isUsed())
	{
		palette.setColor(QColorGroup::Highlight, modeInfo.playListOverviewBackgroundColorSelected());
		palette.setColor(QColorGroup::HighlightedText, modeInfo.playListOverviewFontColorSelected());
	}

	// set even and odd colors for items:
	if (modeInfo.playListOverviewBackgroundColor().isUsed())
		palette.setColor(QColorGroup::Base, modeInfo.playListOverviewBackgroundColor());

	genreListBox_->setPalette(palette);
	artistListBox_->setPalette(palette);
	albumListBox_->setPalette(palette);

	if (modeInfo.playListOverviewHeaderFont().font.isUsed())
	{
		genreHeading_->setFont(modeInfo.playListOverviewHeaderFont().font);
		artistHeading_->setFont(modeInfo.playListOverviewHeaderFont().font);
		albumHeading_->setFont(modeInfo.playListOverviewHeaderFont().font);
	}
	else
	{
		genreHeading_->unsetFont();
		artistHeading_->unsetFont();
		albumHeading_->unsetFont();
	}

	if (modeInfo.playListOverviewHeaderColor().isUsed())
	{
		QPalette palette(modeInfo.playListOverviewHeaderColor());

		if (modeInfo.playListOverviewHeaderFont().fontColor.isUsed())
		{
			palette.setColor(QColorGroup::Text, modeInfo.playListOverviewHeaderFont().fontColor);
			palette.setColor(QColorGroup::ButtonText, modeInfo.playListOverviewHeaderFont().fontColor);
			palette.setColor(QColorGroup::Foreground, modeInfo.playListOverviewHeaderFont().fontColor);
		}

		genreHeading_->setPalette(palette);
		artistHeading_->setPalette(palette);
		albumHeading_->setPalette(palette);
	}
	else
	{
		genreHeading_->unsetPalette();
		artistHeading_->unsetPalette();
		albumHeading_->unsetPalette();
	}

	if (modeInfo.playListOverviewScrollBarColor().isUsed())
	{
		genreListBox_->horizontalScrollBar()->setPalette(QPalette(modeInfo.playListOverviewScrollBarColor()));
		genreListBox_->verticalScrollBar()->setPalette(QPalette(modeInfo.playListOverviewScrollBarColor()));
		artistListBox_->horizontalScrollBar()->setPalette(QPalette(modeInfo.playListOverviewScrollBarColor()));
		artistListBox_->verticalScrollBar()->setPalette(QPalette(modeInfo.playListOverviewScrollBarColor()));
		albumListBox_->horizontalScrollBar()->setPalette(QPalette(modeInfo.playListOverviewScrollBarColor()));
		albumListBox_->verticalScrollBar()->setPalette(QPalette(modeInfo.playListOverviewScrollBarColor()));
	}
	else
	{
		genreListBox_->horizontalScrollBar()->unsetPalette();
		genreListBox_->verticalScrollBar()->unsetPalette();
		artistListBox_->horizontalScrollBar()->unsetPalette();
		artistListBox_->verticalScrollBar()->unsetPalette();
		albumListBox_->horizontalScrollBar()->unsetPalette();
		albumListBox_->verticalScrollBar()->unsetPalette();
	}
}

void PlayListOverview::setSelectionMode(QListBox::SelectionMode mode)
{
	genreListBox_->setSelectionMode(mode);
	artistListBox_->setSelectionMode(mode);
	albumListBox_->setSelectionMode(mode);
}

void PlayListOverview::showEvent(QShowEvent *e)
{
	if (!e->spontaneous())
	{
		if (aspectSource_)
		{
			aspectSource_->registerUser(this);
			aspectSource_->update();
		}
	}
}

void PlayListOverview::hideEvent(QHideEvent *e)
{
	if (!e->spontaneous())
	{
		if (aspectSource_)
			aspectSource_->unregisterUser(this);
	}
}

bool PlayListOverview::eventFilter(QObject *object, QEvent *event)
{
	if (event->type() == QEvent::KeyPress)
	{
		QKeyEvent *e = static_cast<QKeyEvent *>(event);
		int key = e->key();

		QWidget *newFocusedWidget = NULL;

		if (object == genreListBox_)
		{
			if (key == Qt::Key_Right)
				newFocusedWidget = artistListBox_;
			else if (key == Qt::Key_Left)
				emit focusOutLeft();
		}
		else if (object == artistListBox_)
		{
			if (key == Qt::Key_Right)
				newFocusedWidget = albumListBox_;
			else if (key == Qt::Key_Left)
				newFocusedWidget = genreListBox_;
		}
		else if (object == albumListBox_)
		{
			if (key == Qt::Key_Right)
				emit focusOutRight();
			else if (key == Qt::Key_Left)
				newFocusedWidget = artistListBox_;
		}

		if (newFocusedWidget)
		{
			newFocusedWidget->setFocus();
			e->accept();
			return true;
		}

		return QHBox::eventFilter(object, event);
	}

	if (event->type() == QEvent::FocusIn)
		setFocusProxy(static_cast<QWidget *>(object));

	return QHBox::eventFilter(object, event);
}

int PlayListOverview::selectionCount(QListBox *list)
{
	int selectedCount = 0;

	for (int i = 1; i < list->count(); ++i)
		if (list->isSelected(i))
			++selectedCount;

	return selectedCount;
}

void PlayListOverview::beginUpdateOn(QWidget *widget)
{
	widget->setUpdatesEnabled(false);
	widget->blockSignals(true);
}

void PlayListOverview::endUpdateOn(QWidget *widget)
{
	widget->blockSignals(false);
	widget->setUpdatesEnabled(true);
}

void PlayListOverview::doUpdateOnListBox(QListBox *listbox, OverviewFilterList &list)
{
	DENTERMETHOD;

	beginUpdateOn(listbox);

	listbox->clearSelection();
	listbox->clear();

	for (int i = 0; i < list.count(); ++i)
	{
		OverviewFilterListEntry *entry = list.at(i);

		listbox->insertItem(entry->text);
		listbox->setSelected(listbox->count() - 1, entry->selected);
	}

	endUpdateOn(listbox);

	DEXITMETHOD;
}

void PlayListOverview::validateSelection(QListBox *list)
{
	if (list->currentItem() == 0 && list->isSelected(0) || selectionCount(list) == 0)
	{
		list->clearSelection();
		list->setSelected(0, TRUE);
		list->setCurrentItem(0);
	}
	else
		list->setSelected(0, FALSE);
}

void PlayListOverview::genreListBoxSelectionChanged()
{
	if (!aspectSource_) return;

	DENTERMETHOD;
	if (updatingGenres_)
	{
		DEXITMETHOD;
		return;
	}
	updatingGenres_ = true;
	validateSelection(genreListBox_);

	transferSelectionToList(aspectSource_->genreList(), genreListBox_);
	aspectSource_->updateFromGenreList();

	updatingGenres_ = false;
	DEXITMETHOD;
}

void PlayListOverview::artistListBoxSelectionChanged()
{
	if (!aspectSource_) return;

	DENTERMETHOD;
	if (updatingArtists_)
	{
		DEXITMETHOD;
		return;
	}
	updatingArtists_ = true;
	validateSelection(artistListBox_);

	transferSelectionToList(aspectSource_->artistList(), artistListBox_);
	aspectSource_->updateFromArtistList();

	updatingArtists_ = false;
	DEXITMETHOD;
}

void PlayListOverview::albumListBoxSelectionChanged()
{
	if (!aspectSource_) return;

	DENTERMETHOD;
	if (updatingAlbums_)
	{
		DEXITMETHOD;
		return;
	}
	updatingAlbums_ = true;
	validateSelection(albumListBox_);

	transferSelectionToList(aspectSource_->albumList(), albumListBox_);
	aspectSource_->updateFromAlbumList();

	updatingAlbums_ = false;
	DEXITMETHOD;
}

void PlayListOverview::transferSelectionToList(OverviewFilterList &list, QListBox *listbox)
{
	for (int i = 0; i < listbox->count(); ++i)
		list.at(i)->selected = listbox->isSelected(i);
}

/*
void PlayListOverview::reset()
{
	albumListBox_->selectAll(false);
	artistListBox_->selectAll(false);
	genreListBox_->selectAll(false);
}
*/
