/*
 * Copyright (C) 2006-2007	Andre Beckedorf
 * 					 		<evilJazz _AT_ katastrophos _DOT_ net>
 *
 * The few remaining parts in this almost completely rewritten file are
 * Copyright (C) 2005 Atmark <atmarkat _AT_ msn _DOT_ com>
 *                    AGAWA Koji <i _AT_ atty _DOT_ jp>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <cstdlib>
#include <ctime>
#include <time.h>

#include <qstring.h>
#include <qapplication.h>
#include <qdatetime.h>

#ifdef QT4
#include <classcompathack.h>
#else
#ifdef QTOPIA
#include <qpe/config.h>
#include "compat/sharp-qtopia/qt/qlistview.h"
#include <qtextstream.h>
#include <qheader.h>
#else
#include <config.h>
#include <qtextstream.h>
#include <qheader.h>
#include <qlistview.h>
#endif
#endif

#include <qpainter.h>
#include <qimage.h>
#include "mediadatabase.h"

#include "media.h"
#include "playlistview.h"
#include "playbackcontroller.h"
#include "debug.h"
#include "skinmanager.h"
#include "sqlite3.h"

#include "compathack.h"

/* PlayListViewItem */

PlayListViewItem::PlayListViewItem(PlayListItem *item, int index, QListView *parent)
	: QListViewItem(parent),
	  item_(item),
	  index_(index)
{
}

PlayListViewItem::PlayListViewItem(PlayListItem *item, int index, QListView *parent, QListViewItem *after)
	: QListViewItem(parent, after),
	  item_(item),
	  index_(index)
{
}

QString PlayListViewItem::key(int column, bool) const
{
	// omit sorting. We'll do it in our DB backend...
	return NULL;
}

void PlayListViewItem::paintFocus(QPainter *, const QColorGroup &, const QRect &)
{
	// don't paint focus rect...
}

void PlayListViewItem::paintCell(QPainter * p, const QColorGroup & cg, int column, int width, int align )
{
    if (!p)
    	return;

	int ownHeight = height();
	PlayListView *lv = static_cast<PlayListView *>(listView());

	int vy = lv->itemRect(this).y();   // get y inside the viewport
	int x, y;

	lv->viewportToContents(0, vy, x, y);  // get y inside the contents

	QColorGroup grp;

	if (ownHeight != 0 && (y / ownHeight) % 2 == 0)
		grp = lv->evenRowColorGroup_;
	else
		grp = lv->oddRowColorGroup_;

	p->fillRect( 0, 0, width, ownHeight, grp.brush(QColorGroup::Base));

	int r = 1;
	int marg = 1;

	if (isSelected())
	{
		p->fillRect(r - marg, 0, width - r + marg, ownHeight,
					cg.brush(QColorGroup::Highlight));

		p->setPen(cg.highlightedText());
	}
	else
	{
		p->setPen(cg.text());
	}

	if (column == 0)
	{
		if (this == lv->activeItem_)
		{
			const QPixmap *indicator = lv->playbackController_->isPlaying() ? lv->playIndicatorImage_ : lv->pauseIndicatorImage_;
			if (indicator)
			{
				int t = (ownHeight - indicator->height()) / 2;
				p->drawPixmap(r, t, *indicator);
				r += indicator->width();
			}
		}

		if (this->media()->errorDetected() && lv->errorIndicatorImage_)
		{
			const QPixmap *indicator = lv->errorIndicatorImage_;
			int t = (ownHeight - indicator->height()) / 2;
			p->drawPixmap(r, t, *indicator);
		}
	}
	else
	{
		QString t;
		Media *currentMedia = media();

		switch (column)
		{
		case 0:
			t = "    ";
			break;
		case 1:
			t = currentMedia->trackNo() + "  ";
			break;
		case 2:
			t = currentMedia->title();
			break;
		case 3:
			t = currentMedia->artist();
			break;
		case 4:
			t = currentMedia->album();
			break;
		case 5:
			t = currentMedia->playTime();
			break;
		}

		if (!t.isEmpty())
			p->drawText(r, 0, width - marg - r, ownHeight, align, t);
	}
}

int PlayListViewItem::width( const QFontMetrics& fm, const QListView* lv, int c ) const
{
	return lv->columnWidth(c);
}

void PlayListViewItem::setSelected(bool selected)
{
	QListViewItem::setSelected(selected);
	static_cast<PlayListView *>(this->listView())->itemSelectedStateChanged(this, selected);
}

/* PlayListView */

PlayListView::PlayListView(QWidget *parent)
	: KineticListView(parent),
		aspectSource_(NULL),
		playbackController_(NULL),
		updateCount_(0),
		userIsResizingSection_(false),
		autoResizingSections_(false),
		evenRowColorGroup_(colorGroup()),
		oddRowColorGroup_(colorGroup()),
		playIndicatorImage_(NULL),
		pauseIndicatorImage_(NULL),
		errorIndicatorImage_(NULL),
		activeItem_(NULL),
		inputMode_(Select),
		itemOrderDirty_(false)
{
	// set to manual width mode.
	// this boosts performance quite dramatically as the
	// column width doesn't need to be measured
	addColumn(tr("   "), 20);
	setColumnAlignment(addColumn(tr("No."), 100), Qt::AlignRight);
	addColumn(tr("Title"), 100);
	addColumn(tr("Artist"), 100);
	addColumn(tr("Album"), 100);
	setColumnAlignment(addColumn(tr("Length"), 40), Qt::AlignRight);

	setRootIsDecorated(false);
	setAllColumnsShowFocus(true);

#ifdef QTOPIA
	setWFlags(getWFlags() | Qt::WNorthWestGravity | Qt::WRepaintNoErase | Qt::WResizeNoErase);
#else
#ifndef QT4
	setWFlags(getWFlags() | Qt::WStaticContents | Qt::WNoAutoErase);
#endif
#endif

	setInputMode(PlayListView::Select);

	QListView::setSorting(-1);

	header()->setClickEnabled(true, 1);
	header()->setClickEnabled(true, 2);
	header()->setClickEnabled(true, 3);
	header()->setClickEnabled(true, 4);
	header()->setClickEnabled(true, 5);

	header()->setResizeEnabled(false, 0);

	connect(header(), SIGNAL(indexChange(int, int, int)),
			this, SLOT(headerIndexChange(int, int, int)));

	connect(header(), SIGNAL(clicked(int)),
			this, SLOT(headerClicked(int)));

	connect(header(), SIGNAL(sizeChange(int, int, int)),
			this, SLOT(headerSizeChange(int, int, int)));

	setFrameStyle(QFrame::NoFrame);

	connect(this, SIGNAL(doubleClicked(QListViewItem *)), this, SIGNAL(playbackIntent()));
	connect(this, SIGNAL(selectionChanged()), this, SLOT(updateSelectedPlayListItem()));

	connect(this, SIGNAL(currentChanged(QListViewItem *)), this, SIGNAL(currentIndexChanged()));

	setAutoScaleColumnsEnabled(true);
}

PlayListView::~PlayListView()
{
	QListView::clear();
}

void PlayListView::attachTo(PlayListAspectListViewRenderer *source, PlaybackController *controller)
{
	detach();

	if (source && controller)
	{
		aspectSource_ = source;
		playbackController_ = controller;

		connect(aspectSource_, SIGNAL(updateView()), this, SLOT(updateView()));
		connect(aspectSource_, SIGNAL(graphExecutionStarted(bool &)), this, SLOT(graphExecutionStarted(bool &)));

		connect(this, SIGNAL(playbackIntent()), playbackController_, SLOT(stopCurrentPlaybackAndPlayCurrentItem()));
		connect(playbackController_, SIGNAL(activeItemChanged(int, int)), this, SLOT(playbackControllerActiveItemChanged(int, int)));

		header()->setSortIndicator(aspectSource_->sortedColumn(), !aspectSource_->isSortAscending());

		updateView();

		setCurrentItem(findItemByIndex(aspectSource_->selectedItemIndex()));
		setContentsPos(aspectSource_->viewportPosition().x(), aspectSource_->viewportPosition().y());
	}
}

void PlayListView::detach()
{
	if (aspectSource_ && playbackController_)
	{
		ensurePlayListOrderIsValid();

		aspectSource_->setSelectedItemIndex(currentItem() ? itemIndex(currentItem()) : -1);
		aspectSource_->setViewportPosition(QPoint(contentsX(), contentsY()));

		disconnect(playbackController_, SIGNAL(activeItemChanged(int, int)), this, SLOT(playbackControllerActiveItemChanged(int, int)));
		disconnect(this, SIGNAL(playbackIntent()), playbackController_, SLOT(stopCurrentPlaybackAndPlayCurrentItem()));

		disconnect(aspectSource_, SIGNAL(graphExecutionStarted(bool &)), this, SLOT(graphExecutionStarted(bool &)));
		disconnect(aspectSource_, SIGNAL(updateView()), this, SLOT(updateView()));

		activeItem_ = NULL;
		selectedItems_.clear();
		QListView::clear();
		itemOrderDirty_ = false;

		playbackController_ = NULL;
		aspectSource_ = NULL;
	}
}

void PlayListView::playbackControllerActiveItemChanged(int newActiveItemIndex, int oldActiveItemIndex)
{
	PlayListViewItem *lastActiveItem = activeItem_;
	activeItem_ = findItemByIndex(newActiveItemIndex);

	if (lastActiveItem)
		lastActiveItem->repaint();

	if (activeItem_)
	{
		ensureItemVisible(activeItem_);
		activeItem_->repaint();
	}
}

void PlayListView::updateSelectedPlayListItem()
{
	if (aspectSource_)
	{
		if (currentItem())
			aspectSource_->setSelectedItemIndex(static_cast<PlayListViewItem *>(currentItem())->index());
		else
			aspectSource_->setSelectedItemIndex(-1);
	}
}

void PlayListView::beginUpdate()
{
	DENTERMETHOD;

	if (updateCount_ == 0)
	{
		viewport()->setUpdatesEnabled(false);
		setUpdatesEnabled(false);
		blockSignals(true);
	}
	++updateCount_;

	DEXITMETHOD;
}

void PlayListView::endUpdate()
{
	DENTERMETHOD;

	--updateCount_;
	if (updateCount_ == 0)
	{
		DTIMERINIT(time);
		blockSignals(false);
		setUpdatesEnabled(true);
		viewport()->setUpdatesEnabled(true);
		triggerUpdate();
		DTIMERPRINT(time, "list view refresh");
	}

	DEXITMETHOD;
}

void PlayListView::ensurePlayListOrderIsValid()
{
	if (!aspectSource_) return;

	DENTERMETHOD;
	if (!aspectSource_->isGraphFiltered() && itemOrderDirty_)
	{
		DPRINTF("playlist item order is dirty, we have to save the order to the DB...");
		beginUpdate();

		PlayListViewItem *item;
		PlayListItemList itemList;
#ifndef QT4
		itemList.setAutoDelete(false);
#endif
		QListViewItemIterator it(this);
		while (it.current())
		{
			item = static_cast<PlayListViewItem *>(it.current());
			itemList.append(item->item_);
			++it;
		}

		if (itemList.count() > 0)
			aspectSource_->setItemList(itemList);

		itemOrderDirty_ = false;
		itemList.clear();

		endUpdate();
	}
	DEXITMETHOD;
}

void PlayListView::graphExecutionStarted(bool &stopIt)
{
	DENTERMETHOD;

	if (itemOrderDirty_)
	{
		ensurePlayListOrderIsValid();
		stopIt = true;
	}

	DEXITMETHOD;
}

void PlayListView::updateView()
{
	DENTERMETHOD;

	beginUpdate();

	QListView::clear();

	header()->setSortIndicator(aspectSource_->sortedColumn(), !aspectSource_->isSortAscending());

	PlayListViewItem *lastNewItem = NULL;
	PlayListItemList *items = aspectSource_->itemList();

	activeItem_ = NULL;
	PlayListViewItem *selectedItem = NULL;

	for (int i = 0; i < items->count(); ++i)
	{
		lastNewItem = new PlayListViewItem(items->at(i), i, this, lastNewItem);

		if (!activeItem_ && i == playbackController_->activeItemIndex())
			activeItem_ = lastNewItem;

		if (!selectedItem && i == aspectSource_->selectedItemIndex())
			selectedItem = lastNewItem;
	}

	if (selectedItem)
	{
		setCurrentItem(selectedItem);
		ensureItemVisible(selectedItem);
	}

	endUpdate();

	if (autoScaleColumnsEnabled_)
		setAutoSectionSizes();

	DEXITMETHOD;
}

void PlayListView::setInputMode(PlayListView::InputMode inputmode)
{
	switch (inputmode)
	{
	case PlayListView::Select:
		clearSelection();
		setSelectionMode(QListView::Single);
		if (currentItem())
			currentItem()->setSelected(true);
		break;
	case PlayListView::MultiSelect:
	case PlayListView::Move:
		setSelectionMode(QListView::Extended);
		break;
	}

	inputMode_ = inputmode;
}

MediaIDList PlayListView::getSelectedMediaIDs()
{
	MediaIDList list;
	for (int i = 0; i < selectedItems_.count(); ++i)
		list.append(selectedItems_.at(i)->mediaID());
	return list;
}

MediaList PlayListView::getSelectedMedia()
{
	MediaList list;
	for (int i = 0; i < selectedItems_.count(); ++i)
		list.append(selectedItems_.at(i)->media());
	return list;
}

PlayListIDList PlayListView::getSelectedItemIDs()
{
	PlayListIDList list;
	for (int i = 0; i < selectedItems_.count(); ++i)
		list.append(selectedItems_.at(i)->id());
	return list;
}

PlayListViewItemList & PlayListView::selectedItems()
{
	return selectedItems_;
}

void PlayListView::removeSelectedItems()
{
	if (selectedItems_.count() > 0)
	{
		if (itemOrderDirty_)
		{
			for (int i = selectedItems_.count() - 1; i > -1; --i)
			{
				PlayListViewItem *item = selectedItems_.at(i);
				selectedItems_.removeAt(i);
				delete item;
			}
		}
		else
		{
			// important to get the selected IDs here, since beginChangingUpdate
			// will clear the list and thus the selection...
			PlayListIDList itemIDs = getSelectedItemIDs();
			aspectSource_->playListAspect()->playList()->deletePlayListIDs(itemIDs);
		}
	}
}

void PlayListView::makeItemVisibleByID(unsigned long playListID, bool selectItem)
{
	PlayListViewItem *item = findItemByID(playListID);
	if (item)
	{
		ensureItemVisible(item);
		if (selectItem)
			setCurrentItem(item);
	}
}

void PlayListView::insertMediaIDsAtCurrentIndex(const MediaIDList &srcMediaIDs)
{
	// if the playlist is not sorted or filtered, then we can
	// directly insert the new items at the current position...
	if (!aspectSource()->isGraphFiltered())
	{
		PlayListViewItem *lastPlayListViewItem = NULL;
		bool moveFirstItem = false;

		// is anything selected?
		if (selectedItems().count() > 0)
		{
			// if yes, take the first item as place to insert...
			lastPlayListViewItem = selectedItems().at(0);
			DHOP(lastPlayListViewItem->media()->dumpInfo());

			// since we want to place the new items at the position/index of
			// the previously selected insertion point, we need to get the item
			// above it, because we can only tell QListView to insert after
			// a given item...
			if (lastPlayListViewItem->itemAbove())
				lastPlayListViewItem = static_cast<PlayListViewItem *>(lastPlayListViewItem->itemAbove());
			else if (lastPlayListViewItem == firstItem())
				// if there is no item above, this means we have selected the first
				// child in the list. Do a sanity check and then mark the item separately
				// so we can move it to the end of the new stuff after our items have
				// been inserted...
				moveFirstItem = true;
		}
		else
			// insert at end of list if nothing is selected or the playlist is empty...
			lastPlayListViewItem = lastItem();

		clearSelection();

		PlayListItemList newItems;
		aspectSource_->playListAspect()->playList()->appendMediaIDs(srcMediaIDs, &newItems);

		for (int i = 0; i < newItems.count(); ++i)
		{
			lastPlayListViewItem = new PlayListViewItem(newItems.at(i), 0, this, lastPlayListViewItem);
			setSelected(lastPlayListViewItem, true);
		}

		// if we have a marked an item that was (and still is) the first child in the list
		// move it to the end of our inserted items...
		if (moveFirstItem)
			firstItem()->moveItem(lastPlayListViewItem);

		itemOrderDirty_ = true;
	}
	// the playlist is sorted or filtered, so insert at DB level...
	else
		aspectSource_->playListAspect()->playList()->appendMediaIDs(srcMediaIDs);
}

void PlayListView::headerIndexChange(int section, int fromIndex, int toIndex)
{
	// don't allow moving the first column...
	if (header()->mapToIndex(0) != 0)
	{
		header()->moveSection(0, 0); // make sure, blank column is always first.
		header()->update();
	}

	updateResizabilityOfSections();
}

void PlayListView::updateDynamicSectionSizes()
{
	float totalScalingSectionSize = 0;

	totalScalingSectionSize += header()->sectionSize(2);
	totalScalingSectionSize += header()->sectionSize(3);
	totalScalingSectionSize += header()->sectionSize(4);

	headerSectionSizes_.clear();

	headerSectionSizes_.append((float)header()->sectionSize(2) / totalScalingSectionSize * 100);
	headerSectionSizes_.append((float)header()->sectionSize(3) / totalScalingSectionSize * 100);
	headerSectionSizes_.append((float)header()->sectionSize(4) / totalScalingSectionSize * 100);
}

void PlayListView::headerSizeChange(int section, int oldSize, int newSize)
{
	if (!autoScaleColumnsEnabled_ || userIsResizingSection_ || autoResizingSections_)
		return;

	qDebug("PlayListView::headerSizeChange(int section, int oldSize, int newSize): %d, %d, %d", section, oldSize, newSize);

	userIsResizingSection_ = true;

	int index = header()->mapToIndex(section);
	int sizeDiff = newSize - oldSize;

	int nextIndex = index + 1;
	int neighborSection = 0;
	while (nextIndex < header()->count())
	{
		neighborSection = header()->mapToSection(nextIndex);
		if (neighborSection == 2 || neighborSection == 3 || neighborSection == 4 || nextIndex == header()->count() - 1)
			break;

		neighborSection = 0;
		++nextIndex;
	}

	if (neighborSection != 0)
		setColumnWidth(neighborSection, columnWidth(neighborSection) - sizeDiff);

	updateDynamicSectionSizes();

	userIsResizingSection_ = false;
}

void PlayListView::setAutoScaleColumnsEnabled(bool value)
{
	autoScaleColumnsEnabled_ = value;

	if (autoScaleColumnsEnabled_)
	{
		setHScrollBarMode(QListView::AlwaysOff);
		setAutoSectionSizes();
	}
	else
	{
		updateResizabilityOfSections();
		setHScrollBarMode(QListView::Auto);
	}
}

void PlayListView::headerClicked(int section)
{
	if (!aspectSource_) return;

	DENTERMETHOD;
	if (section != aspectSource_->sortedColumn())
		aspectSource_->setSortParams(section, true);
	else
		aspectSource_->setSortParams(aspectSource_->sortedColumn(), !aspectSource_->isSortAscending());

	// aspectSource_->setSortParams(...) will cause aspectSource_->execute();

	DPRINTF("Header section %d clicked. Preparing playlist item index...", section);
	DEXITMETHOD;
}

void PlayListView::setSorting(int column, bool ascending)
{
	// omit sorting. We'll do it in our DB backend...
	// NOTE: This is a no-op stub, because the ListView widget calls it internally
}

PlayListViewItem *PlayListView::firstItem()
{
	return static_cast<PlayListViewItem *>(firstChild());
}

PlayListViewItem *PlayListView::lastItem()
{
	if (childCount() < 1)
		return NULL;

	PlayListViewItem *item = activeItem_;

	if (!item)
		item = firstItem();

	if (!item)
		return NULL;

	while (item->nextSibling())
		item = static_cast<PlayListViewItem *>(item->nextSibling());

	return item;
}

void PlayListView::keyPressEvent(QKeyEvent *e)
{
	if (e->key() == Qt::Key_Delete && currentItem())
	{
		takeItem(currentItem());
		e->accept();
		return;
	}

	QListView::keyPressEvent(e);
}

void PlayListView::updateResizabilityOfSections()
{
	for (int i = 0; i < columns(); ++i)
		header()->setResizeEnabled(true, header()->mapToSection(i));

	header()->setResizeEnabled(false, 0);

	if (autoScaleColumnsEnabled_)
		header()->setResizeEnabled(false, header()->mapToSection(columns() - 1));
}

void PlayListView::setAutoSectionSizes()
{
	if (autoResizingSections_)
		return;

	autoResizingSections_ = true;

	if (headerSectionSizes_.count() == 0)
		updateDynamicSectionSizes();

	float totalSectionsWidth = header()->width();
	totalSectionsWidth -= header()->sectionSize(0);
	totalSectionsWidth -= header()->sectionSize(1);
	totalSectionsWidth -= header()->sectionSize(5);

	setColumnWidth(2, (int)(totalSectionsWidth * headerSectionSizes_[0] / 100));
	setColumnWidth(3, (int)(totalSectionsWidth * headerSectionSizes_[1] / 100));
	setColumnWidth(4, (int)(totalSectionsWidth * headerSectionSizes_[2] / 100));

	updateResizabilityOfSections();

	autoResizingSections_ = false;
}

void PlayListView::resizeEvent(QResizeEvent *e)
{
	QListView::resizeEvent(e);

	if (autoScaleColumnsEnabled_)
		setAutoSectionSizes();
}

void PlayListView::contentsMousePressEvent(QMouseEvent *e)
{
	if (inputMode() == PlayListView::Move)
	{
		buttonDown_ = true;
		moveTriggered_ = false;
		buttonDownPos_ = e->pos();
	}
	else
	{
		QListView::contentsMousePressEvent(e);

		if (e->button() == Qt::RightButton && popupMenu_)
			popupMenu_->exec(e->globalPos());
	}
}

void PlayListView::contentsMouseMoveEvent(QMouseEvent *e)
{
	if (inputMode() == PlayListView::Move)
	{
		if (buttonDown_ && !aspectSource_->isGraphFiltered())
		{
			if (moveTriggered_)
			{
			    QPoint vp = contentsToViewport(e->pos());
			    QListViewItem *i = itemAt(vp);
			    bool moveFirstChild = false;

			    if (i == firstChild())
			    {
			    	moveFirstChild = true;
			    }
			    else if (i)
			    {
			    	QListViewItem *iabove = i->itemAbove();
			    	if (iabove && !selectedItemsForMove_.contains(static_cast<PlayListViewItem *>(iabove)))
			    		i = iabove;
			    }
			    else
			    {
			    	i = firstChild();
			    	while (i->nextSibling())
			    		i = i->nextSibling();
			    }

			    if (!selectedItemsForMove_.contains(static_cast<PlayListViewItem *>(i)))
			    {
				    PlayListViewItem *pli;

				    for (int j = 0; j < selectedItemsForMove_.count(); ++j)
				    {
				    	pli = selectedItemsForMove_.at(j);
				    	pli->moveItem(i);
				    	i = pli;
				    }

				    if (moveFirstChild)
				    	firstChild()->moveItem(i);

				    itemOrderDirty_ = true;
			    }
			}
			else
			{
				if (e->pos().y() > buttonDownPos_.y() + 10 ||
					e->pos().y() < buttonDownPos_.y() - 10)
				{
					moveTriggered_ = true;
					selectedItemsForMove_ = selectedItems();
				}
			}

		    return;
		}
	}

	QListView::contentsMouseMoveEvent(e);
}

void PlayListView::contentsMouseReleaseEvent(QMouseEvent *e)
{
	buttonDown_ = false;
	moveTriggered_ = false;

	QListView::contentsMouseReleaseEvent(e);
}

PlayListViewItem *PlayListView::findItemByID(unsigned long playlistID)
{
	QListViewItemIterator it(this);
	for (; it.current() && static_cast<PlayListViewItem *>(it.current())->id() != playlistID; ++it)
		;

	return static_cast<PlayListViewItem *>(it.current());
}

PlayListViewItem *PlayListView::findItemByIndex(int index)
{
	if (this->childCount() <= index)
		return NULL;

	QListViewItemIterator it(this);
	for (int i = 0; it.current() && i != index; ++it, ++i);

	return static_cast<PlayListViewItem *>(it.current());
}

int PlayListView::itemIndex(QListViewItem* item)
{
	if (this->childCount() == 0)
		return -1;

	/*
	int i = 0;
	for (QListViewItemIterator it(this); it.current() && it.current() != item; ++it, ++i)
		;
	*/

	return static_cast<PlayListViewItem *>(item)->index();
}

void PlayListView::saveColumnOrder(const QString &configName, const QString &groupName)
{
	Config cfg(configName, Config::File);
	cfg.setGroup(groupName);

	for (int i = 0; i < columns(); ++i)
	{
		int section = header()->mapToSection(i);
		cfg.writeEntry("section_" + QString::number(i), QString::number(section));
	}
}

void PlayListView::saveColumnWidth(const QString &configName, const QString &groupName)
{
	Config cfg(configName, Config::File);
	cfg.setGroup(groupName);

	for (int i = 0; i < columns(); ++i)
		cfg.writeEntry("width_" + QString::number(i), columnWidth(header()->mapToSection(i)));
}

void PlayListView::loadColumnOrder(const QString &configName, const QString &groupName)
{
	Config cfg(configName, Config::File);
	cfg.setGroup(groupName);

	for (int i = 0; i < columns(); ++i)
	{
		QString key = "section_" + QString::number(i);
		if (cfg.hasKey(key))
		{
			QStringList args = cfg.readListEntry(key, ',');
			int section = args[0].toInt();
			header()->moveSection(section, i);
		}
	}

	updateResizabilityOfSections();
}

void PlayListView::loadColumnWidth(const QString &configName, const QString &groupName)
{
	Config cfg(configName, Config::File);
	cfg.setGroup(groupName);

	for (int i = 0; i < columns(); ++i)
	{
		QString key = "width_" + QString::number(i);
		if (cfg.hasKey(key))
			setColumnWidth(header()->mapToSection(i), QMAX(10, cfg.readNumEntry(key)));
	}

	setColumnWidth(0, 20);

	if (autoScaleColumnsEnabled_)
		setAutoSectionSizes();
}

void PlayListView::setBackgroundColors(const QColor &evenColor, const QColor &oddColor)
{
	oddRowColorGroup_.setColor(QColorGroup::Base, oddColor);
	evenRowColorGroup_.setColor(QColorGroup::Base, evenColor);
}

void PlayListView::setForegroundColor(const QColor &textColor)
{
	oddRowColorGroup_.setColor(QColorGroup::Text, textColor);
	oddRowColorGroup_.setColor(QColorGroup::Foreground, textColor);
	evenRowColorGroup_.setColor(QColorGroup::Text, textColor);
	evenRowColorGroup_.setColor(QColorGroup::Foreground, textColor);
}

void PlayListView::setPlayIndicatorImage(const QPixmap &image)
{
	playIndicatorImage_ = &image;
}

void PlayListView::setPauseIndicatorImage(const QPixmap &image)
{
	pauseIndicatorImage_ = &image;
}

void PlayListView::setErrorIndicatorImage(const QPixmap &image)
{
	errorIndicatorImage_ = &image;
}

void PlayListView::setSkinMode(const SkinModeInformation &modeInfo)
{
	if (modeInfo.playListFont().font.isUsed())
		setFont(modeInfo.playListFont().font);
	else
		unsetFont();

	// set general palette:
	QPalette palette = QApplication::palette();

	if (modeInfo.playListFont().fontColor.isUsed())
	{
		palette.setColor(QColorGroup::Text, modeInfo.playListFont().fontColor);
		palette.setColor(QColorGroup::Foreground, modeInfo.playListFont().fontColor);
	}

	if (modeInfo.playListBackgroundColorSelected().isUsed())
	{
		palette.setColor(QColorGroup::Highlight, modeInfo.playListBackgroundColorSelected());
		palette.setColor(QColorGroup::HighlightedText, modeInfo.playListFontColorSelected());
	}

	// set even and odd colors for items:
	if (modeInfo.playListBackgroundColorEven().isUsed())
	{
		if (modeInfo.playListBackgroundColorOdd().isUsed())
			setBackgroundColors(modeInfo.playListBackgroundColorEven(), modeInfo.playListBackgroundColorOdd());
		else
			setBackgroundColors(modeInfo.playListBackgroundColorEven(), modeInfo.playListBackgroundColorEven());

		palette.setColor(QColorGroup::Base, modeInfo.playListBackgroundColorEven());
	}
	else
		setBackgroundColors(palette.active().base(), palette.active().base());

	setPalette(palette);

	setPlayIndicatorImage(modeInfo.playListPlayIndicatorImage());
	setPauseIndicatorImage(modeInfo.playListPauseIndicatorImage());
	setErrorIndicatorImage(modeInfo.playListErrorIndicatorImage());

	if (modeInfo.playListHeaderFont().font.isUsed())
		header()->setFont(modeInfo.playListHeaderFont().font);
	else
		header()->unsetFont();

	if (modeInfo.playListHeaderColor().isUsed())
	{
		QPalette palette(modeInfo.playListHeaderColor());

		if (modeInfo.playListHeaderFont().fontColor.isUsed())
		{
			palette.setColor(QColorGroup::Text, modeInfo.playListHeaderFont().fontColor);
			palette.setColor(QColorGroup::ButtonText, modeInfo.playListHeaderFont().fontColor);
			palette.setColor(QColorGroup::Foreground, modeInfo.playListHeaderFont().fontColor);
		}

		header()->setPalette(palette);
	}
	else
		header()->unsetPalette();

	if (modeInfo.playListScrollBarColor().isUsed())
	{
		horizontalScrollBar()->setPalette(QPalette(modeInfo.playListScrollBarColor()));
		verticalScrollBar()->setPalette(QPalette(modeInfo.playListScrollBarColor()));
	}
	else
	{
		horizontalScrollBar()->unsetPalette();
		verticalScrollBar()->unsetPalette();
	}
}

void PlayListView::itemSelectedStateChanged(PlayListViewItem *item, bool selected)
{
	DPRINTF("==============================");

	if (selected)
	{
		DPRINTF("item selected %p", item);
		selectedItems_.append(item);
	}
	else
	{
		DPRINTF("item deselected %p", item);
		selectedItems_.remove(item);
	}

	DPRINTF("selectedItems_.count() = %d", selectedItems_.count());
	DPRINTF("-----------------------------");
}
