/*
 * Copyright (C) 2008 Andre Beckedorf <evilJazz _AT_ katastrophos _DOT_ net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <qevent.h>

#ifdef QT4
#include <QResizeEvent>
#include <QPixmap>
#include <QMouseEvent>
#endif

#include "paintbox.h"
#include "playinfo.h"
#include "helpers.h"
#include "imagefx.h"
#include "configuration.h"
#include "action.h"

#include "compathack.h"

class HotAreaLayer : public Layer
{
public:
	HotAreaLayer(LayeredPaintBox *parent) : Layer(parent) {}

protected:
	virtual bool hitTest(const QPoint &) { return true; }

	virtual void mousePressEvent(QMouseEvent *e)
	{
		int row = e->y() / (parent()->height() / 3);
		int col = e->x() / (parent()->width() / 3);

		int hotAreaNo = row * 3 + col;

		qDebug("row: %d, col: %d, hotArea: %d", row, col, hotAreaNo);

		if (qConfig.hotKeyMap[Configuration::HOTKEY_CONTEXT_HOTAREAS].contains(hotAreaNo))
		{
			int accelID = qConfig.hotKeyMap[Configuration::HOTKEY_CONTEXT_HOTAREAS][hotAreaNo];
			qDebug("accelID: %d", accelID);
			const Action &a = gActionManager.actionByAccelID(accelID);

			QString infoText = a.translatedName();

			int index = infoText.indexOf('/');
			if (index > -1)
				infoText = infoText.mid(index + 1);

			static_cast<PlayInfo *>(parent())->setInfoText(infoText);

			a.execute();
		}
	}
};

class CoverArtLayer : public PositionedLayer
{
public:
	CoverArtLayer(CoverArtImageProvider *imageProvider, LayeredPaintBox *parent)
		:	PositionedLayer(parent),
			imageProvider_(imageProvider)
	{
	}

	void setMediaInfo(const Media &info)
	{
		if (coverArtItem_.album != info.album() || coverArtItem_.artist != info.artist())
		{
			coverArtItem_ = CoverArtImageProvider::createGroupItemFromMedia(info);
			invalidate();
		}
	}

protected:
	virtual void paintToBuffer(QImage &buffer, const QRect &clipRect)
	{
		QImage *coverArtImage = imageProvider_->getImage(coverArtItem_.key, &coverArtItem_);

		//qDebug("coverArtImage %p", coverArtImage);

		if (coverArtImage)
		{
			//qDebug("coverArtImage %d %d", coverArtImage->width(), coverArtImage->height());

			float Cx, Cy;
			float RScaleX, RScaleY;
			float MaxWidth, MaxHeight;

			MaxWidth = width();
			MaxHeight = height() + (height() / 5);

			Cx = coverArtImage->width();
			Cy = coverArtImage->height();

			RScaleX = MaxWidth / Cx;
			RScaleY = MaxHeight / Cy;

			if (RScaleX >= RScaleY)
			{
				Cx = Cx * RScaleY;
				Cy = MaxHeight;
			}
			else
			{
				Cx = MaxWidth;
				Cy = Cy * RScaleX;
			}

			if (Cx > 0 && Cy > 0)
			{
				FloatPoint p1, p2, p3, p4;

				ImageFX::calculatePerspectivePoints(
					left() + width() / 2, top() + (Cx / 5), Cx, Cy,
					-30, 0.2, 0.5, true,
					p1, p2, p3, p4
				);

				ImageFX::projectiveTransformation(*coverArtImage, buffer, clipRect, p1, p2, p3, p4, true);
			}
		}
	}

	virtual void paintOverlay(QPainter &p, const QPixmap &pixmap)
	{

	}

private:
	CoverArtImageProvider *imageProvider_;
	GroupItem coverArtItem_;
};


PlayInfo::PlayInfo(CoverArtImageProvider *imageProvider, const SkinManager *skinmgr, QWidget *parent)
	:	LayeredPaintBox(parent),
		imageProvider_(imageProvider),
		trackLength_(0)
{
	title_ = new TextLayer(this);

	QFont font = title_->font();
	font.setBold(true);
	font.setPointSize(38);

	title_->setFont(font);
	title_->setAlignment(Qt::AlignRight);
	title_->setFontMinAutoScale(true);

	artist_ = new TextLayer(this);
	font.setPointSize(24);
	artist_->setFont(font);
	artist_->setColor(QColor(192, 192, 192));
	artist_->setAlignment(Qt::AlignRight);
	artist_->setFontMinAutoScale(true);

	album_ = new TextLayer(this);
	album_->setFont(font);
	album_->setColor(QColor(192, 192, 192));
	album_->setAlignment(Qt::AlignRight);
	album_->setFontMinAutoScale(true);

	trackNumbers_ = new TextLayer(this);
	font.setPointSize(12);
	trackNumbers_->setFont(font);
	trackNumbers_->setColor(QColor(128, 128, 128));
	trackNumbers_->setAlignment(Qt::AlignRight);

	timePlayed_ = new TextLayer(this);
	font.setPointSize(12);
	timePlayed_->setFont(font);

	timeRemaining_ = new TextLayer(this);
	font.setPointSize(12);
	timeRemaining_->setFont(font);

	info_ = new TextLayer(this);
	font.setPointSize(15);
	info_->setFont(font);

	seekBar_ = new SkinScrollBar(Skin::SeekSliderBar, Skin::SeekSliderKnob, skinmgr, this);

	imageProvider_->setMemoryCacheSize(2);
	imageProvider_->setScale(1);
	imageProvider_->setReflectionHeight(ReflectionHeight);
	imageProvider_->setRefSize(QSize(SlideWidthHeight, SlideWidthHeight + ReflectionHeight));

	connect(imageProvider_, SIGNAL(imageCached(CachedImage *)), this, SLOT(imageWasCached(CachedImage *)));

	coverArt_ = new CoverArtLayer(imageProvider_, this);

	hotAreaLayer_ = new HotAreaLayer(this);

	connect(&infoTimer_, SIGNAL(timeout()), this, SLOT(infoTextTimerTimeout()));
#ifdef QT4
	infoTimer_.setSingleShot(true);
#endif

	clear();
}

void PlayInfo::imageWasCached(CachedImage *)
{
	coverArt_->invalidate();
}

void PlayInfo::precacheCoverArt(const Media &info)
{
	GroupItem data = CoverArtImageProvider::createGroupItemFromMedia(info);
	imageProvider_->queue(data.key, &data);
}

void PlayInfo::resizeEvent(QResizeEvent *e)
{
	LayeredPaintBox::resizeEvent(e);
	realignLayers();
}

void PlayInfo::realignLayers()
{
	int totalHeight =
		title_->renderedTextSize().height() +
		5 +
		artist_->renderedTextSize().height() +
		5 +
		album_->renderedTextSize().height() +
		10 +
		trackNumbers_->renderedTextSize().height() +
		20 +
		timeRemaining_->renderedTextSize().height();

	int totalWidth = width() / 2;

	int topPos = (height() - totalHeight) / 2;

	title_->setGeometry(QRect(0, topPos, totalWidth, title_->renderedTextSize().height()));
	artist_->setGeometry(QRect(0, title_->geometry().bottom() + 5, totalWidth, artist_->renderedTextSize().height()));
	album_->setGeometry(QRect(0, artist_->geometry().bottom() + 5, totalWidth, album_->renderedTextSize().height()));
	trackNumbers_->setGeometry(QRect(0, album_->geometry().bottom() + 10, totalWidth, trackNumbers_->renderedTextSize().height()));

	int topSeekBar = trackNumbers_->geometry().bottom() + 20;

	timePlayed_->setGeometry(QRect(10, topSeekBar, timePlayed_->renderedTextSize().width(), timePlayed_->renderedTextSize().height()));
	timeRemaining_->setGeometry(QRect(totalWidth - timeRemaining_->renderedTextSize().width(), topSeekBar, timeRemaining_->renderedTextSize().width(), timeRemaining_->renderedTextSize().height()));

	seekBar_->setGeometry(QRect(timePlayed_->geometry().right() + 5, topSeekBar - 2, timeRemaining_->left() - timePlayed_->geometry().right() - 10, 25));

	info_->setGeometry(QRect(0, 10, width(), 30));

	coverArt_->setGeometry(QRect(totalWidth, 0, totalWidth, height()));
}

void PlayInfo::setMediaInfo(const Media &info)
{
	title_->setText(info.title());
	album_->setText(info.album());
	artist_->setText(info.artist());

	trackLength_ = info.length();
	seekBar_->setItemCount(trackLength_);
	coverArt_->setMediaInfo(info);

	setCurrentPlayTime(0);

	setPlaybackLayersVisible(true);

	if (!infoTimer_.isActive())
		setInfoText("", false);

	realignLayers();
}

void PlayInfo::setTrackNumbers(unsigned long trackNo, unsigned long trackCount)
{
	trackNumbers_->setText(QString().sprintf("%d of %d", trackNo, trackCount));
}

void PlayInfo::setCurrentPlayTime(unsigned long secs)
{
	timePlayed_->setText(formatPlayTime(secs, false));
	timeRemaining_->setText(formatPlayTime(-trackLength_ + secs, false));

	seekBar_->setPosition(secs);
}

void PlayInfo::clear()
{
	trackLength_ = 0;
	setCurrentPlayTime(0);
	setPlaybackLayersVisible(false);
	setInfoText("Stopped", false);
}

void PlayInfo::setPlaybackLayersVisible(bool value)
{
	title_->setVisible(value);
	artist_->setVisible(value);
	album_->setVisible(value);

	coverArt_->setVisible(value);

	timePlayed_->setVisible(value);
	timeRemaining_->setVisible(value);
	trackNumbers_->setVisible(value);
	seekBar_->setVisible(value);
}

void PlayInfo::setInfoText(const QString &text, bool vanish)
{
	info_->setVisible(true);
	info_->setText(text);

	if (vanish)
#ifdef QT4
		infoTimer_.start(2500);
#else
		infoTimer_.start(2500, true);
#endif
}

void PlayInfo::infoTextTimerTimeout()
{
	info_->setVisible(false);
}
