#include "directory.h"
#include "setbg.h"
#include "setscale.h"
#include "setcenter.h"
#include "setnibble.h"

#include <qdir.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qlistview.h>
#include <qapplication.h>
#include <qpe/qpeapplication.h>
#include <qtextstream.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

/* 
 *  Constructs a Example which is a child of 'parent', with the 
 *  name 'name' and widget flags set to 'f' 
 */

Directory::Directory( const QString & dir_name, QWidget* parent,  const char* name, WFlags fl)
    : QWidget( parent, name, fl )
{

  //
  // standard setting
  //
  br = 0;
  option.bgcolor=0xffffff;    // Initial background color -> white
  option.autoscale = AS_Both;         // Initial scale -> auto scale
  option.cent_X = 320;
  option.cent_Y = 240;
  option.nibble_mode = NIB_None;
  option.nibble_H = 1;
  option.nibble_V = 1;
  option.nibLeft = false;

  dirName = new QLineEdit( this, "DirName" );
  dirName->setGeometry( QRect( 0, 30, width(), 27 ) ); 
  dirName->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)7,
				       (QSizePolicy::SizeType)7,
				       dirName->sizePolicy().hasHeightForWidth() ) );
  dirName->setText(dir_name);
  connect(dirName, SIGNAL(textChanged(const QString &)),
	  this, SLOT(showDirectory()) );
  

  showDir = new QListView( this, "ShowDir" );
  showDir->setGeometry( QRect( 0, 60, width(), height()-60 ) ); 
  showDir->setSizePolicy( QSizePolicy( (QSizePolicy::SizeType)3, (QSizePolicy::SizeType)3,
				       showDir->sizePolicy().hasHeightForWidth() ) );
  showDir->addColumn("FileName");
  showDir->setColumnWidthMode(0, QListView::Manual);
  showDir->setColumnWidth(0, 200);
  showDir->addColumn("Ext");
  showDir->addColumn("Size");
  showDir->setColumnAlignment(2, Qt::AlignRight);
  showDir->addColumn("Perm");
  showDir->addColumn("Date");
  
  showDir->setFocus();
  
  connect(showDir, SIGNAL(doubleClicked(QListViewItem*)),
	  this, SLOT(fileView(QListViewItem*)) );
  connect(showDir, SIGNAL(returnPressed(QListViewItem*)),
	  this, SLOT(fileView(QListViewItem*)) );
  

  menuBar = new QMenuBar( this, "MenuBar");
  QPopupMenu* fileMenu = new QPopupMenu( this );
  fileMenu->insertItem( "Open",  this, SLOT(open()) );
  fileMenu->insertItem( "Quit", this, SLOT(close()) );
  QPopupMenu* settingMenu = new QPopupMenu( this );
  settingMenu->insertItem( "Background",  this, SLOT(setBGColor()) );
  settingMenu->insertItem( "Scale", this, SLOT(setScale()) );
  settingMenu->insertItem( "Center", this, SLOT(setCenter()) );
  settingMenu->insertItem( "Nibble", this, SLOT(setNibble()) );
  settingMenu->insertItem( "SaveSetting", this, SLOT(saveSetting()) );
  optionMenu = new QPopupMenu( this );
  bookmarkOn = optionMenu->insertItem( "Bookmark", this,
				       SLOT(bookmarkCheck() ));
  auto_Bookmark = optionMenu->insertItem( "AutoBookmmark", this,
					  SLOT(autoBookmarkCheck()) );

  menuBar->insertItem( "&File", fileMenu );  
  menuBar->insertItem( "&Setting", settingMenu );  
  menuBar->insertItem( "&Option", optionMenu );

  //
  // loading preconfig
  //
  configSetting();

  showDirectory();

}

/*  
 *  Destroys the object and frees any allocated resources
 */

Directory::~Directory()
{
  // no need to delete child widgets, Qt does it all for us
}

void Directory::open()
{
  fileView(showDir->selectedItem());
}

void Directory::setBGColor()            //Change Background color  
{
  SetBG * setbg = new SetBG(option);
  setbg->show();
  if( setbg->exec() ){
    option.bgcolor = setbg->rgbColor();
  }
  delete setbg;
}

void Directory::setScale()              //Change Standard Scale
{
  SetScale *setscale = new SetScale(option);
  setscale->show();
  if ( setscale->exec() ){
    option.ratio = setscale->getScale();
    option.autoscale = setscale->getASFlag();
  }
  delete setscale;
}

void Directory::setCenter()              //Change Standard Center Posision
{
  SetCenter *setcenter = new SetCenter(option);
  setcenter->show();
  if ( setcenter->exec() ){
    option.cent_X = setcenter->x();
    option.cent_Y = setcenter->y();
  }
  delete setcenter;
}

void Directory::setNibble()              //Change Nibble mode
{
  SetNibble *setnibble = new SetNibble(option);
  setnibble->show();
  if ( setnibble->exec() ){
    option.nibble_mode = setnibble->getNIBFlag();
    option.nibble_H = setnibble->getH();
    option.nibble_V = setnibble->getV();
    option.nibLeft = setnibble->getNIBLeft();
  }
  delete setnibble;
}


void Directory::showDirectory()         // Make Directory List & show 
{
  QString dir_name = dirName->text();   // Current Directory
   
   QDir cur_dir( dir_name );
   cur_dir.setFilter(QDir::All | QDir::Hidden );
   
   if(cur_dir.exists() != TRUE){
     return;
   }
   showDir->clear();
   cur_dir_num=cur_dir.count();
   
   for ( int i=0; i<cur_dir_num; i++ ){       //Reading File infomation
     QFileInfo f(cur_dir,cur_dir[i]);
     QListViewItem *file;
     
     QString fname;
     QString ext;
     QString size;
     QString perm;
     QString fdate;
     QString isDir;
     
     if (cur_dir[i].left(1) != "." && f.extension(FALSE) != "") {
       ext = "."+f.extension(FALSE);
       fname = cur_dir[i].replace(cur_dir[i].length()-ext.length(), ext.length(), "");
     }else {
       ext = "";
       fname = cur_dir[i];
     }
     perm  = (f.isDir() ? "d" : "-");         //permission check
     perm += (f.isReadable() ? "r" : "-");
     perm += (f.isWritable() ? "w" : "-");
     perm += (f.isExecutable() ? "x" : "-");
     perm += (f.isSymLink() ? "s" : "-");
     
     if (f.isDir()){                          //directory check
       size = "<DIR>";
       isDir = "D";
     }else {
       size = QString::number(f.size());
       isDir = "N";
     }
     
     file = new QListViewItem(showDir, fname, ext, size, perm, f.lastModified().toString(), isDir+cur_dir[i]);
     showDir->insertItem(file);
   }
   QString caption;
   showDir->setSorting(5);
   showDir->sort();
   caption = QString::QString("Files ") + " ("+QString::number(cur_dir_num)+")";
   setCaption( caption );
   
}

/*
  Directory Change
*/

void Directory::changeDir(const QString & chgDir)
{
  if (chgDir == "."){
    return;
  }
  if (chgDir == ".."){ 
    QDir d(dirName->text());
    if(d.cdUp() == TRUE){
      dirName->setText(d.absPath());
    }      
    return;
  }
  if (chgDir.left(1) == "/" ){
    QDir d(chgDir);
    if(d.exists() == TRUE){
      dirName->setText(d.absPath());
    }      
    return;
  }
    
  QDir d(dirName->text()+"/"+chgDir);
  if(d.exists() == TRUE){
    dirName->setText(d.absPath());
  }      
  
}


/*
  Directory Class Events
*/

void Directory::resizeEvent(QResizeEvent * )
{
  showDir->setGeometry( QRect( 0, 60, width(), height()-60));
  dirName->setGeometry( QRect( 0, 30, width(), 27 ));
  
}



void Directory::fileView(QListViewItem * item)
{
  if( item != 0 ){  
    if( (item->text(3).left(1) == "d") ){
      changeDir(item->text(0));
      return;
    }
    
    if( (item->text(1).upper() != ".JPG") )
      return;
  }else if ( !optionMenu->isItemChecked(bookmarkOn) ){
    return;
  }

    
  QString buf;
  option.rot = false;
  
  QFile f( "/tmp/qtembedded-zaurus/QtEmbedded-0.spec");  // Rotation check
  if ( f.open(IO_ReadOnly) ) {
    if( f.readLine(buf, 80) != -1)
      if( strstr( buf.latin1(), "270") != NULL){
	option.rot = true;
      }
    f.close();
  }
  
  fileList.clear();
  
  QListViewItemIterator it( showDir );             //Make jpg file list
  for ( ; it.current() ; it++){
    if(it.current()->text(1).upper() == ".JPG"){
      fileList.append(it.current()->text(0)+it.current()->text(1));
    }
  }
  option.filelist = &fileList;

  
  if(br != 0){
    delete br;
  }
  
  QString filename;
  if( optionMenu->isItemChecked(bookmarkOn) ){
    QFile f(dirName->text()+"/.bookmark");
    if (f.open(IO_ReadOnly)){
      char t[100];
      if ( f.readLine(t, 100) != -1 ){
	option.filename = QString(t).simplifyWhiteSpace();
      }
      QFileInfo f(dirName->text()+"/"+filename);
      if ( !f.exists() ){
	return;
      }
      
    }else{
      return;
    }
    
  }else{
    option.filename = item->text(0)+item->text(1);
  }

  option.dir_name = dirName->text();
  option.auto_bookmark = optionMenu->isItemChecked(auto_Bookmark);

  br=new BookReader(option);
  br->show();
  connect(br, SIGNAL(closed()), this, SLOT(fileViewClosed() ));

}

void Directory::fileViewClosed()
{
  delete br;
  br=0;
}

void Directory::configSetting()
{

  QFile f("/home/zaurus/Settings/bookreader.conf");

  if ( f.open(IO_ReadOnly) ){
    QString s;
    char t[100];
    while ( f.readLine(t, 100) != -1 ){
      s = QString(t);
      
      QStringList list = QStringList::split(',', s, FALSE);

      if (list[0] == "HomeDir"){
	dirName->setText(list[1].stripWhiteSpace());
      }

      if (list[0] == "Scale"){
	bool ok;
	int tmp_scale;
	tmp_scale = list[1].toInt( &ok, 10);
	if( ok ) 
	  option.ratio = tmp_scale;
      }

      if (list[0] == "AutoScale"){
	bool ok;
	int tmp_scale;
	tmp_scale = list[1].toInt( &ok, 10);
	if( ok ) 
	  option.autoscale = (ASFlags)tmp_scale;
      }

      if (list[0] == "BGColor"){
	bool ok;
	int tmp_color;
	tmp_color = list[1].toInt( &ok, 16);
	if( ok ) 
	  option.bgcolor = tmp_color;
      }

      if (list[0] == "CenterX"){
	bool ok;
	int tmp_x;
	tmp_x = list[1].toInt( &ok, 10);
	if( ok ) 
	 option.cent_X = tmp_x;
      }

      if (list[0] == "CenterY"){
	bool ok;
	int tmp_y;
	tmp_y = list[1].toInt( &ok, 10);
	if( ok ) 
	  option.cent_Y = tmp_y;
      }

      if (list[0] == "AutoBookmark"){
	bool ok;
	int tmp_ab;
	tmp_ab = list[1].toInt( &ok, 10);
	if( ok && tmp_ab) 
	  optionMenu->setItemChecked(auto_Bookmark,true);
	else
	  optionMenu->setItemChecked(auto_Bookmark,false);
      }

      if (list[0] == "Bookmark"){
	bool ok;
	int tmp_ab;
	tmp_ab = list[1].toInt( &ok, 10);
	if( ok && tmp_ab) 
	  optionMenu->setItemChecked(bookmarkOn,true);
	else
	  optionMenu->setItemChecked(bookmarkOn,false);
      }

      if (list[0] == "NibbleMode"){
	bool ok;
	int tmp_mode;
	tmp_mode = list[1].toInt( &ok, 10);
	if( ok ) 
	  option.nibble_mode = (NIBFlags)tmp_mode;
      }

      if (list[0] == "NibbleH"){
	bool ok;
	int tmp_H;
	tmp_H = list[1].toInt( &ok, 10);
	if( ok ) 
	 option.nibble_H = tmp_H;
      }

      if (list[0] == "NibbleV"){
	bool ok;
	int tmp_V;
	tmp_V = list[1].toInt( &ok, 10);
	if( ok ) 
	  option.nibble_V = tmp_V;
      }

      if (list[0] == "NibLeft"){
	bool ok;
	int tmp_Left;
	tmp_Left = list[1].toInt( &ok, 10);
	if( ok ) 
	  option.nibLeft = tmp_Left;
      }


    }
  f.close();
  }

}

void Directory::saveSetting()
{

  QFile f("/home/zaurus/Settings/bookreader.conf");

  if ( f.open(IO_WriteOnly) ){
    QString s;

    s = "HomeDir,"+ dirName->text();
    f.writeBlock(s.latin1(), s.length());
    f.writeBlock("\n", 1);

    s = "Scale,"+ QString::number(option.ratio, 10);
    f.writeBlock(s.latin1(), s.length());
    f.writeBlock("\n", 1);

    s = "AutoScale,"+ QString::number(option.autoscale, 10);
    f.writeBlock(s.latin1(), s.length());
    f.writeBlock("\n", 1);

    s = "BGColor,"+ QString::number(option.bgcolor, 16);
    f.writeBlock(s.latin1(), s.length());
    f.writeBlock("\n", 1);

    s = "CenterX,"+ QString::number(option.cent_X, 10);
    f.writeBlock(s.latin1(), s.length());
    f.writeBlock("\n", 1);

    s = "CenterY,"+ QString::number(option.cent_Y, 10);
    f.writeBlock(s.latin1(), s.length());
    f.writeBlock("\n", 1);

    s = "AutoBookmark,"+ 
      QString::number(optionMenu->isItemChecked(auto_Bookmark), 10);
    f.writeBlock(s.latin1(), s.length());
    f.writeBlock("\n", 1);

    s = "Bookmark,"+ 
      QString::number(optionMenu->isItemChecked(bookmarkOn), 10);
    f.writeBlock(s.latin1(), s.length());
    f.writeBlock("\n", 1);

    s = "NibbleMode,"+ QString::number(option.nibble_mode, 10);
    f.writeBlock(s.latin1(), s.length());
    f.writeBlock("\n", 1);

    s = "NibbleH,"+ QString::number(option.nibble_H, 10);
    f.writeBlock(s.latin1(), s.length());
    f.writeBlock("\n", 1);

    s = "NibbleV,"+ QString::number(option.nibble_V, 10);
    f.writeBlock(s.latin1(), s.length());
    f.writeBlock("\n", 1);

    s = "NibLeft,"+ QString::number(option.nibLeft, 10);
    f.writeBlock(s.latin1(), s.length());
    f.writeBlock("\n", 1);

  f.close();
  }
}

