/*
 * Copyright (C) 2002 SHARP CORPORATION All rights reserved.
 */
#ifndef _SLZDTM_H_INCLUDED
#define _SLZDTM_H_INCLUDED


#include <qobject.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qdatetime.h>

class QWidget;

namespace SlZDtm {

#define SLZDTM_CARDID_32BIT

#if defined(SLZDTM_CARDID_32BIT)
typedef ulong CardId;
#else
typedef ushort CardId;
#endif
typedef CardId IndexNo;
typedef int CategoryId;
typedef uchar DataType;
typedef ulong APLID;
typedef void **BoxHandle;
typedef enum FileStatus {
	Ready,
	Retry,
	Fatal
} FileStatus;

extern bool FilerCheckCategoryId(CategoryId);
extern const QCString SjisString(const QString &s);

struct ItemData;
class SlZDataManagerItemInfo : public QValueList<ItemData> {
public:
	SlZDataManagerItemInfo();
	~SlZDataManagerItemInfo();
	bool addTextItem(const char *itemName,const QString &title);
	bool addUcharItem(const char *itemName,const QString &title);
	bool addBitItem(const char *itemName,const QString &title);
	bool addUshortItem(const char *itemName,const QString &title,bool isGSearch=false);
	bool addUlongItem(const char *itemName,const QString &title);
	bool addColorItem(const char *itemName,const QString &title);
	bool addCategoryItem();
	bool addTimeItem(const char *itemName,const QString &title);
	bool addBinaryItem(const char *itemName,const QString &title);
};

struct SortKey;
class SlZDataManagerSortKey : public QValueList<SortKey> {
public:
	SlZDataManagerSortKey();
	~SlZDataManagerSortKey();
	void addSortKey(const char *itemName,bool isString=false,
					bool isAscendent=true);
	ulong createSortExp(char *buf,ulong bufLen) const;
};

struct IndexFileInfo;
class SlZDataManagerIndexInfo : public QValueList<IndexFileInfo> {
public:
	SlZDataManagerIndexInfo();
	~SlZDataManagerIndexInfo();
	void addIndexInfo(const char *indexFName,
					  const char *searchExp,ulong searchExpLen,
					  const char *sortExp,ulong sortExpLen);
};

class SlZDataManagerPrivate;
class SlZDataManager : public QObject {
	Q_OBJECT
public:
	static bool createFile(const char *boxFName,
						   const char *masterIdxFName,
						   APLID aplId,
						   bool isEasyMode,
						   const SlZDataManagerItemInfo &items,
						   const SlZDataManagerSortKey &keys,
						   const SlZDataManagerIndexInfo &indexes);
	static bool createFile(const char *boxFName,
						   const char *masterIdxFName,
						   APLID aplId,
						   bool isEasyMode,
						   const SlZDataManagerItemInfo &items,
						   const char *sortExp,int sortLen,
						   const SlZDataManagerIndexInfo &indexes);
	static bool createZTime(void *zTime,const QDateTime &date);
	static bool createZTime(void *zTime,
							int y,int mon,int d,int h=0,int m=0,int sec=0);
	static void getAttrValues(uchar *mask,uchar *cond,uchar *flag);
	static bool deleteFile(const char *boxFName);
	static bool checkGarbageAlert(const char *boxFName,QWidget *parent=NULL);
	
	static bool checkFile(const char *fName);
	static void checkDtmGlobal();

	SlZDataManager(const char *boxFName,QWidget *parent=NULL,bool isReadOnly=false);
	~SlZDataManager();
	

	enum DeleteMode {
		
		OneCard,
		
		CurrentIndex,
		
		AllCard
	};
	

	enum FilterFor {
		
		ForCurrentIndex,
		
		ForMasterIndex,
		
		ForSuspendedIndex
	};
	bool open(bool isReadOnly=false);
	bool close();
	bool isOpened();
	int count(const char *idxName=NULL);
	QString readItem(const char *itemName,CardId cardId=0);
	QDateTime readTimeItem(const char *itemName,CardId cardId=0,bool isUtc=false);
	QDate readDateItem(const char *itemName,CardId cardId=0);
	int readUcharItem(const char* itemName,CardId cardId=0);
	int readUshortItem(const char* itemName,CardId cardId=0);
	bool readUlongItem(const char* itemName,ulong *data,CardId cardId=0);
	QArray<int> readCategories(CardId cardId=0);
	bool first(const char *idxName=NULL);
	bool last(const char *idxName=NULL);
	bool next(CardId cardId=0,const char *idxName=NULL);
	bool prev(CardId cardId=0,const char *idxName=NULL);
	CardId firstCard(const char *idxName=NULL);
	CardId lastCard(const char *idxName=NULL);
	CardId nextCard(CardId cardId=0,const char *idxName=NULL);
	CardId prevCard(CardId cardId=0,const char *idxName=NULL);
	CardId cardId() const;
	CardId cardIdFromNumber(IndexNo number);
	IndexNo numberFromCardId(CardId cardId=0);

	bool writeItem(const char* itemName,const char *data,ulong len);
	bool writeItem(const char* itemName,const QDateTime &time);
	bool writeItem(const char* itemName,const QDate &time); 
	bool writeItem(const char* itemName,const QString &data);
	

	bool writeItem(const char* itemName,uchar value){
		return writeItem(itemName,(const char*)&value,1);
	}
	

	bool writeItem(const char* itemName,ushort value){
		return writeItem(itemName,(const char*)&value,2);
	}
	

	bool writeItem(const char* itemName,ulong value){
		return writeItem(itemName,(const char*)&value,4);
	}
	bool writeCurrentTime(const char* itemName,bool isUtc);
	

	bool clearItem(const char* itemName){
		return writeItem(itemName,(const char*)NULL,(ulong)0);
	}
	bool writeAttribute(CardId cardId,uchar bitData,bool isSet);
	bool startEditCard(CardId cardId);
	bool cancelEditCard();
	bool finishEditCard(CardId *cardId);
	bool editNewCard();

	bool deleteCard(CardId *cardId,DeleteMode mode=OneCard);

	bool sort(const char* itemName,bool isAscendent);
	bool sort(int keyItemNo,bool isAscendent);
	bool sort(const char *sortExp,int sortExpLen,bool isAscendent);
	void sort(bool isAscendent);
	bool search(CardId *cardId,const char *serCond,bool isNext,
				uchar attrMask,uchar attrCond,uchar attrFlag,
				const char *idxName=NULL);
	bool search(CardId *cardId,const char *serCond,bool isNext,
				const char *idxName=NULL);
	bool search(CardId *cardId,const QString &string,
				const char *item,uchar searchMode,bool isNext=true);
	bool filter(const char *searchExp,ulong searchExpLen,
				const char *sortExp,ulong sortExpLen,
				FilterFor mode,bool isSuspendIndex=false);
	bool unfilter(bool isResumeIndex=true);
	void killIdxResume();
	bool isFiltered() const;
	bool enableFilter(CategoryId categoryId);
	bool disableFilter(bool isIdxDelete);
	bool addCardToCtgr(CardId cardId,
					   CategoryId categoryId,DeleteMode mode=OneCard,
					   bool isAlert=false);
	bool removeCardFromCtgr(CardId cardId,
							CategoryId categoryId,DeleteMode mode=OneCard,
							bool isAlert=false);
	bool updateCategories(CardId cardId,QArray<int> ids,
						  DeleteMode mode=OneCard,bool isAlert=false);
	DataType readItemType(const char *itemName) const;
	int nameToItemId(const char *itemName) const;
	int sortItem() const;
	bool ascendent() const;
	bool setCurrentIndex(const char* indexFName,bool isMasterIdx=false);

	bool count(int *countNo,const char *serCond,CardId cardId,
				uchar attrMask,uchar attrCond,uchar attrFlag,
				const char *idxName=NULL);
	bool count(int *countNo,const char *serCond,CardId cardId,
				const char *idxName=NULL);

	ulong garbageSize();
	CardId expectedCardId();
	CardId resumeCardId() const;

	bool setInternalUtc(const char* itemName);

	bool categoryRemoved(CategoryId id);
	static void updateForCategoryRemoved(CategoryId id);

	QString current1stSortKey() const;
	BoxHandle boxHandle() const;
	const SlZDataManagerItemInfo items() const;
	void dumpIndexStatus() const;
private slots:
	void systemMessage(const QCString &msg,const QByteArray &data);
private:
	SlZDataManagerPrivate *p;
};

} 
#endif 
