/*
 * kernel swicher for sl zaurus
 *  Version 0.2
 *                         piro
 */


#undef __KERNEL__
#undef MODULE
#define __KERNEL__
#define MODULE

#include <linux/kernel.h>
#include <linux/modsetver.h>
#include <linux/module.h>

#include <linux/elf.h>
#include <linux/errno.h>
#include <linux/fs.h>
#include <linux/pagemap.h>
#include <linux/file.h>
#include <linux/slab.h>
#include <linux/elf.h>
#include <linux/types.h>
#include <linux/proc_fs.h>
#include <asm/mach/map.h>
#include <asm/memory.h>
#include <asm/uaccess.h>

#define DEBUG


#define ZBOOTDEV_MAJOR	99
#define ZBOOTDEV_MODE	0222
#define ZBOOTDEV_NAME	"kernelsw"
#define ZBOOTMOD_NAME	"kernelsw"

#define KERNEL_SIZE     1024 * 1024 * 2
#define KERNEL_OFFSET   0xa0008000

#define PARAM_SIZE      0x900
#define PARAM_OFFSET    0xa0000100

#define LOADER_SIZE     0x1000
#define LOADER_OFFSET   0xa0006000

#define LDRPARAM_SIZE   0x800
#define LDRPARAM_OFFSET 0xa0007000

#define TMP_STACK       0xa0007c00

#define INITRD_SIZE     1024 * 1024 * 4     //  max initrd size 4M 
#define INITRD_OFFSET   0xa1000000

#define INDEX_OFFSET    0xa0001000
#define INDEX_SIZE      0x4000

#define NO_IMAGE        0
#define KERNEL_IMAGE    1
#define PARAM_DATA      2
#define INITRD_IMAGE    4
#define BOOT            5

#define P2V(x)          (x+0x20000000)

static int  last_page;    // kernel image page number (allocate page number)
static unsigned int *index;
static int  isize;         // stored kernel image size


/* Prototypes */
int	init_module(void);
void	cleanup_module(void);

ssize_t	kernelsw_write(struct file *, const char *, size_t, loff_t *);
int	kernelsw_open(struct inode *, struct file *);
int	kernelsw_close(struct inode *, struct file *);

static	struct file_operations fops = {
	0,			/* struct module *owner */
	0,			/* lseek */
	0,			/* read */
	kernelsw_write,		/* write */
	0,			/* readdir */
	0,			/* poll */
	0,			/* ioctl */
	0,			/* mmap */
	kernelsw_open,		/* open */
	0,			/* flush */
	kernelsw_close,		/* release */
	0,			/* sync */
	0,			/* async */
	0,			/* check media change */
	0,			/* revalidate */
	0,			/* lock */
};

static	int isopen;
static	loff_t position;

static	char *addr;

static  int kernel_size   = KERNEL_SIZE; 
static  int param_size    = PARAM_SIZE; 
static  int initrd_size   = INITRD_SIZE; 

static  int use_param = 0;
MODULE_PARM(use_param, "i");

static  int use_initrd = 0;
MODULE_PARM(use_initrd, "i");

static int kernel26 = 1;
MODULE_PARM(kernel26, "i");

static int mach_type = 0;
MODULE_PARM(mach_type, "i");


static int load_mode = NO_IMAGE;

static void *func;

typedef struct LdrParam {
  int  mach_type;                   // +0
  uint addr;                  // +1
  int  use_initrd;                  // +2
  int  use_param;                   // +3
  int  stored_index[4];
  int  total_pages;
  int  index_page_start;            // +10
  int  param_page_start;
  int  kernel_page_start;
  int  initrd_page_start;
  int  loader_page_start;
  int  ldrparam_page_start;         // +20
  int  boot_page_start;
} LdrParam;

unsigned int total_pages, index_pages, param_pages, kernel_pages,
             initrd_pages, loader_pages, ldrparam_pages, boot_pages;

unsigned int index_page_start, param_page_start, kernel_page_start, 
             initrd_page_start, loader_page_start, ldrparam_page_start, boot_page_start;


/*

 */

void boot_loader(void)
{

  /*
    copy initrd image in KERNEL_OFFSET to INITRD_OFFSET
  */
  
  char *sp, *dp;
  int i;
  LdrParam *param;
  
  param = (LdrParam *)P2V(LDRPARAM_OFFSET);
  
  if(param->use_initrd){
    sp = (char*)P2V(KERNEL_OFFSET + ( ((KERNEL_SIZE - 1)/PAGE_SIZE) + 1) * PAGE_SIZE );
    dp = (char*)P2V(INITRD_OFFSET);
    for(i = 0; i < INITRD_SIZE; i++){
      *dp++ = *sp++;
    }
  }
  
  
  __asm__ volatile (
		    
		    /*
		      kernel boot setting
		    */
		    "mov  r1, %0;\n"          // mach type   
		    "mov  r0, %1;\n"          // addr
		    
		    
		    /*
		      cache & MMU setting
		    */
		    
		    "mov  r3, #0;\n"
		    "mov  r4, #0x78;\n"
		    "mcr  15, 0, r3, c1, c0, 0;\n"
		    "mcr  15, 0, r4, c8, c7, 0;\n"
		    
		    /*
		      boot
		    */
		    
		    "mov  pc, r0;\n"
		    : : "r"(param->mach_type), "r"(param->addr) : "r0","r1","r3","r4","r5" );
  
}

int boot_2nd(int param)
{

  uint *tmp;
  void (*loader)(void); 
  int i, j;
  char *sp,*dp;
  int cpsr;
  int *boot_index;
  LdrParam *ldrparam;

  ldrparam = (LdrParam*)param;


  __asm__ volatile ("mrs %0, cpsr_all" : "=r" (cpsr));
  cpsr |= 0xc0;  // set FI 
  __asm__ volatile ("msr cpsr_all, %0" :: "r" (cpsr));
  

  /*
    set stack pointer
  */

  sp = (char *)P2V(TMP_STACK);
  __asm__ volatile (
		    "mov  %0, sp;\n"
		    : : "r"(sp) );
  
  
  /*
    copy index to INDEX_OFFSET
  */
  
  
  dp = (char *)(P2V(INDEX_OFFSET));
  for(j = 0; j < 4; j++){
    sp = (char *)(ldrparam->stored_index[j]);
    for(i = 0; i < PAGE_SIZE; i++){
      *dp++ = *sp++;
    }
  }
  
  boot_index = (int *)(P2V(INDEX_OFFSET));
  
  /*
    copy loader param to LDRPARAM_OFFSET
  */
  
  sp = (char *)(ldrparam);
  dp = (char *)(P2V(LDRPARAM_OFFSET));
  for(i=0; i <PAGE_SIZE; i++){
    *dp++ = *sp++;
  }
  
  ldrparam = (LdrParam*)(P2V(LDRPARAM_OFFSET));
  
  /*
    copy boot loader to LOADER_OFFSET
  */
  
  sp = (char *)(boot_index[ldrparam->loader_page_start]);
  dp = (char *)(P2V(LOADER_OFFSET));
  for(i=0; i <PAGE_SIZE; i++){
    *dp++ = *sp++;
  }
  
  
  /*
    copy param to PARAM_OFFSET
  */
  if(ldrparam->use_param){
    sp = (char *)(boot_index[param_page_start]);;
    dp = (char *)(P2V(PARAM_OFFSET));
    for(i=0; i <PARAM_SIZE; i++){
      *dp++ = *sp++;
    }
  }
  
  
  /*
    copy kernel image, initrd image and loader to KERNEL_OFFSET
  */
  
  dp = (char*)P2V(KERNEL_OFFSET);
  for(i = ldrparam->kernel_page_start; i < ldrparam->loader_page_start; i++){
    sp = (char*)(boot_index[i]);
    for(j = 0; j < PAGE_SIZE; j++){
      *dp++ = *sp++;
    }
  }

  /*
    jump to loader
  */
  
  loader = (void *)P2V(LOADER_OFFSET);
  loader();
  
}


void boot_1st(void)
{
  void (*loader)(void) = boot_loader; 
  int (*boot2nd)(int) = boot_2nd; 
  LdrParam *ldrparam;
  int i, j;
  char *sp,*dp;
  int *tmp;
  int cpsr;

  ldrparam = (LdrParam*)index[ldrparam_page_start];

  ldrparam->addr = (uint)KERNEL_OFFSET;
  ldrparam->use_param = use_param;
  ldrparam->use_initrd = use_initrd;


  
  if(mach_type == 0){
    if (kernel26)
      ldrparam->mach_type = 423;  // MACH_TYPE_CORGI on kernel 2.6
    else
      ldrparam->mach_type = 309;  // MACH_TYPE_CORGI on kernel 2.4
  }else {
    ldrparam->mach_type = mach_type;
    
  }
  
  
  /*
    param_struct setting
    from include/asm-arm/setup.h
  */
  
  if(use_param){
    
    tmp = (int*)index[param_page_start];
    
    *tmp = 0x00001000; //  page_size 
    tmp += 1;   *tmp = 0x00002000; //  nr_pages
    tmp += 2; 
    if (use_initrd){
      *tmp = 0x00000008; //  flags
    }else {
      *tmp = 0x0000000d; //  flags
    }
    tmp += 1; *tmp = 0x000000ff; //  rootdev
    
    if (use_initrd){
      tmp += 12;*tmp = 0xc1000000; //  initrd_start 
      tmp += 1; *tmp = 0x00800000; //  initrd_size
    }
    
  }
  
  /*
    copy index address
  */
  
  sp = (char *)(index);
  for(j = 0; j < 4; j++){
    dp = (char *)(index[index_page_start + j]);
    for(i = 0; i < PAGE_SIZE; i++){
      *dp++ = *sp++;
    }
  }
  
  /*
    copy boot_2nd
  */
  
  sp = (char *)boot_2nd;
  dp = (char *)(index[boot_page_start]);
  for(i=0; i < PAGE_SIZE; i++){
    *dp++ = *sp++;
  }
  
  /*
    copy loader
  */
  
  sp = (char *)loader;
  dp = (char *)(index[loader_page_start]);
  for(i=0; i < PAGE_SIZE; i++){
    *dp++ = *sp++;
  }
  

  /* 
     setting param for boot loader 
  */
  
  ldrparam->stored_index[0] = index[index_page_start + 0];
  ldrparam->stored_index[1] = index[index_page_start + 1];
  ldrparam->stored_index[2] = index[index_page_start + 2];
  ldrparam->stored_index[3] = index[index_page_start + 3];
  ldrparam->total_pages    = total_pages;
  ldrparam->index_page_start    = index_page_start;            
  ldrparam->param_page_start    = param_page_start;
  ldrparam->kernel_page_start   = kernel_page_start;
  ldrparam->initrd_page_start   = initrd_page_start;
  ldrparam->loader_page_start   = loader_page_start;
  ldrparam->ldrparam_page_start = ldrparam_page_start;         
  ldrparam->boot_page_start     = boot_page_start;
  
  boot2nd = (void *)index[boot_page_start];
  
  
  /*
    set stack pointer
  */

  sp = (char *)index[ldrparam_page_start] + 0x0c00;
  __asm__ volatile (
		    "mov  %0, sp;\n"
		    : : "r"(sp) );

  

  i=boot2nd((int)ldrparam);                // not use pointer
  
}



/*
  memory allocation
*/

static void sort(unsigned long item[], int num)
{
  int i,j,tmp;
  for(i=0;i<num-1;i++){
    for(j=num-1; j>i; j--){
      if(item[j] < item[j-1]){
	tmp       = item[j];
	item[j]   = item[j-1];
	item[j-1] = tmp;
      }
    }
  }
}

static void free_mem(int num)
{
  int i;

  for(i = 0; i < num; i++){
    kfree((unsigned long *)index[i]);
  }
  if(index != NULL){
    kfree((unsigned long *)index);
  }

  index = NULL;
  last_page = 0;
  isize      = 0;
}

int memory_alloc(int num)
{
  int i,j;
  void *tmp;
  unsigned long *k;

  free_mem(last_page);

  if((index = (void *)kmalloc(PAGE_SIZE*4, GFP_KERNEL)) == NULL){
    printk("kernelsw : memory allocation error !\n");
    return -1;
  }

  for(i = 0; i<num; i++){
    if((tmp = (void *)kmalloc(PAGE_SIZE, GFP_KERNEL)) == NULL){
      printk("kernelsw : memory allocation error ! (index[%d])\n",i);
      free_mem(i);
      return 0;
    }
    index[i] = (unsigned long) tmp;

    k = (unsigned long*)index[i];
    for(j = 0; j < (PAGE_SIZE/4); j++){
      *k++ = 0;
    }

  }

  
  sort((unsigned long *)index, num);

  if(index[0] < P2V(KERNEL_OFFSET)){
    return -1;
  }

  last_page = num;

  return num;
}

/*
 * Initialize the LKM.
 */

int init_module()
{
  struct proc_dir_entry *entry;
  int rc;
  
  isize = 0; last_page = 0; index = NULL;
  
  rc = register_chrdev(ZBOOTDEV_MAJOR, ZBOOTDEV_NAME, &fops);
  if (rc != 0) {
    printk("%s: register_chrdev(%d, ...): error %d\n",
	   ZBOOTMOD_NAME, -rc);
    return 1;
  }
  
  entry = proc_mknod(ZBOOTDEV_NAME, ZBOOTDEV_MODE | S_IFCHR,
		     &proc_root, MKDEV(ZBOOTDEV_MAJOR, 0));
  if (entry == (struct proc_dir_entry *)0) {
    (void)unregister_chrdev(ZBOOTDEV_MAJOR, ZBOOTDEV_NAME);
    return 1;
  }
  
  printk("%s: kernelsw/" MACHINE " bootstrap device is %d,0\n",
	 ZBOOTMOD_NAME, ZBOOTDEV_MAJOR);
  
  /*                   */
  /* allocate memory   */
	/*                   */
  index_pages    = 4; //
  param_pages    = use_param  == 1 ? ( (param_size - 1)  / PAGE_SIZE) + 1 : 0;
  kernel_pages   = ( (kernel_size - 1) / PAGE_SIZE) + 1;
  initrd_pages   = use_initrd == 1 ? ( (initrd_size - 1) / PAGE_SIZE) + 1 : 0;
  loader_pages   = 1;
  ldrparam_pages = 1;
  boot_pages     = 1;
  
  index_page_start    = 0;
  param_page_start    = index_page_start + index_pages;
  kernel_page_start   = param_page_start + param_pages;
  initrd_page_start   = kernel_page_start + kernel_pages;
  loader_page_start   = initrd_page_start + initrd_pages;
  ldrparam_page_start = loader_page_start + loader_pages;
  boot_page_start     = ldrparam_page_start + ldrparam_pages;
  
  total_pages = (index_pages + param_pages + kernel_pages + initrd_pages + 
		 loader_pages + ldrparam_pages + boot_pages);
  
  if(memory_alloc(total_pages) <= 0){
    printk("kernelsw : memory allocation error !\n");
    
    if(last_page){
      free_mem(last_page);
    }
    
    return 1;
  }
  
  load_mode = KERNEL_IMAGE;
  
  printk("%s: %s loading mode.\n",
	 ZBOOTMOD_NAME, kernel26 ? "kernel 2.6" : "kernel 2.4");
  
  return 0;
  
}

/*
 * Cleanup - undo whatever init_module did.
 */

void cleanup_module()
{

  if(last_page){
    free_mem(last_page);
  }
  
  (void)unregister_chrdev(ZBOOTDEV_MAJOR, ZBOOTDEV_NAME);
  remove_proc_entry(ZBOOTDEV_NAME, &proc_root);
  
  printk("%s: kernelsw/" MACHINE " bootstrap device unloaded\n",
	 ZBOOTMOD_NAME);
}


ssize_t kernelsw_write(struct file *f, const char *buf, size_t len, loff_t *offp)
{
  int size;
  int *dist;
  int start_pages;
  int page,offs,rest,bytes;
  int count = len;
  
  if (load_mode == KERNEL_IMAGE){
    start_pages = kernel_page_start;
  }else if(load_mode == PARAM_DATA){
    start_pages = param_page_start; isize = 0x500;
  }else if(load_mode == INITRD_IMAGE){
    start_pages = initrd_page_start;
  }else if(load_mode == BOOT){
    return len;
  }	  
  
  
	
  if(len > PAGE_SIZE){
	  len = PAGE_SIZE;
  }
  
  if((isize + len) > PAGE_SIZE*(last_page)){
    printk("kernelsw : It is too large image !\n");
    isize = 0;
    return -ENODEV;
  }
  
  page = ((isize) / PAGE_SIZE ) + start_pages;
  offs = (isize) % PAGE_SIZE;
  
  if( (len+offs) > PAGE_SIZE ){
    rest = ((len+offs) - PAGE_SIZE );
    bytes = PAGE_SIZE - offs;
  }else{
    bytes = len;
    rest = 0;
  }
  
  memcpy( ((char*)index[page]) + offs, buf, bytes);  // can't use copy_from_user ?
  
  count = len-rest;
  isize += count;
  
  if (isize > position)
    position = isize;
  
  return count;
  
}
	  

int kernelsw_open(struct inode *ino, struct file *f)
{

  /* XXX superuser check */
  
  if (isopen)
    return -EBUSY;
  
  isopen = 1;
  position = 0;
  isize = 0;
  return 0;
}

int kernelsw_close(struct inode *ino, struct file *f)
{
  if (isopen) {
    if(load_mode == BOOT){
      
      boot_1st();
      printk("%s: boot failed\n", ZBOOTDEV_NAME);       
      
    }else if (position > 0) {
      
      if(load_mode == KERNEL_IMAGE){
	
	printk("%s: Kernel image loaded %d bytes\n", ZBOOTDEV_NAME, position);
	
	if(use_param){
	  load_mode = PARAM_DATA;
	}else if(use_initrd){
	  load_mode = INITRD_IMAGE;
	}else {
		load_mode = BOOT;
	}
	
      }else if(load_mode == PARAM_DATA){

	printk("%s: Param data loaded %d bytes\n", ZBOOTDEV_NAME, position);
	
	if(use_initrd){
	  load_mode = INITRD_IMAGE;
	}else {
	  load_mode = BOOT;
	}
	
      }else if(load_mode == INITRD_IMAGE){
	printk("%s: Initrd image loaded %d bytes\n", ZBOOTDEV_NAME, position);
	load_mode = BOOT;
      }
      
    }
    
    isopen = 0;
    return 0;
  }
  
  return -EBUSY;
}


